import apiClient from "@/lib/apiClient";

export type Attachment = {
  id: string;
  original_name: string;
  mime_type: string | null;
  size_bytes: string | number | null;
  uploaded_at: string;
  uploaded_by: string | null;
  downloadUrl?: string;
  previewUrl?: string;
};

function parseFilenameFromContentDisposition(v: string | null | undefined) {
  if (!v) return null;

  // 1) RFC5987: filename*=UTF-8''<urlencoded>
  // pl: filename*=UTF-8''M%C3%A1rta%20d%C3%B6nt%C3%A9s.pdf
  const mStar = /filename\*\s*=\s*UTF-8''([^;]+)/i.exec(v);
  if (mStar?.[1]) {
    try {
      return decodeURIComponent(mStar[1].trim());
    } catch {
      // ha valamiért nem URL-encoded rendesen, akkor visszaadjuk nyersen
      return mStar[1].trim();
    }
  }

  // 2) sima filename="..."
  const m = /filename\s*=\s*"([^"]+)"/i.exec(v);
  if (m?.[1]) {
    try {
      return decodeURIComponent(m[1]);
    } catch {
      return m[1];
    }
  }

  return null;
}

export const attachmentsApi = {
  uploadToThread: (threadRef: string, file: File) => {
    const form = new FormData();
    form.append("file", file);
    return apiClient.post<{ id: string }>(
      `/api/admin/attachments/thread/${threadRef}`,
      form,
      { headers: { "Content-Type": "multipart/form-data" } }
    );
  },

  uploadToMeeting: (meetingId: string, file: File) => {
    const form = new FormData();
    form.append("file", file);
    return apiClient.post<{ id: string }>(
      `/api/admin/attachments/meeting/${meetingId}`,
      form,
      { headers: { "Content-Type": "multipart/form-data" } }
    );
  },
  downloadUrl: (attachmentId: string) =>
    `/api/admin/attachments/${attachmentId}/download`,

  previewUrl: (attachmentId: string) =>
    `/api/admin/attachments/${attachmentId}/preview`,

  /** 🔽 letöltés (force download, NO NAVIGATION) */
  downloadByFetch: async (attachmentId: string) => {
    const r = await apiClient.get(
      `/api/admin/attachments/${attachmentId}/download`,
      { responseType: "blob" }
    );

    const blob = r.data as Blob;
    const cd = r.headers["content-disposition"] as string | undefined;

    const filename = parseFilenameFromContentDisposition(cd) ?? "file";

    const url = window.URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = filename; // ✅ itt már az ékezetes név jön
    document.body.appendChild(a);
    a.click();
    a.remove();
    window.URL.revokeObjectURL(url);
  },

  /** 👁 preview (inline) */
  previewByFetch: async (attachmentId: string) => {
    const r = await apiClient.get(
      `/api/admin/attachments/${attachmentId}/preview`,
      { responseType: "blob" }
    );
    const blob = r.data as Blob;
    const url = URL.createObjectURL(blob);
    return { url, mime: r.headers["content-type"] as string | undefined };
  },

  deleteById: (attachmentId: string) =>
    apiClient.delete<{ unlinked: boolean; deleted: boolean }>(
      `/api/admin/attachments/${attachmentId}`
    ),

  removeFromThread: (threadRef: string, attachmentId: string) =>
    apiClient.delete(
      `/api/admin/attachments/thread/${threadRef}/${attachmentId}`
    ),

  removeFromActionItem: (actionItemRef: string, attachmentId: string) =>
    apiClient.delete(
      `/api/admin/attachments/action-item/${actionItemRef}/${attachmentId}`
    ),
  uploadToDecision: (decisionRef: string, file: File) => {
    const form = new FormData();
    form.append("file", file);
    return apiClient.post<{ id: string }>(
      `/api/admin/attachments/decision/${decisionRef}`,
      form,
      { headers: { "Content-Type": "multipart/form-data" } }
    );
  },

  uploadToActionItem: (actionItemRef: string, file: File) => {
    const form = new FormData();
    form.append("file", file);
    return apiClient.post<{ id: string }>(
      `/api/admin/attachments/action-item/${actionItemRef}`,
      form,
      { headers: { "Content-Type": "multipart/form-data" } }
    );
  },
};

export default attachmentsApi;
