// app/api/routes/domain-registry.ts
import apiClient from "@/lib/apiClient";
import { wrapApiWithTokenRefresh } from "@/lib/withTokenRefresh";

type ListParams = {
  page: number;
  pageSize: number;
  search?: string;
  company?: string;
  expBefore?: string;
  autoRenew?: "yes" | "no";
  cancel?: "yes" | "no";
  sortField: string;
  sortDir: "asc" | "desc";
};

const baseDomainRegistryApi = {
  // Accept EITHER a query string or a params object
  getList: (queryOrParams: string | ListParams) => {
    if (typeof queryOrParams === "string") {
      return apiClient.get(`/api/admin/domain-registry/list${queryOrParams}`);
    }
    return apiClient.get(`/api/admin/domain-registry/list`, {
      params: queryOrParams,
    });
  },

  upload: (form: FormData) =>
    apiClient.post(`/api/admin/domain-registry/upload`, form),

  exportCsv: (query: string) =>
    apiClient.get(
      `/api/admin/domain-registry/export${query}${query ? "&" : "?"}format=csv`,
      { responseType: "blob" }
    ),

  exportXlsx: (query: string) =>
    apiClient.get(
      `/api/admin/domain-registry/export${query}${
        query ? "&" : "?"
      }format=xlsx`,
      { responseType: "blob" }
    ),

  companyOptions: (params: { kind?: string; q?: string }) =>
    apiClient.get("/api/admin/domain-registry/options", { params }),

  create: (payload: {
    domain_name: string;
    current_company_id?: string | null;
    new_biller_id?: string | null;
    current_company?: string | null;
    new_biller?: string | null;
    registrar?: string | null;
    expires_on?: string | null;
    auto_renew?: boolean;
    cancel_flag?: boolean;
    notes?: string | null;
  }) => apiClient.post(`/api/admin/domain-registry`, payload),

  update: (
    id: string,
    payload: Partial<{
      domain_name: string;
      current_company_id: string | null;
      new_biller_id: string | null;
      current_company: string | null;
      new_biller: string | null;
      registrar: string | null;
      expires_on: string | null;
      auto_renew: boolean;
      cancel_flag: boolean;
      notes: string | null;
    }>
  ) => apiClient.put(`/api/admin/domain-registry/${id}`, payload),

  remove: (id: string) => apiClient.delete(`/api/admin/domain-registry/${id}`),

  getOne: (id: string) => apiClient.get(`/api/admin/domain-registry/${id}`),
};

const domainRegistryApi = wrapApiWithTokenRefresh(baseDomainRegistryApi, []);
export default domainRegistryApi;
