// src/api/hr-admin.ts (or inside your central api-calls file)
import apiClient from "@/lib/apiClient";
import { wrapApiWithTokenRefresh } from "@/lib/withTokenRefresh";

const baseHrAdminApi = {
  // task categories
  getTaskCategories: () => apiClient.get(`/api/admin/hr/task-categories`),
  upsertTaskCategory: (payload: any) =>
    apiClient.post(`/api/admin/hr/task-categories`, payload),
  deleteTaskCategory: (id: string) =>
    apiClient.delete(`/api/admin/hr/task-categories/${id}`),

  // task templates
  getTaskTemplates: () => apiClient.get(`/api/admin/hr/task-templates`),
  getTaskTemplate: (id: string) =>
    apiClient.get(`/api/admin/hr/task-templates/${id}`),
  upsertTaskTemplate: (payload: any) =>
    apiClient.post(`/api/admin/hr/task-templates`, payload),
  deleteTaskTemplate: (id: string) =>
    apiClient.delete(`/api/admin/hr/task-templates/${id}`),

  // doc templates list (for mapping to tasks)
  getDocTemplates: () => apiClient.get(`/api/admin/hr/document-templates`),

  getAdminDocTemplates() {
    return apiClient.get("/api/admin/hr/document-templates/admin");
  },

  upsertDocTemplate(payload: {
    id?: string;
    code: string;
    name: string;
    content: string;
    sort_order: number | null;
    is_active: boolean;
  }) {
    return apiClient.post("/api/admin/hr/document-templates", payload);
  },

  deleteDocTemplate(id: string) {
    return apiClient.delete(`/api/admin/hr/document-templates/${id}`);
  },

  generatePersonDocument(personId: string, companyId: string, code: string) {
    return apiClient.get(
      `/api/admin/people/${personId}/companies/${companyId}/documents/${encodeURIComponent(
        code
      )}`,
      {
        responseType: "text", // HTML
      }
    );
  },

  // NEW: list saved documents for one person
  listPersonDocuments(personId: string, params?: { companyId?: string }) {
    const qs = params?.companyId
      ? `?companyId=${encodeURIComponent(params.companyId)}`
      : "";
    // FIX: drop the /hr in the middle to match router.get("/people/:personId/documents")
    return apiClient.get(`/api/admin/people/${personId}/documents${qs}`);
  },
  getPersonDocument: (id: string) =>
    apiClient.get(`/api/admin/hr/person-documents/${id}`, {
      responseType: "text", // because backend returns HTML, not JSON
    }),
  listTaskPackages() {
    return apiClient.get("/api/admin/hr/task-packages");
  },
  getTaskPackage(id: string) {
    return apiClient.get(`/api/admin/hr/task-packages/${id}`);
  },
  upsertTaskPackage(payload: {
    id?: string;
    code: string;
    name: string;
    description?: string | null;
    kind: "onboarding" | "offboarding" | "other";
    is_active: boolean;
  }) {
    return apiClient.post("/api/admin/hr/task-packages", payload);
  },
  saveTaskPackageItems(
    packageId: string,
    payload: {
      items: {
        id?: string;
        template_id: string;
        sort_order: number;
        offset_days: number | null;
      }[];
    }
  ) {
    return apiClient.post(
      `/api/admin/hr/task-packages/${packageId}/items`,
      payload
    );
  },
  listTaskTemplates() {
    return apiClient.get("/api/admin/hr/task-templates");
  },
  getPersonDocumentAuditLog(id: string) {
    return apiClient.get(`/api/admin/hr/person-documents/${id}/audit-log`);
  },
  // 🔹 NEW: meta (status, form_values, template_document_id, etc.)
  getPersonDocumentMeta(id: string) {
    return apiClient.get(`/api/admin/hr/person-documents/${id}/meta`);
  },

  // 🔹 NEW: update form values (will re-render HTML on backend)
  updatePersonDocumentValues(id: string, values: Record<string, any>) {
    return apiClient.post(`/api/admin/hr/person-documents/${id}/update`, {
      values,
    });
  },

  // 🔹 NEW: approve (will send emails + set status)
  // approvePersonDocument(id: string) {
  //   return apiClient.post(`/api/admin/hr/person-documents/${id}/approve`, {});
  // },

  // 🔹 NEW: reject with optional reason
  // rejectPersonDocument(id: string, reason?: string) {
  //   return apiClient.post(`/api/admin/hr/person-documents/${id}/reject`, {
  //     reason: reason ?? null,
  //   });
  // },
  submitPersonDocumentForApproval: (id: string) =>
    apiClient.post(`/api/admin/hr/person-documents/${id}/submit`, {}),
  decidePersonDocument(
    id: string,
    decision: "approved" | "rejected",
    comment?: string
  ) {
    return apiClient.post(`/api/admin/hr/person-documents/${id}/decision`, {
      decision,
      comment,
    });
  },
  getPersonDocumentApprovers: (id: string) =>
    apiClient.get(`/api/admin/hr/person-documents/${id}/approvers`, {}),
  searchApprovers: ({ search }) =>
    apiClient.get("/api/admin/hr/approvers", { params: { search } }),
  getMyApprovalsSummary() {
    return apiClient.get("/api/admin/hr/my-approvals/summary");
  },
  getPersonDocumentApproversList: (id: string) =>
    apiClient.get(`/api/admin/hr/person-documents/${id}/approvers`, {}),
  getMyApprovalsList() {
    return apiClient.get("/api/admin/hr/my-approvals");
  },
};

const hrAdminApi = wrapApiWithTokenRefresh(baseHrAdminApi, []);
export default hrAdminApi;
