"use client";
import { useEffect, useMemo, useRef, useState } from "react";
import aclApi from "@/api/routes/acl";
import { safeToast } from "@/lib/safeToast";

type Role = { id: string; name: string };
type Resource = {
  id: string;
  key: string;
  pattern: string;
  description?: string | null;
};
type Effect = "allow" | "deny" | null;
const ACTIONS = ["read", "create", "update", "delete", "manage"] as const;
type Action = (typeof ACTIONS)[number];

type RoleGrantRow = {
  resource_id: string;
  resource_key: string;
  resource_pattern: string;
  read: Effect;
  create: Effect;
  update: Effect;
  delete: Effect;
  manage: Effect;
};

type GrantsMap = Record<string, Record<Action, Effect>>;

const EMPTY_GRANTS: Record<Action, Effect> = {
  read: null,
  create: null,
  update: null,
  delete: null,
  manage: null,
};

const ROLE_ORDER = [
  "super-admin",
  "site-admin",
  "admin",
  "editor",
  "writer",
  "user",
  "finance",
  "legal",
  "hr",
] as const;

export default function PermissionsPage() {
  const [roles, setRoles] = useState<Role[]>([]);
  const [resources, setResources] = useState<Resource[]>([]);
  const [selectedRole, setSelectedRole] = useState<string>("");
  const [grants, setGrants] = useState<GrantsMap>({});
  const [loading, setLoading] = useState(true);
  const [columnBusy, setColumnBusy] = useState<Record<Action, boolean>>({
    read: false,
    create: false,
    update: false,
    delete: false,
    manage: false,
  });
  const roleIndex = (name: string) => {
    const i = ROLE_ORDER.indexOf(name as (typeof ROLE_ORDER)[number]);
    return i === -1 ? Number.POSITIVE_INFINITY : i;
  };

  // load roles/resources
  useEffect(() => {
    (async () => {
      try {
        const [r, res] = await Promise.all([
          aclApi.listRoles(),
          aclApi.listResources(),
        ]);

        const fetchedRoles: Role[] = r.data.roles ?? [];
        const sortedRoles = [...fetchedRoles].sort(
          (a, b) => roleIndex(a.name) - roleIndex(b.name)
        );

        setRoles(sortedRoles);
        setResources(res.data.resources ?? []);

        const first = sortedRoles[0]?.id ?? "";
        setSelectedRole(first);
      } catch (e: any) {
        safeToast.error(
          e?.response?.data?.message || "Failed to load roles/resources"
        );
      }
    })();
  }, []);

  // load grants for current role
  useEffect(() => {
    if (!selectedRole) return;
    (async () => {
      setLoading(true);
      try {
        const resp = await aclApi.listRolePermissions(selectedRole);
        const arr: RoleGrantRow[] = resp.data.grants ?? [];
        const map: GrantsMap = {};
        arr.forEach((g) => {
          map[g.resource_id] = {
            read: g.read ?? null,
            create: g.create ?? null,
            update: g.update ?? null,
            delete: g.delete ?? null,
            manage: g.manage ?? null,
          };
        });
        setGrants(map);
      } catch (e: any) {
        safeToast.error(
          e?.response?.data?.message || "Failed to load role grants"
        );
      } finally {
        setLoading(false);
      }
    })();
  }, [selectedRole]);

  const isChecked = (resId: string, action: Action) =>
    (grants[resId]?.[action] ?? null) === "allow";

  const apply = async (
    roleId: string,
    resourceId: string,
    action: Action,
    checked: boolean
  ) => {
    try {
      if (checked) {
        await aclApi.grantRolePermission({
          roleId,
          resourceId,
          action,
          effect: "allow",
        });
        setGrants((prev) => {
          const next = { ...prev };
          const base = next[resourceId] ?? EMPTY_GRANTS;
          next[resourceId] = { ...base, [action]: "allow" };
          return next;
        });
      } else {
        await aclApi.revokeRolePermission({ roleId, resourceId, action });
        setGrants((prev) => {
          const next = { ...prev };
          const base = next[resourceId] ?? EMPTY_GRANTS;
          next[resourceId] = { ...base, [action]: null };
          return next;
        });
      }
    } catch (e: any) {
      safeToast.error(e?.response?.data?.message || "Update failed");
    }
  };

  /** ---------- Column helpers (select all / clear all) ---------- */

  // compute column state: "all" | "some" | "none"
  const columnState = (action: Action): "all" | "some" | "none" => {
    if (!resources.length) return "none";
    let on = 0;
    for (const r of resources) {
      if (isChecked(r.id, action)) on++;
    }
    if (on === 0) return "none";
    if (on === resources.length) return "all";
    return "some";
  };

  // bulk toggle column to target (true = select all, false = clear all)
  // bulk toggle column to target (true = select all, false = clear all)
  const toggleColumn = async (action: Action, target: boolean) => {
    if (!selectedRole || !resources.length) return;

    // which resources actually need changing?
    const todo = resources
      .filter((r) => isChecked(r.id, action) !== target)
      .map((r) => r.id);

    if (todo.length === 0) return;

    setColumnBusy((b) => ({ ...b, [action]: true }));

    // --- OPTIMISTIC UPDATE (typed & full rows) ---
    // if your TS/lib doesn't have structuredClone, replace with:
    // const snapshot: GrantsMap = JSON.parse(JSON.stringify(grants));
    const snapshot: GrantsMap = structuredClone(grants);

    setGrants((prev) => {
      const next: GrantsMap = { ...prev };
      for (const resId of todo) {
        const base = next[resId] ?? EMPTY_GRANTS;
        next[resId] = { ...base, [action]: target ? "allow" : null };
      }
      return next;
    });

    try {
      // run in small batches to be gentle with the API/DB
      const BATCH = 15;
      for (let i = 0; i < todo.length; i += BATCH) {
        const chunk = todo.slice(i, i + BATCH);
        await Promise.all(
          chunk.map((resId) =>
            target
              ? aclApi.grantRolePermission({
                  roleId: selectedRole,
                  resourceId: resId,
                  action,
                  effect: "allow",
                })
              : aclApi.revokeRolePermission({
                  roleId: selectedRole,
                  resourceId: resId,
                  action,
                })
          )
        );
      }
      safeToast.success(
        `${target ? "Granted" : "Cleared"} "${action}" on ${
          todo.length
        } resource${todo.length === 1 ? "" : "s"}`
      );
    } catch (e: any) {
      // rollback on error
      setGrants(snapshot);
      safeToast.error(
        e?.response?.data?.message ||
          `Failed to ${target ? "grant" : "clear"} "${action}"`
      );
    } finally {
      setColumnBusy((b) => ({ ...b, [action]: false }));
    }
  };

  /** A small header checkbox that supports "indeterminate" visual state */
  const HeaderCheckbox: React.FC<{
    action: Action;
  }> = ({ action }) => {
    const state = columnState(action);
    const ref = useRef<HTMLInputElement>(null);

    useEffect(() => {
      if (ref.current) {
        ref.current.indeterminate = state === "some";
      }
    }, [state]);

    return (
      <label className="inline-flex items-center gap-2 select-none">
        <input
          ref={ref}
          type="checkbox"
          className="align-middle"
          checked={state === "all"}
          onChange={(e) => toggleColumn(action, e.currentTarget.checked)}
          disabled={columnBusy[action] || loading}
          title={
            columnBusy[action]
              ? "Working…"
              : state === "all"
              ? `Clear all ${action}`
              : `Select all ${action}`
          }
        />
        <span className="capitalize">{action}</span>
        {columnBusy[action] && <span className="text-xs text-gray-500">…</span>}
      </label>
    );
  };

  return (
    <div className="p-6 space-y-4">
      <h1 className="text-2xl font-semibold">
        Permissions (Roles → Resources)
      </h1>

      <div className="flex items-center gap-3">
        <label className="text-sm">Role</label>
        <select
          className="border rounded px-2 py-1"
          value={selectedRole}
          onChange={(e) => setSelectedRole(e.currentTarget.value)}>
          {roles.map((r) => (
            <option key={r.id} value={r.id}>
              {r.name}
            </option>
          ))}
        </select>
      </div>

      {!selectedRole ? (
        <p>Pick a role</p>
      ) : (
        <div className="overflow-auto border rounded">
          <table className="w-full text-sm">
            <thead>
              <tr className="bg-gray-50">
                <th className="p-2 text-left">Resource</th>
                {ACTIONS.map((a) => (
                  <th key={a} className="p-2">
                    {/* Column header with select-all/clear-all */}
                    <HeaderCheckbox action={a} />
                  </th>
                ))}
              </tr>
            </thead>
            <tbody>
              {resources.map((res) => (
                <tr key={res.id} className="border-t">
                  <td className="p-2">
                    <div className="font-mono">{res.key}</div>
                    <div className="text-xs text-gray-500">{res.pattern}</div>
                  </td>
                  {ACTIONS.map((a) => (
                    <td key={a} className="p-2 text-center">
                      <input
                        type="checkbox"
                        checked={isChecked(res.id, a)}
                        onChange={(e) =>
                          apply(
                            selectedRole,
                            res.id,
                            a,
                            e.currentTarget.checked
                          )
                        }
                        disabled={columnBusy[a] || loading}
                      />
                    </td>
                  ))}
                </tr>
              ))}
              {loading && (
                <tr>
                  <td className="p-2" colSpan={1 + ACTIONS.length}>
                    Loading…
                  </td>
                </tr>
              )}
            </tbody>
          </table>
        </div>
      )}

      <p className="text-xs text-gray-500">
        Tri-state (allow/deny/none) is easy to add: display a three-state toggle
        and call <code>grantRolePermission(..., effect: deny)</code> for deny.
      </p>
    </div>
  );
}
