"use client";
import { useEffect, useMemo, useState } from "react";
import aclApi from "@/api/routes/acl";
import { safeToast } from "@/lib/safeToast";

type UserRow = {
  id: string;
  username: string;
  email: string;
  last_login: string | null;
  role_name: string | null;
};

type RoleRow = { id: string; name: string; description?: string | null };

export default function UsersPage() {
  const [users, setUsers] = useState<UserRow[]>([]);
  const [roles, setRoles] = useState<RoleRow[]>([]);
  const [loading, setLoading] = useState(true);
  const [creating, setCreating] = useState(false);
  const [roleBusy, setRoleBusy] = useState<Record<string, boolean>>({});
  const fetchAll = async () => {
    setLoading(true);
    try {
      const [u, r] = await Promise.all([
        aclApi.listUsers(),
        aclApi.listRoles(),
      ]);
      setUsers(u.data.users ?? u.data ?? []);
      setRoles(r.data.roles ?? r.data ?? []);
    } catch (e: any) {
      safeToast.error(
        e?.response?.data?.message || "Failed to load users/roles"
      );
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchAll();
  }, []);

  const handleCreate = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();

    if (creating) return; // 🚫 guard against double submit
    setCreating(true);
    const fd = new FormData(e.currentTarget);
    const payload = {
      username: String(fd.get("username") || "").trim(),
      email: String(fd.get("email") || "").trim(),
      password: String(fd.get("password") || "").trim(),
      roleId: String(fd.get("roleId") || "") || undefined,
      // title: String(fd.get("title") || "").trim() || undefined,
      // name: String(fd.get("name") || "").trim() || undefined,
    };
    if (!payload.username || !payload.email || !payload.password) {
      return safeToast.error("username, email, password required");
    }

    // 🔒 Only the create call is inside this try; if it throws -> we show "Create failed".
    await aclApi.createUser(payload);

    safeToast.success("User created");
    // e.currentTarget.reset();

    // Refresh list, but don’t show "Create failed" on errors from here.
    try {
      await fetchAll();
    } catch {
      // fetchAll already toasts "Failed to load users/roles" internally; do nothing here.
    }
  };

  const handleUpdate = async (
    id: string,
    patch: Partial<UserRow> & { password?: string; isActive?: boolean }
  ) => {
    try {
      await aclApi.updateUser(id, patch);
      safeToast.success("User updated");
      fetchAll();
    } catch (e: any) {
      safeToast.error(e?.response?.data?.message || "Update failed");
    }
  };

  const handleDelete = async (id: string) => {
    if (!confirm("Delete user?")) return;
    try {
      await aclApi.deleteUser(id);
      safeToast.success("User deleted");
      fetchAll();
    } catch (e: any) {
      safeToast.error(e?.response?.data?.message || "Delete failed");
    }
  };

  const handleAssignRole = async (userId: string, roleId: string) => {
    try {
      await aclApi.assignRoleToUser(userId, roleId);
      safeToast.success("Role assigned");
      fetchAll();
    } catch (e: any) {
      safeToast.error(e?.response?.data?.message || "Assign role failed");
    }
  };

  const handleRevokeRole = async (userId: string, roleId: string) => {
    try {
      await aclApi.revokeRoleFromUser(userId, roleId);
      safeToast.success("Role revoked");
      fetchAll();
    } catch (e: any) {
      safeToast.error(e?.response?.data?.message || "Revoke role failed");
    }
  };
  const changeUserRole = async (user: UserRow, newRoleId: string) => {
    if (!newRoleId) return;
    setRoleBusy((b) => ({ ...b, [user.id]: true }));
    try {
      const currentRoleId = roleByName[user.role_name || ""]?.id;
      if (currentRoleId && currentRoleId !== newRoleId) {
        await aclApi.revokeRoleFromUser(user.id, currentRoleId);
      }
      await aclApi.assignRoleToUser(user.id, newRoleId);

      // optimistic UI update
      const newName = roles.find((r) => r.id === newRoleId)?.name ?? null;
      setUsers((prev) =>
        prev.map((u) => (u.id === user.id ? { ...u, role_name: newName } : u))
      );

      safeToast.success("Role updated");
    } catch (e: any) {
      safeToast.error(e?.response?.data?.message || "Update role failed");
      // optional: refetch to get back in sync
      fetchAll();
    } finally {
      setRoleBusy((b) => ({ ...b, [user.id]: false }));
    }
  };
  const roleByName = useMemo(
    () => Object.fromEntries(roles.map((r) => [r.name, r])),
    [roles]
  );

  return (
    <div className="p-6 space-y-6">
      <h1 className="text-2xl font-semibold">Users</h1>

      {/* Create user */}
      <form
        onSubmit={handleCreate}
        autoComplete="off"
        className="grid grid-cols-1 md:grid-cols-5 gap-2 items-end">
        <div className="sr-only" aria-hidden="true">
          <input
            type="text"
            name="fake-username"
            autoComplete="username"
            tabIndex={-1}
          />
          <input
            type="password"
            name="fake-password"
            autoComplete="current-password"
            tabIndex={-1}
          />
        </div>
        <div>
          <label className="block text-sm">Username</label>
          <input
            name="username"
            className="border rounded px-2 py-1 w-full"
            autoComplete="off"
            autoCapitalize="none"
            spellCheck={false}
            inputMode="text"
          />
        </div>
        <div>
          <label className="block text-sm">Email</label>
          <input
            name="email"
            type="email"
            className="border rounded px-2 py-1 w-full"
            autoComplete="off" // don’t hint it's an email field, to avoid suggestions
            autoCapitalize="none"
            spellCheck={false}
            inputMode="email"
          />
        </div>
        <div>
          <label className="block text-sm">Password</label>
          <input
            name="password"
            type="password"
            className="border rounded px-2 py-1 w-full"
            autoComplete="new-password" // creation flow -> use new-password
            // optional trick: make it readOnly until focus, then remove to deter some autofills
            readOnly
            onFocus={(e) => e.currentTarget.removeAttribute("readOnly")}
          />
        </div>
        <div>
          <label className="block text-sm">Initial role</label>
          <select name="roleId" className="border rounded px-2 py-1 w-full">
            <option value="">(none)</option>
            {roles.map((r) => (
              <option key={r.id} value={r.id}>
                {r.name}
              </option>
            ))}
          </select>
        </div>
        <button className="bg-black text-white rounded px-3 py-2">
          Create
        </button>
      </form>

      {/* List */}
      <div className="border rounded">
        <table className="w-full text-sm">
          <thead>
            <tr className="bg-gray-50 text-left">
              <th className="p-2">Username</th>
              <th className="p-2">Email</th>
              <th className="p-2">Role</th>
              <th className="p-2">Last login</th>
              <th className="p-2 w-64">Actions</th>
            </tr>
          </thead>
          <tbody>
            {loading ? (
              <tr>
                <td className="p-3" colSpan={5}>
                  Loading…
                </td>
              </tr>
            ) : users.length === 0 ? (
              <tr>
                <td className="p-3" colSpan={5}>
                  No users
                </td>
              </tr>
            ) : (
              users.map((u) => (
                <tr key={u.id} className="border-t">
                  <td className="p-2">{u.username}</td>
                  <td className="p-2">{u.email}</td>
                  <td className="p-2">
                    <div className="flex gap-2 items-center">
                      <span className="inline-block rounded bg-gray-100 px-2 py-0.5">
                        {u.role_name || "user"}
                      </span>
                      <select
                        className="border rounded px-2 py-1"
                        value={roleByName[u.role_name || ""]?.id || ""} // make it controlled
                        disabled={roleBusy[u.id]}
                        onChange={(e) =>
                          changeUserRole(u, e.currentTarget.value)
                        }>
                        <option value="">(pick role)</option>
                        {roles.map((r) => (
                          <option key={r.id} value={r.id}>
                            {r.name}
                          </option>
                        ))}
                      </select>

                      {/* <select
                        className="border rounded px-2 py-1"
                        defaultValue={roleByName[u.role_name || ""]?.id || ""}
                        onChange={(e) => {
                          const newRoleId = e.currentTarget.value;
                          if (!newRoleId) return;
                          handleAssignRole(u.id, newRoleId);
                        }}>
                        <option value="">(pick role)</option>
                        {roles.map((r) => (
                          <option key={r.id} value={r.id}>
                            {r.name}
                          </option>
                        ))}
                      </select> */}
                      {u.role_name && roleByName[u.role_name] && (
                        <button
                          className="text-red-600 underline"
                          onClick={() =>
                            handleRevokeRole(u.id, roleByName[u.role_name!].id)
                          }>
                          revoke
                        </button>
                      )}
                    </div>
                  </td>
                  <td className="p-2">
                    {u.last_login
                      ? new Date(u.last_login).toLocaleString()
                      : "-"}
                  </td>
                  <td className="p-2">
                    <div className="flex gap-2">
                      <button
                        className="border rounded px-2 py-1"
                        onClick={() => {
                          const email = prompt("New email", u.email || "");
                          if (email) handleUpdate(u.id, { email });
                        }}>
                        Edit
                      </button>
                      <button
                        className="border rounded px-2 py-1"
                        onClick={() => {
                          const pwd = prompt("Set new password");
                          if (pwd) handleUpdate(u.id, { password: pwd });
                        }}>
                        Set password
                      </button>
                      <button
                        className="text-red-600 underline"
                        onClick={() => handleDelete(u.id)}>
                        Delete
                      </button>
                    </div>
                  </td>
                </tr>
              ))
            )}
          </tbody>
        </table>
      </div>
    </div>
  );
}
