"use client";

import { useEffect, useMemo, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import type {
  ActionItem,
  ActionItemStatus,
  ActionItemAssignee,
} from "@/api/routes/action-items";
import AssigneesPicker from "@/components/cases/AssigneesPicker";
import { useRouter, useSearchParams } from "next/navigation";

import {
  Page,
  Card,
  ToolbarRow,
  TextInput,
  Select,
  Button,
  SecondaryButton,
  DangerButton,
  GridCards,
  CardTitle,
  CardMeta,
  Chip,
  SectionTitle,
  EmptyState,
  TextArea,
  Divider,
  Field,
} from "@/components/ui/CardBits";

import { Drawer } from "@/components/ui/Drawer";
import { Tabs } from "@/components/ui/Tabs";

function fmt(iso?: string | null) {
  if (!iso) return "-";
  try {
    return new Date(iso).toLocaleString();
  } catch {
    return iso ?? "-";
  }
}
function isOverdue(due?: string | null) {
  if (!due) return false;
  return new Date(due).getTime() < Date.now();
}

type ViewTab = "mine" | "all" | "archived";

export default function ActionItemsClient() {
  const router = useRouter();
  const searchParams = useSearchParams();

  const [tab, setTab] = useState<ViewTab>("mine");

  const [items, setItems] = useState<ActionItem[]>([]);
  const [loading, setLoading] = useState(false);

  // filters
  const [q, setQ] = useState("");
  const [status, setStatus] = useState<ActionItemStatus | "">("");

  // create drawer
  const [createOpen, setCreateOpen] = useState(false);
  const [caseId, setCaseId] = useState("");
  const [title, setTitle] = useState("");
  const [creating, setCreating] = useState(false);

  // edit drawer
  const [open, setOpen] = useState(false);
  const [current, setCurrent] = useState<ActionItem | null>(null);

  const [eTitle, setETitle] = useState("");
  const [eDesc, setEDesc] = useState("");
  const [eStatus, setEStatus] = useState<ActionItemStatus>("open");
  const [eDue, setEDue] = useState("");

  const [assignees, setAssignees] = useState<Array<{ user: any; role: any }>>(
    []
  );
  const [loadingAssignees, setLoadingAssignees] = useState(false);
  const [saving, setSaving] = useState(false);

  const includeArchived = tab === "archived" ? 1 : 0;
  const mine = tab === "mine" ? 1 : 0;

  async function load() {
    setLoading(true);
    try {
      const r = await ApiCall.actionItems.list({
        mine,
        includeArchived,
        q: q.trim() || undefined,
        status: status || undefined,
      });
      setItems(r.data.items ?? []);
    } catch (e: any) {
      toast.error(
        e?.response?.data?.message ?? "Action items betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  }

  useEffect(() => {
    const qp = (searchParams.get("q") ?? "").trim();
    if (qp) {
      setQ(qp);
      setTab("all");
    }
    void load();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  useEffect(() => {
    void load();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [tab]);

  async function onCreate() {
    if (!caseId.trim() || !title.trim()) return;
    setCreating(true);
    try {
      await ApiCall.actionItems.create({
        caseId: caseId.trim(),
        title: title.trim(),
      });
      toast.success("Feladat létrehozva");
      setTitle("");
      setCaseId("");
      setCreateOpen(false);
      await load();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Létrehozás sikertelen");
    } finally {
      setCreating(false);
    }
  }

  async function openEdit(it: ActionItem) {
    setCurrent(it);
    setETitle(it.title ?? "");
    setEDesc(it.description ?? "");
    setEStatus((it.status ?? "open") as any);
    setEDue(it.due_at ? new Date(it.due_at).toISOString().slice(0, 16) : "");
    setOpen(true);

    setLoadingAssignees(true);
    try {
      const r = await ApiCall.actionItems.assignees.get(it.id);
      const a = (r.data.assignees ?? []) as ActionItemAssignee[];
      setAssignees(
        a.map((x) => ({
          user: {
            id: x.user_id,
            name: x.name,
            email: x.email,
            username: x.username,
          },
          role: x.role,
        }))
      );
    } catch {
      setAssignees([]);
    } finally {
      setLoadingAssignees(false);
    }
  }

  async function saveEdit() {
    if (!current) return;
    setSaving(true);
    try {
      await ApiCall.actionItems.patch(current.id, {
        title: eTitle.trim() || undefined,
        description: eDesc ?? undefined,
        status: eStatus,
        dueAt: eDue ? new Date(eDue).toISOString() : null,
      } as any);

      await ApiCall.actionItems.assignees.patch(current.id, {
        assignees: assignees.map((x) => ({ userId: x.user.id, role: x.role })),
      });

      toast.success("Mentve");
      setOpen(false);
      setCurrent(null);
      await load();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Mentés sikertelen");
    } finally {
      setSaving(false);
    }
  }

  async function archive(id: string) {
    if (!confirm("Archiválod ezt a feladatot?")) return;
    try {
      await ApiCall.actionItems.archive(id);
      toast.success("Archiválva");
      await load();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Archiválás sikertelen");
    }
  }

  async function unarchive(id: string) {
    if (!confirm("Visszaállítod ezt a feladatot?")) return;
    try {
      await ApiCall.actionItems.unarchive(id);
      toast.success("Visszaállítva");
      await load();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Visszaállítás sikertelen");
    }
  }

  const visible = useMemo(() => {
    // backend már szűr q/status szerint, itt csak extra: archived tabnál csak archived
    if (tab === "archived") return items.filter((x) => !!x.archived_at);
    return items.filter((x) => !x.archived_at);
  }, [items, tab]);

  const counts = useMemo(() => {
    return {
      total: visible.length,
      overdue: visible.filter((x) => isOverdue(x.due_at)).length,
      done: visible.filter((x) => x.status === "done").length,
      blocked: visible.filter((x) => x.status === "blocked").length,
    };
  }, [visible]);

  return (
    <Page
      title="Action Items"
      subtitle={`összes: ${counts.total} · lejárt: ${counts.overdue} · done: ${counts.done} · blocked: ${counts.blocked}`}
      actions={
        <>
          <SecondaryButton onClick={() => setCreateOpen(true)}>
            + Új
          </SecondaryButton>
          <SecondaryButton onClick={() => void load()} disabled={loading}>
            {loading ? "..." : "Frissít"}
          </SecondaryButton>
        </>
      }>
      <div className="mb-4 flex flex-wrap items-center justify-between gap-3">
        <Tabs<ViewTab>
          value={tab}
          onChange={setTab}
          tabs={[
            { id: "mine", label: "Csak az enyém" },
            { id: "all", label: "Minden" },
            { id: "archived", label: "Archivált" },
          ]}
        />

        <div className="flex flex-wrap items-center gap-2">
          <TextInput
            value={q}
            onChange={(e) => setQ(e.target.value)}
            placeholder="Keresés…"
            style={{ minWidth: 260 }}
            onKeyDown={(e: any) => {
              if (e.key === "Enter") void load();
            }}
          />
          <Select
            value={status}
            onChange={(e) => setStatus(e.target.value as any)}>
            <option value="">Összes státusz</option>
            <option value="open">open</option>
            <option value="in_progress">in_progress</option>
            <option value="blocked">blocked</option>
            <option value="done">done</option>
            <option value="cancelled">cancelled</option>
          </Select>

          <Button onClick={() => void load()} disabled={loading}>
            Keres
          </Button>

          <SecondaryButton
            onClick={() => {
              setQ("");
              setStatus("");
            }}
            disabled={!q.trim() && !status}>
            Töröl
          </SecondaryButton>
        </div>
      </div>

      <SectionTitle>Lista</SectionTitle>

      {visible.length ? (
        <GridCards>
          {visible.map((it) => {
            const overdue = isOverdue(it.due_at);
            const tone =
              it.status === "done"
                ? "good"
                : it.status === "blocked"
                ? "bad"
                : overdue
                ? "warn"
                : "neutral";

            return (
              <Card key={it.id}>
                <CardTitle>{it.title}</CardTitle>
                <CardMeta>
                  {it.case_title
                    ? `Ügy: ${it.case_title}`
                    : `Ügy: ${it.case_id?.slice(0, 8) ?? "—"}…`}
                </CardMeta>

                <div className="mt-2 flex flex-wrap gap-2">
                  <Chip tone={tone as any}>{it.status}</Chip>
                  <Chip tone={overdue ? "warn" : "neutral"}>
                    due: {fmt(it.due_at)}
                  </Chip>
                  {it.code ? <Chip tone="info">{it.code}</Chip> : null}
                </div>

                <Divider />

                <ToolbarRow>
                  <SecondaryButton
                    onClick={() =>
                      router.push(`/admin/cases/${it.caseRef ?? it.case_id}`)
                    }>
                    Ügy
                  </SecondaryButton>
                  <SecondaryButton
                    onClick={() => openEdit(it)}
                    disabled={!!it.archived_at}>
                    Szerkeszt
                  </SecondaryButton>

                  {it.archived_at ? (
                    <SecondaryButton onClick={() => void unarchive(it.id)}>
                      Visszaállítás
                    </SecondaryButton>
                  ) : (
                    <DangerButton onClick={() => void archive(it.id)}>
                      Archivál
                    </DangerButton>
                  )}
                  {/* <DangerButton
                    onClick={() => void archive(it.id)}
                    disabled={!!it.archived_at}>
                    Archivál
                  </DangerButton> */}
                </ToolbarRow>
              </Card>
            );
          })}
        </GridCards>
      ) : (
        <EmptyState
          title="Nincs találat"
          hint="Próbálj másik szűrőt vagy hozz létre újat."
        />
      )}

      {/* Create drawer */}
      <Drawer
        open={createOpen}
        onClose={() => setCreateOpen(false)}
        title="Új Action Item"
        widthClass="max-w-lg">
        <div className="grid gap-3">
          <Field label="CaseId (UUID)" hint="kötelező">
            <TextInput
              value={caseId}
              onChange={(e) => setCaseId(e.target.value)}
              placeholder="..."
            />
          </Field>
          <Field label="Cím" hint="kötelező">
            <TextInput
              value={title}
              onChange={(e) => setTitle(e.target.value)}
              placeholder="Feladat címe"
            />
          </Field>
          <Button
            onClick={() => void onCreate()}
            disabled={creating || !caseId.trim() || !title.trim()}>
            {creating ? "..." : "Létrehoz"}
          </Button>
        </div>
      </Drawer>

      {/* Edit drawer */}
      <Drawer
        open={open}
        onClose={() => setOpen(false)}
        title="Szerkesztés"
        widthClass="max-w-2xl">
        {current ? (
          <div className="grid gap-3">
            <Field label="Cím">
              <TextInput
                value={eTitle}
                onChange={(e) => setETitle(e.target.value)}
              />
            </Field>

            <Field label="Leírás">
              <TextArea
                value={eDesc}
                onChange={(e) => setEDesc(e.target.value)}
                placeholder="..."
              />
            </Field>

            <div className="grid gap-3 md:grid-cols-2">
              <Field label="Státusz">
                <Select
                  value={eStatus}
                  onChange={(e) => setEStatus(e.target.value as any)}>
                  <option value="open">open</option>
                  <option value="in_progress">in_progress</option>
                  <option value="blocked">blocked</option>
                  <option value="done">done</option>
                  <option value="cancelled">cancelled</option>
                </Select>
              </Field>

              <Field label="Határidő">
                <TextInput
                  type="datetime-local"
                  value={eDue}
                  onChange={(e) => setEDue(e.target.value)}
                />
              </Field>
            </div>

            <Divider />

            <div>
              <CardMeta>Hozzárendeltek</CardMeta>
              {loadingAssignees ? (
                <Chip tone="neutral">assignees betöltése...</Chip>
              ) : (
                <AssigneesPicker
                  value={assignees as any}
                  onChange={setAssignees as any}
                />
              )}
            </div>

            <ToolbarRow>
              <Button
                onClick={() => void saveEdit()}
                disabled={saving || !eTitle.trim()}>
                {saving ? "..." : "Mentés"}
              </Button>
              <SecondaryButton onClick={() => setOpen(false)}>
                Mégse
              </SecondaryButton>
            </ToolbarRow>
          </div>
        ) : null}
      </Drawer>
    </Page>
  );
}

// "use client";

// import { useEffect, useMemo, useState } from "react";
// import toast from "react-hot-toast";
// import ApiCall from "@/api/api-calls";
// import type {
//   ActionItem,
//   ActionItemStatus,
//   ActionItemAssignee,
// } from "@/api/routes/action-items";
// import AssigneesPicker from "@/components/cases/AssigneesPicker";
// import { useRouter, useSearchParams } from "next/navigation";

// import {
//   Page,
//   Card,
//   ToolbarRow,
//   TextInput,
//   Select,
//   Button,
//   SecondaryButton,
//   DangerButton,
//   GridCards,
//   CardTitle,
//   CardMeta,
//   Chip,
//   SectionTitle,
//   EmptyState,
//   TextArea,
//   Divider,
//   Field,
// } from "@/components/ui/CardBits";

// import { Drawer } from "@/components/ui/Drawer";
// import { Collapsible } from "@/components/ui/Collapsible";

// function fmt(iso?: string | null) {
//   if (!iso) return "-";
//   try {
//     return new Date(iso).toLocaleString();
//   } catch {
//     return iso ?? "-";
//   }
// }
// function isOverdue(due?: string | null) {
//   if (!due) return false;
//   return new Date(due).getTime() < Date.now();
// }

// export default function ActionItemsPage() {
//   const router = useRouter();
//   const searchParams = useSearchParams();

//   const [items, setItems] = useState<ActionItem[]>([]);
//   const [loading, setLoading] = useState(false);

//   // filters
//   const [filtersOpen, setFiltersOpen] = useState(false);
//   const [q, setQ] = useState("");
//   const [mine, setMine] = useState<1 | 0>(1);
//   const [status, setStatus] = useState<ActionItemStatus | "">("");

//   // quick create (collapsed)
//   const [createOpen, setCreateOpen] = useState(false);
//   const [caseId, setCaseId] = useState("");
//   const [title, setTitle] = useState("");
//   const [creating, setCreating] = useState(false);

//   // drawer edit
//   const [open, setOpen] = useState(false);
//   const [current, setCurrent] = useState<ActionItem | null>(null);

//   const [eTitle, setETitle] = useState("");
//   const [eDesc, setEDesc] = useState("");
//   const [eStatus, setEStatus] = useState<ActionItemStatus>("open");
//   const [eDue, setEDue] = useState("");

//   const [assignees, setAssignees] = useState<Array<{ user: any; role: any }>>(
//     []
//   );
//   const [loadingAssignees, setLoadingAssignees] = useState(false);
//   const [saving, setSaving] = useState(false);

//   async function load() {
//     setLoading(true);
//     try {
//       const r = await ApiCall.actionItems.list({
//         ...(mine ? { mine: 1 } : {}),
//         q: q.trim() || undefined,
//         status: status || undefined,
//       });

//       setItems(r.data.items ?? []);
//     } catch (e: any) {
//       toast.error(
//         e?.response?.data?.message ?? "Action items betöltése sikertelen"
//       );
//     } finally {
//       setLoading(false);
//     }
//   }

//   useEffect(() => {
//     const qp = (searchParams.get("q") ?? "").trim();
//     if (qp) {
//       setQ(qp);
//       setMine(0);
//       setFiltersOpen(true);
//     }
//     void load();
//     // eslint-disable-next-line react-hooks/exhaustive-deps
//   }, []);

//   async function onCreate() {
//     if (!caseId.trim() || !title.trim()) return;
//     setCreating(true);
//     try {
//       await ApiCall.actionItems.create({
//         caseId: caseId.trim(),
//         title: title.trim(),
//       });
//       toast.success("Feladat létrehozva");
//       setTitle("");
//       setCreateOpen(false);
//       await load();
//     } catch (e: any) {
//       toast.error(e?.response?.data?.message ?? "Létrehozás sikertelen");
//     } finally {
//       setCreating(false);
//     }
//   }

//   async function openEdit(it: ActionItem) {
//     setCurrent(it);
//     setETitle(it.title ?? "");
//     setEDesc(it.description ?? "");
//     setEStatus((it.status ?? "open") as any);
//     setEDue(it.due_at ? new Date(it.due_at).toISOString().slice(0, 16) : "");
//     setOpen(true);

//     setLoadingAssignees(true);
//     try {
//       const r = await ApiCall.actionItems.assignees.get(it.id);
//       const a = (r.data.assignees ?? []) as ActionItemAssignee[];
//       setAssignees(
//         a.map((x) => ({
//           user: {
//             id: x.user_id,
//             name: x.name,
//             email: x.email,
//             username: x.username,
//           },
//           role: x.role,
//         }))
//       );
//     } catch {
//       setAssignees([]);
//     } finally {
//       setLoadingAssignees(false);
//     }
//   }

//   async function saveEdit() {
//     if (!current) return;
//     setSaving(true);
//     try {
//       await ApiCall.actionItems.patch(current.id, {
//         title: eTitle.trim() || undefined,
//         description: eDesc ?? undefined,
//         status: eStatus,
//         dueAt: eDue ? new Date(eDue).toISOString() : null,
//       } as any);

//       await ApiCall.actionItems.assignees.patch(current.id, {
//         assignees: assignees.map((x) => ({ userId: x.user.id, role: x.role })),
//       });

//       toast.success("Mentve");
//       setOpen(false);
//       setCurrent(null);
//       await load();
//     } catch (e: any) {
//       toast.error(e?.response?.data?.message ?? "Mentés sikertelen");
//     } finally {
//       setSaving(false);
//     }
//   }

//   async function archive(id: string) {
//     if (!confirm("Archiválod ezt a feladatot?")) return;
//     try {
//       await ApiCall.actionItems.archive(id);
//       toast.success("Archiválva");
//       await load();
//     } catch (e: any) {
//       toast.error(e?.response?.data?.message ?? "Archiválás sikertelen");
//     }
//   }

//   const counts = useMemo(() => {
//     return {
//       total: items.length,
//       overdue: items.filter((x) => isOverdue(x.due_at)).length,
//       done: items.filter((x) => x.status === "done").length,
//       blocked: items.filter((x) => x.status === "blocked").length,
//     };
//   }, [items]);

//   return (
//     <Page
//       title="Action Items"
//       subtitle={`összes: ${counts.total} · lejárt: ${counts.overdue} · done: ${counts.done} · blocked: ${counts.blocked}`}
//       actions={
//         <>
//           <SecondaryButton onClick={() => setCreateOpen(true)}>
//             + Új
//           </SecondaryButton>
//           <SecondaryButton onClick={() => setFiltersOpen((v) => !v)}>
//             Szűrők
//           </SecondaryButton>
//           <SecondaryButton onClick={() => void load()} disabled={loading}>
//             {loading ? "..." : "Frissít"}
//           </SecondaryButton>
//         </>
//       }>
//       <div className="grid gap-3">
//         <Collapsible
//           title="Szűrők"
//           open={filtersOpen}
//           onToggle={() => setFiltersOpen((v) => !v)}>
//           <div className="grid gap-3 md:grid-cols-3">
//             <Field label="Keresés">
//               <TextInput
//                 value={q}
//                 onChange={(e) => setQ(e.target.value)}
//                 placeholder="kód / cím / case..."
//               />
//             </Field>
//             <Field label="Státusz">
//               <Select
//                 value={status}
//                 onChange={(e) => setStatus(e.target.value as any)}>
//                 <option value="">Összes</option>
//                 <option value="open">open</option>
//                 <option value="in_progress">in_progress</option>
//                 <option value="blocked">blocked</option>
//                 <option value="done">done</option>
//                 <option value="cancelled">cancelled</option>
//               </Select>
//             </Field>
//             <Field label="Nézet">
//               <Select
//                 value={mine}
//                 onChange={(e) => setMine(Number(e.target.value) as any)}>
//                 <option value={1}>Csak az enyém</option>
//                 <option value={0}>Minden</option>
//               </Select>
//             </Field>
//           </div>
//           <div className="mt-3">
//             <Button onClick={() => void load()} disabled={loading}>
//               Keres
//             </Button>
//           </div>
//         </Collapsible>

//         <SectionTitle>Lista</SectionTitle>

//         {items.length ? (
//           <GridCards>
//             {items.map((it) => {
//               const overdue = isOverdue(it.due_at);
//               const tone =
//                 it.status === "done"
//                   ? "good"
//                   : it.status === "blocked"
//                   ? "bad"
//                   : overdue
//                   ? "warn"
//                   : "neutral";

//               return (
//                 <Card key={it.id}>
//                   <CardTitle>{it.title}</CardTitle>
//                   <CardMeta>
//                     {it.case_title
//                       ? `Ügy: ${it.case_title}`
//                       : `Ügy: ${it.case_id?.slice(0, 8) ?? "—"}…`}
//                   </CardMeta>

//                   <div className="mt-2 flex flex-wrap gap-2">
//                     <Chip tone={tone as any}>{it.status}</Chip>
//                     <Chip tone={overdue ? "warn" : "neutral"}>
//                       due: {fmt(it.due_at)}
//                     </Chip>
//                     {it.archived_at ? <Chip tone="warn">archived</Chip> : null}
//                   </div>

//                   <Divider />

//                   <ToolbarRow>
//                     <SecondaryButton
//                       onClick={() =>
//                         it.case_id
//                           ? router.push(`/admin/cases/${it.case_id}`)
//                           : null
//                       }>
//                       Ügy
//                     </SecondaryButton>
//                     <SecondaryButton
//                       onClick={() => openEdit(it)}
//                       disabled={!!it.archived_at}>
//                       Szerkeszt
//                     </SecondaryButton>
//                     <DangerButton
//                       onClick={() => void archive(it.id)}
//                       disabled={!!it.archived_at}>
//                       Archivál
//                     </DangerButton>
//                   </ToolbarRow>
//                 </Card>
//               );
//             })}
//           </GridCards>
//         ) : (
//           <EmptyState
//             title="Nincs találat"
//             hint="Próbálj másik szűrőt, vagy hozz létre új feladatot."
//           />
//         )}
//       </div>

//       {/* Create drawer */}
//       <Drawer
//         open={createOpen}
//         onClose={() => setCreateOpen(false)}
//         title="Új Action Item"
//         widthClass="max-w-lg">
//         <div className="grid gap-3">
//           <Field label="CaseId (UUID)">
//             <TextInput
//               value={caseId}
//               onChange={(e) => setCaseId(e.target.value)}
//               placeholder="..."
//             />
//           </Field>
//           <Field label="Cím">
//             <TextInput
//               value={title}
//               onChange={(e) => setTitle(e.target.value)}
//               placeholder="Feladat címe"
//             />
//           </Field>
//           <Button
//             onClick={() => void onCreate()}
//             disabled={creating || !caseId.trim() || !title.trim()}>
//             {creating ? "..." : "Létrehoz"}
//           </Button>
//         </div>
//       </Drawer>

//       {/* Edit drawer */}
//       <Drawer
//         open={open}
//         onClose={() => setOpen(false)}
//         title="Szerkesztés"
//         widthClass="max-w-2xl">
//         {current ? (
//           <div className="grid gap-3">
//             <Field label="Cím">
//               <TextInput
//                 value={eTitle}
//                 onChange={(e) => setETitle(e.target.value)}
//               />
//             </Field>
//             <Field label="Leírás">
//               <TextArea
//                 value={eDesc}
//                 onChange={(e) => setEDesc(e.target.value)}
//                 placeholder="..."
//               />
//             </Field>

//             <div className="grid gap-3 md:grid-cols-2">
//               <Field label="Státusz">
//                 <Select
//                   value={eStatus}
//                   onChange={(e) => setEStatus(e.target.value as any)}>
//                   <option value="open">open</option>
//                   <option value="in_progress">in_progress</option>
//                   <option value="blocked">blocked</option>
//                   <option value="done">done</option>
//                   <option value="cancelled">cancelled</option>
//                 </Select>
//               </Field>
//               <Field label="Határidő">
//                 <TextInput
//                   type="datetime-local"
//                   value={eDue}
//                   onChange={(e) => setEDue(e.target.value)}
//                 />
//               </Field>
//             </div>

//             <Divider />

//             <div>
//               <CardMeta>Hozzárendeltek</CardMeta>
//               {loadingAssignees ? (
//                 <Chip tone="neutral">assignees betöltése...</Chip>
//               ) : (
//                 <AssigneesPicker
//                   value={assignees as any}
//                   onChange={setAssignees as any}
//                 />
//               )}
//             </div>

//             <ToolbarRow>
//               <Button
//                 onClick={() => void saveEdit()}
//                 disabled={saving || !eTitle.trim()}>
//                 {saving ? "..." : "Mentés"}
//               </Button>
//               <SecondaryButton onClick={() => setOpen(false)}>
//                 Mégse
//               </SecondaryButton>
//             </ToolbarRow>
//           </div>
//         ) : null}
//       </Drawer>
//     </Page>
//   );
// }

// // "use client";

// // import { useEffect, useState } from "react";
// // import toast from "react-hot-toast";
// // import ApiCall from "@/api/api-calls";
// // import type { ActionItem, ActionItemStatus } from "@/api/routes/action-items";
// // import AssigneesPicker from "@/components/cases/AssigneesPicker";
// // import type { ActionItemAssignee } from "@/api/routes/action-items";
// // import { useRouter, useSearchParams } from "next/navigation";

// // import {
// //   Page,
// //   Card,
// //   ToolbarRow,
// //   TextInput,
// //   Select,
// //   Button,
// //   SecondaryButton,
// //   DangerButton,
// //   GridCards,
// //   CardTitle,
// //   CardMeta,
// //   Chip,
// //   SectionTitle,
// //   EmptyState,
// //   TextArea,
// // } from "@/components/ui/CardBits";

// // function fmt(iso?: string | null) {
// //   if (!iso) return "-";
// //   try {
// //     return new Date(iso).toLocaleString();
// //   } catch {
// //     return iso ?? "-";
// //   }
// // }
// // function isOverdue(due?: string | null) {
// //   if (!due) return false;
// //   return new Date(due).getTime() < Date.now();
// // }

// // export default function ActionItemsPage() {
// //   const router = useRouter();
// //   const searchParams = useSearchParams();

// //   const [items, setItems] = useState<ActionItem[]>([]);
// //   const [loading, setLoading] = useState(false);

// //   const [q, setQ] = useState("");
// //   const [mine, setMine] = useState<1 | 0>(1);
// //   const [status, setStatus] = useState<ActionItemStatus | "">("");

// //   // quick create
// //   const [caseId, setCaseId] = useState("");
// //   const [title, setTitle] = useState("");
// //   const [creating, setCreating] = useState(false);

// //   // edit state
// //   const [editId, setEditId] = useState<string | null>(null);
// //   const [eTitle, setETitle] = useState("");
// //   const [eDesc, setEDesc] = useState("");
// //   const [eStatus, setEStatus] = useState<ActionItemStatus>("open");
// //   const [eDue, setEDue] = useState("");
// //   const [saving, setSaving] = useState(false);

// //   const [assignees, setAssignees] = useState<Array<{ user: any; role: any }>>(
// //     []
// //   );
// //   const [loadingAssignees, setLoadingAssignees] = useState(false);

// //   async function load() {
// //     setLoading(true);
// //     try {
// //       const r = await ApiCall.actionItems.list({
// //         ...(mine ? { mine: 1 } : {}),
// //         q: q.trim() || undefined,
// //         status: status || undefined,
// //       });

// //       setItems(r.data.items ?? []);
// //     } catch (e: any) {
// //       toast.error(
// //         e?.response?.data?.message ?? "Action items betöltése sikertelen"
// //       );
// //     } finally {
// //       setLoading(false);
// //     }
// //   }

// //   useEffect(() => {
// //     const qp = (searchParams.get("q") ?? "").trim();
// //     if (qp) {
// //       setQ(qp);
// //       setMine(0); // ha közvetlenül rákeresel egy kódra, ne "csak az enyém" legyen
// //     }
// //     void load();
// //     // eslint-disable-next-line react-hooks/exhaustive-deps
// //   }, []);

// //   async function onCreate() {
// //     if (!caseId.trim() || !title.trim()) return;
// //     setCreating(true);
// //     try {
// //       await ApiCall.actionItems.create({
// //         caseId: caseId.trim(),
// //         title: title.trim(),
// //       });
// //       toast.success("Feladat létrehozva");
// //       setTitle("");
// //       await load();
// //     } catch (e: any) {
// //       toast.error(e?.response?.data?.message ?? "Létrehozás sikertelen");
// //     } finally {
// //       setCreating(false);
// //     }
// //   }

// //   async function startEdit(it: ActionItem) {
// //     setEditId(it.id);
// //     setETitle(it.title ?? "");
// //     setEDesc(it.description ?? "");
// //     setEStatus((it.status ?? "open") as any);
// //     setEDue(it.due_at ? new Date(it.due_at).toISOString().slice(0, 16) : "");

// //     setLoadingAssignees(true);
// //     try {
// //       const r = await ApiCall.actionItems.assignees.get(it.id);
// //       const a = (r.data.assignees ?? []) as ActionItemAssignee[];
// //       setAssignees(
// //         a.map((x) => ({
// //           user: {
// //             id: x.user_id,
// //             name: x.name,
// //             email: x.email,
// //             username: x.username,
// //           },
// //           role: x.role,
// //         }))
// //       );
// //     } catch {
// //       setAssignees([]);
// //     } finally {
// //       setLoadingAssignees(false);
// //     }
// //   }

// //   async function saveEdit() {
// //     if (!editId) return;
// //     setSaving(true);
// //     try {
// //       await ApiCall.actionItems.patch(editId, {
// //         title: eTitle.trim() || undefined,
// //         description: eDesc ?? undefined,
// //         status: eStatus,
// //         dueAt: eDue ? new Date(eDue).toISOString() : null,
// //       } as any);

// //       await ApiCall.actionItems.assignees.patch(editId, {
// //         assignees: assignees.map((x) => ({ userId: x.user.id, role: x.role })),
// //       });

// //       toast.success("Mentve");
// //       setEditId(null);
// //       await load();
// //     } catch (e: any) {
// //       toast.error(e?.response?.data?.message ?? "Mentés sikertelen");
// //     } finally {
// //       setSaving(false);
// //     }
// //   }

// //   async function archive(id: string) {
// //     if (!confirm("Archiválod ezt a feladatot?")) return;
// //     try {
// //       await ApiCall.actionItems.archive(id);
// //       toast.success("Archiválva");
// //       await load();
// //     } catch (e: any) {
// //       toast.error(e?.response?.data?.message ?? "Archiválás sikertelen");
// //     }
// //   }

// //   return (
// //     <Page
// //       title="Action Items"
// //       subtitle="Szerkeszthető + archiválható lista"
// //       actions={
// //         <>
// //           <SecondaryButton onClick={() => void load()} disabled={loading}>
// //             {loading ? "..." : "Frissít"}
// //           </SecondaryButton>
// //         </>
// //       }>
// //       <SectionTitle>Szűrők</SectionTitle>
// //       <Card>
// //         <ToolbarRow>
// //           <TextInput
// //             value={q}
// //             onChange={(e) => setQ(e.target.value)}
// //             placeholder="Keresés..."
// //             style={{ flex: 1, minWidth: 260 }}
// //           />
// //           <Select
// //             value={status}
// //             onChange={(e) => setStatus(e.target.value as any)}>
// //             <option value="">Összes státusz</option>
// //             <option value="open">open</option>
// //             <option value="in_progress">in_progress</option>
// //             <option value="blocked">blocked</option>
// //             <option value="done">done</option>
// //             <option value="cancelled">cancelled</option>
// //           </Select>
// //           <Select
// //             value={mine}
// //             onChange={(e) => setMine(Number(e.target.value) as any)}>
// //             <option value={1}>Csak az enyém</option>
// //             <option value={0}>Minden (jogosultság szerint)</option>
// //           </Select>
// //           <Button onClick={() => void load()} disabled={loading}>
// //             Keres
// //           </Button>
// //         </ToolbarRow>
// //       </Card>

// //       <SectionTitle>Gyors létrehozás</SectionTitle>
// //       <Card>
// //         <ToolbarRow>
// //           <TextInput
// //             value={caseId}
// //             onChange={(e) => setCaseId(e.target.value)}
// //             placeholder="caseId (UUID)"
// //           />
// //           <TextInput
// //             value={title}
// //             onChange={(e) => setTitle(e.target.value)}
// //             placeholder="Feladat címe"
// //             style={{ flex: 1, minWidth: 260 }}
// //           />
// //           <Button
// //             onClick={() => void onCreate()}
// //             disabled={creating || !caseId.trim() || !title.trim()}>
// //             {creating ? "..." : "Létrehoz"}
// //           </Button>
// //         </ToolbarRow>
// //       </Card>

// //       <SectionTitle>Lista</SectionTitle>
// //       {items.length ? (
// //         <GridCards>
// //           {items.map((it) => {
// //             const overdue = isOverdue(it.due_at);
// //             const tone =
// //               it.status === "done"
// //                 ? "good"
// //                 : it.status === "blocked"
// //                 ? "bad"
// //                 : overdue
// //                 ? "warn"
// //                 : "neutral";

// //             const editing = editId === it.id;

// //             return (
// //               <Card key={it.id}>
// //                 <CardTitle>{it.title}</CardTitle>
// //                 <CardMeta>
// //                   {it.case_title
// //                     ? `Ügy: ${it.case_title}`
// //                     : `Ügy: ${it.case_id?.slice(0, 8) ?? "—"}…`}
// //                 </CardMeta>

// //                 <div
// //                   style={{
// //                     display: "flex",
// //                     gap: 8,
// //                     flexWrap: "wrap",
// //                     marginTop: 10,
// //                   }}>
// //                   <Chip tone={tone as any}>{it.status}</Chip>
// //                   <Chip tone={overdue ? "warn" : "neutral"}>
// //                     due: {fmt(it.due_at)}
// //                   </Chip>
// //                   {it.archived_at ? <Chip tone="warn">archived</Chip> : null}
// //                 </div>

// //                 <div style={{ marginTop: 12 }}>
// //                   {!editing ? (
// //                     <ToolbarRow>
// //                       <SecondaryButton
// //                         onClick={() =>
// //                           it.case_id
// //                             ? router.push(`/admin/cases/${it.case_id}`)
// //                             : null
// //                         }>
// //                         Ügy megnyitása
// //                       </SecondaryButton>
// //                       <SecondaryButton
// //                         onClick={() => startEdit(it)}
// //                         disabled={!!it.archived_at}>
// //                         Szerkeszt
// //                       </SecondaryButton>
// //                       <DangerButton
// //                         onClick={() => void archive(it.id)}
// //                         disabled={!!it.archived_at}>
// //                         Archivál
// //                       </DangerButton>
// //                     </ToolbarRow>
// //                   ) : (
// //                     <div style={{ display: "grid", gap: 8 }}>
// //                       <TextInput
// //                         value={eTitle}
// //                         onChange={(e) => setETitle(e.target.value)}
// //                       />
// //                       <TextArea
// //                         value={eDesc}
// //                         onChange={(e) => setEDesc(e.target.value)}
// //                         placeholder="Leírás..."
// //                       />

// //                       <div
// //                         style={{
// //                           display: "grid",
// //                           gap: 8,
// //                           gridTemplateColumns: "1fr 1fr",
// //                         }}>
// //                         <Select
// //                           value={eStatus}
// //                           onChange={(e) => setEStatus(e.target.value as any)}>
// //                           <option value="open">open</option>
// //                           <option value="in_progress">in_progress</option>
// //                           <option value="blocked">blocked</option>
// //                           <option value="done">done</option>
// //                           <option value="cancelled">cancelled</option>
// //                         </Select>

// //                         <TextInput
// //                           type="datetime-local"
// //                           value={eDue}
// //                           onChange={(e) => setEDue(e.target.value)}
// //                         />
// //                       </div>

// //                       {/* ✅ IDE jön */}
// //                       <div style={{ marginTop: 6 }}>
// //                         <CardMeta>Hozzárendeltek</CardMeta>
// //                         {loadingAssignees ? (
// //                           <Chip tone="neutral">assignees betöltése...</Chip>
// //                         ) : (
// //                           <AssigneesPicker
// //                             value={assignees as any}
// //                             onChange={setAssignees as any}
// //                           />
// //                         )}
// //                       </div>

// //                       <ToolbarRow>
// //                         <Button
// //                           onClick={() => void saveEdit()}
// //                           disabled={saving || !eTitle.trim()}>
// //                           {saving ? "..." : "Mentés"}
// //                         </Button>
// //                         <SecondaryButton onClick={() => setEditId(null)}>
// //                           Mégse
// //                         </SecondaryButton>
// //                       </ToolbarRow>
// //                     </div>
// //                   )}
// //                 </div>
// //               </Card>
// //             );
// //           })}
// //         </GridCards>
// //       ) : (
// //         <EmptyState
// //           title="Nincs feladat"
// //           hint="Próbálj szűrőket módosítani, vagy hozz létre egy újat."
// //         />
// //       )}
// //     </Page>
// //   );
// // }
