"use client";

import { useEffect, useState, useCallback } from "react";
import { useRouter } from "next/navigation";
import Select from "react-select";
import toast from "react-hot-toast";

import ApiCall from "@/api/api-calls";
import { useForm } from "@/hooks/custom-form-hook";
import Input from "@/components/shared/input/Input";
import UniversalIconPicker from "@/components/shared/iconPicker/UniversalIconPicker";
import {
  VALIDATOR_MINLENGTH,
  VALIDATOR_MAXLENGTH,
  VALIDATOR_REQUIRE,
} from "@/utils/validators";
import { useAdminMenu } from "@/hooks/useAdminMenu";
import { showTypedToast } from "@/lib/typedToast";

type Option = { value: string; label: string };

const AdminMenuCategoryCreate = () => {
  const router = useRouter();

  const [sectionOptions, setSectionOptions] = useState<Option[]>([]);
  const [selectedSection, setSelectedSection] = useState<Option | null>(null);

  const [menuOptions, setMenuOptions] = useState<Option[]>([]);
  const [selectedMenus, setSelectedMenus] = useState<Option[]>([]);

  const [selectedIcon, setSelectedIcon] = useState<string>("FaFolder");

  // Category-only form
  const [formState, handleInput] = useForm(
    {
      categoryTitle: { value: "", isValid: false },
      categorySlug: { value: "", isValid: true },
    },
    false
  );
  const { refreshMenu } = useAdminMenu();

  // Load create metadata (sections + menus)
  useEffect(() => {
    (async () => {
      const res = await ApiCall.adminMenuCategory.getCreate();
      if (res?.status !== 200) {
        showTypedToast({
          type: "error",
          message: "Nem sikerült betölteni a létrehozás űrlapot",
        });
        return;
      }
      const body = res.data?.body || {};

      setSectionOptions(
        (body.sectionOptions || []).map((s: any) => ({
          value: s.id,
          label: s.title ?? s.slug ?? s.id,
        }))
      );

      setMenuOptions(
        (body.menuOptions || []).map((m: any) => ({
          value: m.id,
          label: m.title ?? m.slug ?? m.id,
        }))
      );
    })();
  }, []);

  const slugify = (value: string) =>
    value
      .toLowerCase()
      .normalize("NFD")
      .replace(/\p{Diacritic}/gu, "")
      .replace(/[^a-z0-9]+/g, "-")
      .replace(/(^-|-$)/g, "");

  // // Auto-slugify when title changes (if slug empty)
  // const titleValue = formState.inputs.categoryTitle?.value ?? "";
  // const slugValue = formState.inputs.categorySlug?.value ?? "";
  // useEffect(() => {
  //   if (!titleValue || slugValue) return;
  //   const s = titleValue
  //     .toLowerCase()
  //     .normalize("NFD")
  //     .replace(/\p{Diacritic}/gu, "")
  //     .replace(/[^a-z0-9]+/g, "-")
  //     .replace(/(^-|-$)/g, "");
  //   // @ts-ignore: custom hook signature (id, value, isValid)
  //   handleInput("categorySlug", s, true);
  // }, [titleValue, slugValue, handleInput]);

  // const handleSubmit = useCallback(async () => {
  //   const payload = {
  //     categoryTitle: formState.inputs.categoryTitle, // { value, isValid }
  //     categorySlug: formState.inputs.categorySlug, // { value, isValid }
  //     icon: selectedIcon,
  //     selectedSections: selectedSection
  //       ? [{ id: selectedSection.value, sorting: 0 }]
  //       : [],
  //     selectedMenus: selectedMenus.map((o, i) => ({
  //       id: o.value,
  //       sorting: i,
  //     })),
  //   };

  //   const out = await ApiCall.adminMenuCategory.postCreate(payload);
  //   if (out?.status === 200) {
  //     toast.success(out.data?.message || "Sikeres mentés");
  //     await refreshMenu();

  //     router.push("/admin/admin-menu-category/list");
  //   } else {
  //     toast.error(out?.data?.error || "Sikertelen mentés");
  //   }
  // }, [formState, selectedIcon, selectedSection, selectedMenus, router]);

  const handleSubmit = useCallback(async () => {
    const titleInput = formState.inputs.categoryTitle;
    const slugInput = formState.inputs.categorySlug;

    const rawTitle = titleInput?.value ?? "";
    let finalSlug = slugInput?.value ?? "";

    // If user didn't type a slug but title exists, generate it here
    if (!finalSlug && rawTitle) {
      finalSlug = slugify(rawTitle);
    }

    const payload = {
      categoryTitle: titleInput, // already { value, isValid }
      categorySlug: {
        ...slugInput,
        value: finalSlug,
        isValid: true, // you can keep previous if you want
      },
      icon: selectedIcon,
      selectedSections: selectedSection
        ? [{ id: selectedSection.value, sorting: 0 }]
        : [],
      selectedMenus: selectedMenus.map((o, i) => ({
        id: o.value,
        sorting: i,
      })),
    };

    const out = await ApiCall.adminMenuCategory.postCreate(payload);
    if (out?.status === 200) {
      // toast.success(out.data?.message || "Sikeres mentés");
      showTypedToast({
        type: "success",
        message: out.data?.message || "Sikeres mentés",
      });
      await refreshMenu();
      router.push("/admin/admin-menu-category/list");
    } else {
      showTypedToast({
        type: "error",
        message: out.data?.error || "Sikertelen mentés",
      });
    }
  }, [formState, selectedIcon, selectedSection, selectedMenus, router]);

  return (
    <div className="max-w-2xl mx-auto space-y-3">
      <h1 className="text-xl font-semibold">Új menü kategória</h1>

      <Input
        element="input"
        id="categoryTitle"
        name="categoryTitle"
        type="text"
        placeholder="Kategória neve"
        inputcls="w-full border p-2"
        validators={[
          VALIDATOR_REQUIRE(),
          VALIDATOR_MINLENGTH(2),
          VALIDATOR_MAXLENGTH(254),
        ]}
        errorText="Nem lehet üres"
        initialValid={false}
        onInput={handleInput}
        value={formState.inputs.categoryTitle?.value ?? ""}
      />

      <Input
        element="input"
        id="categorySlug"
        name="categorySlug"
        type="text"
        placeholder="Slug (opcionális)"
        inputcls="w-full border p-2"
        validators={[VALIDATOR_MAXLENGTH(254)]}
        errorText="Túl hosszú slug"
        initialValid={true}
        onInput={handleInput}
        value={formState.inputs.categorySlug?.value ?? ""}
      />

      <div>
        <label className="block text-sm mb-1">Szekció</label>
        <Select
          options={sectionOptions}
          value={selectedSection}
          onChange={(opt) => setSelectedSection(opt as Option)}
          className="react-select-container"
          classNamePrefix="react-select"
          placeholder="Válassz szekciót"
          isClearable
        />
      </div>

      <div>
        <label className="block text-sm mb-1">Menük ebben a kategóriában</label>
        <Select
          options={menuOptions}
          value={selectedMenus}
          onChange={(opts) => setSelectedMenus((opts as Option[]) || [])}
          isMulti
          className="react-select-container"
          classNamePrefix="react-select"
          placeholder="Válassz menü(ke)t"
        />
      </div>

      <div className="pt-2">
        <label className="block text-sm mb-2">Ikon</label>
        <UniversalIconPicker value={selectedIcon} onSelect={setSelectedIcon} />
      </div>

      <button
        className={`w-full mt-2 px-4 py-2 rounded text-white ${
          formState.isValid ? "bg-green-600 hover:bg-green-700" : "bg-gray-300"
        }`}
        disabled={!formState.isValid}
        onClick={handleSubmit}>
        Mentés
      </button>
    </div>
  );
};

export default AdminMenuCategoryCreate;
