"use client";

import { useEffect, useMemo, useState, useCallback } from "react";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import Loader from "@/components/shared/loader/global-loading";
import { TextField } from "@mui/material";
import Autocomplete from "@mui/material/Autocomplete";
import Link from "next/link";
import toast from "react-hot-toast";
import useTableDebounce from "@/hooks/useTableDebounce";
import Button from "@/components/shared/button/Button";
import { Pencil, Trash2 } from "lucide-react";
import { useAdminMenu } from "@/hooks/useAdminMenu";

const AdminMenuCategoryList = () => {
  const token = ""; // Optional: auth token
  const [data, setData] = useState([]);
  const [page, setPage] = useState(1);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [pageSize, setPageSize] = useState(25);
  const [sortField, setSortField] = useState("updated_at");
  const [sortDirection, setSortDirection] = useState("desc");
  const [totalRecords, setTotalRecords] = useState(0);
  const [inputValue, setInputValue] = useState("");
  const [selectedRows, setSelectedRows] = useState([]);
  const [toggleCleared, setToggleCleared] = useState(false);
  const debouncedSearchTerm = useTableDebounce(inputValue, 500); // ✅ Use custom hook
  const { refreshMenu } = useAdminMenu();

  const query = `?page=${page}&pageSize=${pageSize}&sortField=${sortField}&sortDirection=${sortDirection}&searchTerm=${debouncedSearchTerm.trim()}`;

  const fetchData = async () => {
    setLoading(true);
    try {
      const response = await ApiCall.adminMenu.getAll(query);
      setData(response?.data?.tableData ?? []);
      setTotalRecords(response?.data?.totalRecords ?? 0);
    } catch (err) {
      console.error(err);
    }
    setLoading(false);
  };

  useEffect(() => {
    fetchData();
  }, [query]);

  const handleSelectAction = async (actionType, rowsOrRow) => {
    const selected = Array.isArray(rowsOrRow) ? rowsOrRow : [rowsOrRow];
    const ids = selected.map((r) => r._id);

    if (actionType === "delete") {
      const confirmDelete = confirm(
        `Biztosan törölni szeretnéd a következő elemeket?\n${ids.join(", ")}`
      );
      if (confirmDelete) {
        try {
          const response = await ApiCall.adminMenu.deleteAdminMenu(ids);
          if (response.status === 200) {
            toast.success(response?.data?.message);
            await refreshMenu();
          } else {
            toast.error(response?.data?.error);
          }
          fetchData();
          setToggleCleared(!toggleCleared);
        } catch (err) {
          console.error("Delete failed:", err);
        }
      }
    }

    if (actionType === "edit") {
      if (selected.length !== 1) {
        alert("Csak egy elem szerkeszthető egyszerre.");
        return;
      }
      const id = selected[0]._id;
      window.location.href = `/admin/admin-menu/edit/${id}`;
    }
  };

  const handleSortingChange = async (id, newValue) => {
    const newSorting = parseInt(newValue, 10);
    if (isNaN(newSorting)) return;

    try {
      await ApiCall.adminMenu.updateSorting(id, newSorting);
      toast.success("Sorrend frissítve");
      fetchData();
    } catch (err) {
      console.error("Sorting update failed:", err);
      toast.error("Hiba a sorrend frissítésekor");
    }
  };

  const tableSelectOptions = [
    {
      text: "Szerkesztés",
      type: "edit",
      className: "bg-blue-500 text-white",
    },
    {
      text: "Törlés",
      type: "delete",
      className: "bg-red-500 text-white",
    },
  ];

  const columns = useMemo(
    () => [
      { name: "Menü ID", selector: (row) => row._id, sortable: true },
      {
        name: "Admin menü neve",
        selector: (row) => row.adminMenuTitle,
        sortable: true,
      },
      {
        id: "adminMenuSlug",
        name: "Admin menü url",
        selector: (row) => row.adminMenuSlug,
        sortable: true,
      },
      {
        id: "adminMenuLink",
        name: "Admin menü link",
        selector: (row) => row.adminMenuLink,
        sortable: true,
      },
      {
        name: "Sorrend", // Sorting
        selector: (row) => row.sorting,
        sortable: true,
        cell: (row) => (
          <input
            type="number"
            className="w-16 rounded border px-2 py-1 text-sm"
            value={row.sorting}
            onChange={(e) => handleSortingChange(row._id, e.target.value)}
          />
        ),
      },
      { name: "Frissítve", selector: (row) => row.updatedAt, sortable: true },
      {
        name: "Műveletek",
        cell: (row) => (
          <div className="flex space-x-2">
            <Button
              href={`/admin/admin-menu/edit/${row._id}`}
              type="button"
              variant="info"
              icononly
              rounded="full"
              size="sm"
              aria-label="Edit"
              leftIcon={<Pencil />}
            />
            <Button
              type="button"
              variant="danger"
              icononly
              rounded="full"
              size="sm"
              aria-label="Delete" // ✅ important for screen readers
              onClick={() => handleSelectAction("delete", row)}
              leftIcon={<Trash2 className="w-6 h-6" aria-hidden="true" />}
            />
            {/* <Link
              href={`/admin/admin-menu/edit/${row._id}`}
              className="rounded bg-blue-500/75 px-3 py-1 text-sm text-white hover:bg-blue-600">
              Szerkesztés
            </Link>
            <button
              onClick={() => handleSelectAction("delete", row)}
              className="rounded bg-red-500/75 px-3 py-1 text-sm text-white hover:bg-red-600">
              Törlés
            </button> */}
          </div>
        ),
      },
    ],
    []
  );

  const subHeaderComponent = (
    <div className="flex flex-wrap w-full justify-between items-center gap-4">
      <Autocomplete
        freeSolo
        id="subHeaderSearch"
        className="w-72"
        disableClearable
        options={[]}
        inputValue={inputValue}
        onInputChange={(event, newInputValue) => {
          setInputValue(newInputValue);
        }}
        renderInput={(params) => (
          <TextField
            {...params}
            label="Keresés"
            type="search"
            variant="outlined"
          />
        )}
      />
      {selectedRows.length > 0 && (
        <div className="flex items-center gap-2 bg-gray-100 px-4 py-2 border rounded">
          <span className="text-sm text-gray-600">
            {selectedRows.length} kiválasztva
          </span>
          {tableSelectOptions.map((option) => (
            <button
              key={option.text}
              onClick={() => {
                handleSelectAction(option.type, selectedRows);
              }}
              className={`rounded px-3 py-1 text-sm ${option.className}`}>
              {option.text}
            </button>
          ))}
        </div>
      )}
    </div>
  );

  return (
    <DataTableWrapper
      tableTitle="Admin menü lista"
      data={data}
      columns={columns}
      progressPending={loading}
      paginationDefaultPage={page}
      paginationServer
      onChangePage={(pageNumber) => setPage(pageNumber)}
      progressComponent={<Loader />}
      paginationTotalRows={totalRecords}
      paginationPerPage={pageSize}
      onSort={(col, dir) => {
        setSortField(col.id);
        setSortDirection(dir);
        setPage(1);
      }}
      onChangeRowsPerPage={(size) => {
        setPageSize(size);
        setPage(1);
      }}
      onFilterClear={fetchData}
      subHeader
      subHeaderComponent={subHeaderComponent}
      subHeaderAlign="right"
      subHeaderWrap
      tableOptionsPath="admin-menu/create"
      selectableRows={true}
      onRowSelected={setSelectedRows}
      toggleCleared={toggleCleared}
    />
  );
};

export default AdminMenuCategoryList;
