"use client";

import React, { useEffect, useState } from "react";
import ApiCall from "@/api/api-calls";
import ExpandedRowTabs from "@/components/inventory/ExpandedRowTabs";
import InventoryElementCreate from "@/components/inventory/InventoryElementCreate";
import InventoryRegDocument from "@/components/inventory/InventoryRegDocument";
import InventoryInvoice from "@/components/inventory/InventoryInvoice";
import InventoryInsurance from "@/components/inventory/InventoryInsurance";
import InventoryReport from "@/components/inventory/InventoryReport";

const CreateInventoryPage = () => {
  const [references, setReferences] = useState(null);
  const [assetId, setAssetId] = useState(null);
  const [activeTab, setActiveTab] = useState(1);

  useEffect(() => {
    const fetchReferences = async () => {
      try {
        const [
          sites,
          departments,
          buyers,
          assetTypes,
          locations,
          brands,
          currencies,
        ] = await Promise.all([
          ApiCall.inventory.getSites(),
          ApiCall.inventory.getDepartments(),
          ApiCall.inventory.getBuyers(),
          ApiCall.inventory.getAssetTypes(),
          ApiCall.inventory.getLocations(),
          ApiCall.inventory.getBrands(),
          ApiCall.currency.getAll(),
        ]);

        setReferences({
          sites: sites.data,
          departments: departments.data,
          ownCompanies: buyers.data,
          assetTypes: assetTypes.data,
          locations: locations.data,
          brands: brands.data,
          currencies: currencies.data.currencies,
          document_types: [], // load if needed
        });
      } catch (err) {
        console.error("Error loading references", err);
      }
    };

    fetchReferences();
  }, []);

  const getDocumentTypeId = (name) => {
    return references?.document_types.find((dt) => dt.name === name)?.id;
  };

  if (!references) return <div className="p-4">Loading references...</div>;

  return (
    <div className="w-full p-6">
      <ExpandedRowTabs activeTab={activeTab} onTabChange={setActiveTab} />

      {activeTab === 1 && references && (
        <InventoryElementCreate
          references={references}
          onCreated={(newAssetId) => setAssetId(newAssetId)}
        />
      )}

      {activeTab > 1 && !assetId && (
        <div className="text-red-600">Először mentse el az eszközt!</div>
      )}

      {activeTab === 2 && assetId && (
        <InventoryRegDocument
          assetId={assetId}
          references={references}
          onRowUpdated={() => {}}
          documentTypeId={getDocumentTypeId("törzslap")}
        />
      )}
      {activeTab === 3 && assetId && (
        <InventoryInvoice
          assetId={assetId}
          references={references}
          onRowUpdated={() => {}}
          documentTypeId={getDocumentTypeId("invoice")}
        />
      )}
      {activeTab === 4 && assetId && (
        <InventoryInsurance
          assetId={assetId}
          references={references}
          onRowUpdated={() => {}}
          documentTypeId={getDocumentTypeId("biztosítási-dokumentumok")}
        />
      )}
      {activeTab === 5 && assetId && (
        <InventoryReport
          assetId={assetId}
          references={references}
          onRowUpdated={() => {}}
          documentTypeId={getDocumentTypeId("jegyzőkönyv")}
        />
      )}
    </div>
  );
};

export default CreateInventoryPage;
