// AssetsList.jsx (replace your file contents or merge carefully)
"use client";

import { useCallback, useEffect, useMemo, useState, useRef } from "react";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import Loader from "@/components/shared/loader/global-loading";
import { TextField } from "@mui/material";
import Autocomplete from "@mui/material/Autocomplete";
import toast from "react-hot-toast";
import { Pencil, Trash2 } from "lucide-react";
import useTableDebounce from "@/hooks/useTableDebounce";
import ExpandedRow from "@/components/inventory/ExpandedRow";
import { convertGetDateString } from "@/utils/converDateString";
import { currencyDisplayHandler } from "@/utils/currencyDisplay";
import Select from "react-select";
import { FileDown, Plus } from "lucide-react";
import SlideOver from "@/components/shared/SlideOver";
import InventoryElement from "@/components/inventory/InventoryElement";
import InventoryElementCreate from "@/components/inventory/InventoryElementCreate";

const AssetsList = () => {
  const [data, setData] = useState([]);
  const [page, setPage] = useState(1);
  const [loading, setLoading] = useState(true);
  const [pageSize, setPageSize] = useState(25);
  const [sortField, setSortField] = useState("name");
  const [sortDirection, setSortDirection] = useState("asc");
  const [totalRecords, setTotalRecords] = useState(0);
  const [inputValue, setInputValue] = useState("");
  const [selectedRows, setSelectedRows] = useState([]);
  const [toggleCleared, setToggleCleared] = useState(false);
  const [expandedRow, setExpandedRow] = useState(null);
  const expandedRowRef = useRef(null);

  // filters + options
  const [filters, setFilters] = useState({});
  const debouncedSearchTerm = useTableDebounce(inputValue, 500);

  const [filterOptions, setFilterOptions] = useState({
    sites: [],
    departments: [],
    buyers: [],
    assetTypes: [],
    locations: [],
    brands: [],
  });
  const leasedOptions = [
    { value: "true", label: "Lízingelve" },
    { value: "false", label: "Saját tulajdon" },
  ];
  // 🆕 status → color mapping
  const STATUS_BG = [
    { name: "új eszköz", rgb: "255,255,0" }, // yellow
    { name: "selejt", rgb: "255,0,0" }, // red
    { name: "hibás rögzítés", rgb: "0,0,255" }, // blue
    { name: "költözés miatt áthelyezve", rgb: "255,192,203" }, // pink(ish)
    { name: "ideiglenesen áthelyezve", rgb: "0,255,0" }, // green
    { name: "eladva", rgb: "128,0,128" }, // purple
  ];
  const conditionalRowStyles = STATUS_BG.map(({ name, rgb }) => ({
    when: (row) =>
      row.item_status_name_hu === name || row.item_status_name === name,
    style: {
      backgroundColor: `rgba(${rgb}, 0.18)`,
    },
  }));
  // slide-over state
  const [formOpen, setFormOpen] = useState(false);
  const [formMode, setFormMode] = useState("create"); // 'create' | 'edit'
  const [editingId, setEditingId] = useState(null);
  const [formRefs, setFormRefs] = useState(null); // references for InventoryElement
  const [formAsset, setFormAsset] = useState(null); // asset for edit
  console.log("formMode", formMode);
  const query = useMemo(() => {
    const params = new URLSearchParams();
    params.append("page", page);
    params.append("pageSize", pageSize);
    params.append("sortField", sortField);
    params.append("sortDirection", sortDirection);
    params.append("searchTerm", debouncedSearchTerm.trim());
    Object.entries(filters).forEach(([key, value]) => {
      if (value) params.append(key, value);
    });
    return `?${params.toString()}`;
  }, [page, pageSize, sortField, sortDirection, debouncedSearchTerm, filters]);

  const downloadQuery = useMemo(() => {
    const params = new URLSearchParams();
    params.append("sortField", sortField);
    params.append("sortDirection", sortDirection);
    params.append("searchTerm", debouncedSearchTerm.trim());
    Object.entries(filters).forEach(([key, value]) => {
      if (value) params.append(key, value);
    });
    return `?${params.toString()}`;
  }, [sortField, sortDirection, debouncedSearchTerm, filters]);
  // Normalize whatever DataTableWrapper gives us
  const selectedCount = useMemo(() => {
    if (Array.isArray(selectedRows)) {
      return selectedRows.length;
    }
    // react-data-table style object: { selectedCount, selectedRows }
    if (selectedRows && typeof selectedRows === "object") {
      return selectedRows.selectedCount || 0;
    }
    return 0;
  }, [selectedRows]);

  const selectedRowsArray = useMemo(() => {
    if (Array.isArray(selectedRows)) {
      return selectedRows;
    }
    if (selectedRows && typeof selectedRows === "object") {
      return selectedRows.selectedRows || [];
    }
    return [];
  }, [selectedRows]);
  // initial filters
  useEffect(() => {
    const fetchFilters = async () => {
      try {
        const [sites, departments, buyers, assetTypes, locations, brands] =
          await Promise.all([
            ApiCall.inventory.getSites(),
            ApiCall.inventory.getDepartments(), // unfiltered first load
            ApiCall.inventory.getBuyers(),
            ApiCall.inventory.getAssetTypes(),
            ApiCall.inventory.getLocations(), // unfiltered first load
            ApiCall.inventory.getBrands(),
          ]);
        setFilterOptions({
          sites: sites.data,
          departments: departments.data,
          buyers: buyers.data,
          assetTypes: assetTypes.data,
          locations: locations.data,
          brands: brands.data,
        });
      } catch {
        toast.error("Hiba a szűrők betöltésekor");
      }
    };
    fetchFilters();
  }, []);

  // cascading re-fetch for departments/locations based on selected site/department
  useEffect(() => {
    const refreshCascading = async () => {
      try {
        // departments by site
        if (filters.siteId) {
          const deps = await ApiCall.inventory.getDepartments(
            `?siteId=${filters.siteId}`
          );
          setFilterOptions((prev) => ({ ...prev, departments: deps.data }));
          // clear invalid department
          if (
            filters.departmentId &&
            !deps.data.some((d) => d.id === filters.departmentId)
          ) {
            setFilters((p) => ({ ...p, departmentId: null }));
          }
        }

        // locations by department OR site
        if (filters.departmentId) {
          const locs = await ApiCall.inventory.getLocations(
            `?departmentId=${filters.departmentId}`
          );
          setFilterOptions((prev) => ({ ...prev, locations: locs.data }));
        } else if (filters.siteId) {
          const locs = await ApiCall.inventory.getLocations(
            `?siteId=${filters.siteId}`
          );
          setFilterOptions((prev) => ({ ...prev, locations: locs.data }));
        } else {
          const locs = await ApiCall.inventory.getLocations();
          setFilterOptions((prev) => ({ ...prev, locations: locs.data }));
        }

        // clear cross-site/location
        setFilters((prev) => ({
          ...prev,
          placedAtId: prev.siteId ? prev.placedAtId : null,
          usedAtId: prev.siteId ? prev.usedAtId : null,
        }));
      } catch {
        // swallow to avoid noisy toasts
      }
    };
    refreshCascading();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [filters.siteId, filters.departmentId]);

  const locationOptionsWithSite = useMemo(() => {
    const all = filterOptions.locations || [];
    const selectedSiteId = filters.siteId || null;
    return all
      .filter((l) => !selectedSiteId || l.site_id === selectedSiteId)
      .map((l) => ({
        value: l.id,
        label: selectedSiteId ? l.name : `${l.name} — ${l.site_name}`,
      }));
  }, [filterOptions.locations, filters.siteId]);

  const fetchData = async () => {
    setLoading(true);
    try {
      const response = await ApiCall.inventory.getList(query);
      setData(response?.data?.tableData ?? []);
      setTotalRecords(response?.data?.totalRecords ?? 0);
    } catch {
      toast.error("Hiba az adatok betöltésekor");
    }
    setLoading(false);
  };

  useEffect(() => {
    fetchData();
  }, [query]);

  const handleClearFilters = () => {
    setFilters({});
    setInputValue("");
    setPage(1);
  };

  // toolbar select helper
  const filterSelect = (label, field, options) => {
    const normalizedOptions = (options || []).map((o) =>
      "value" in o && "label" in o
        ? o
        : { value: o.id ?? o.name, label: o.name }
    );
    const current =
      normalizedOptions.find((opt) => opt.value === filters[field]) || null;

    return (
      <Select
        className="w-auto min-w-72 flex-1 text-left"
        placeholder={label}
        options={normalizedOptions}
        value={current}
        onChange={(selected) => {
          setFilters((prev) => ({
            ...prev,
            [field]: selected ? selected.value : null,
          }));
          setPage(1);
        }}
        styles={{
          control: (base) => ({
            ...base,
            height: "100%",
            minHeight: "100%",
            padding: "10px 0",
          }),
          valueContainer: (base) => ({ ...base, height: "100%" }),
        }}
        isClearable
      />
    );
  };
  const handleSave = () => {
    setFormOpen(false);
    fetchData();
  };
  const handleInventoryDownload = useCallback(() => {
    (async () => {
      try {
        const res = await ApiCall.inventory.download(downloadQuery);
        const url = window.URL.createObjectURL(new Blob([res.data]));
        const link = document.createElement("a");
        link.href = url;
        link.setAttribute("download", "leltar.xlsx");
        document.body.appendChild(link);
        link.click();
        link.remove();
      } catch (error) {
        console.error("Download failed", error);
        toast.error("A letöltés sikertelen");
      }
    })();
  }, [downloadQuery]);

  // open Create
  const openCreate = async () => {
    try {
      setFormMode("create");
      setEditingId(null);
      // load references fresh for the form
      const references = await ApiCall.inventory.getReferences();
      const data = references.data.references;
      console.log("references", references);
      setFormRefs({
        sites: data.sites || [],
        departments: data.departments || [],
        ownCompanies: data.ownCompanies || [],
        assetTypes: data.assetTypes || [],
        locations: data.locations || [],
        currencies: (data.currencies || []).map((c) => ({
          value: c.value ?? c.id,
          label: c.label ?? c.code ?? c.name,
        })),
        partnerCompanies: data.partnerCompanies || [],
        users: data.people || [],
        deviceBrands: data.deviceBrands || [],
        deviceModels: data.deviceModels || [],
        document_types: data.documentTypes || [],
      });

      setFormAsset({
        name: "",
        inventory_number: "",
        invoice_number: "",
        revision_frequency: "",
        serial_number: "",
        external_id: "",
        description: "",
        extended_functions: "",
        moved_at_date: null,
        purchase_date: null,
        revise_date: null,
        price: 0,
        netto_price: 0,
        asset_type_id: null,
        buyer_id: null,
        department_id: null,
        placed_at: null,
        site_id: null,
        used_at: null,
        device_brand: "",
        device_model: "",
        price_currency_id: null,
        netto_price_currency_id: null,
        device_reg_page: false,
        device_insurance: false,
        isLeased: false,
        device_isLeased: null, // document id or null
        service_phone: "",
        contact_person: "",
        contact_phone: "",
        install_steps: "",
        install_requirements: "",
      });

      setFormOpen(true);
    } catch {
      toast.error("Űrlap előkészítése sikertelen");
    }
  };

  // open Edit
  const openEdit = async (id) => {
    try {
      setFormMode("edit");
      setEditingId(id);

      const row = await ApiCall.inventory.getOne(id);
      const a = row.data.asset;
      // const refs = row.data.references;
      const refResponse = await ApiCall.inventory.getReferences();
      const refs = refResponse.data.refResponse;

      setFormRefs(refs);
      setFormAsset(a);
      setFormOpen(true);
    } catch {
      toast.error("Betöltés sikertelen");
    }
  };

  const subHeaderComponent = (
    <div className="flex flex-wrap w-full justify-between items-center gap-4">
      <div className="w-full flex flex-wrap gap-2 items-stretch">
        <Autocomplete
          freeSolo
          id="subHeaderSearch"
          disableClearable
          options={[]}
          inputValue={inputValue}
          onInputChange={(_, v) => setInputValue(v)}
          renderInput={(params) => (
            <TextField
              {...params}
              label="Keresés"
              type="search"
              variant="outlined"
              className="!w-auto !min-w-72 h-full"
              sx={{ height: "100%" }}
              InputProps={{ ...params.InputProps, sx: { height: "100%" } }}
              InputLabelProps={{ sx: { height: "100%" } }}
            />
          )}
        />

        {filterSelect("Telephely", "siteId", filterOptions.sites)}
        {filterSelect(
          "Szervezeti egység",
          "departmentId",
          filterOptions.departments
        )}
        {filterSelect("Beszerző", "buyerId", filterOptions.buyers)}
        {filterSelect(
          "Eszköz csoport",
          "assetTypeId",
          filterOptions.assetTypes
        )}
        {filterSelect(
          "Feltalálási hely",
          "placedAtId",
          locationOptionsWithSite
        )}
        {filterSelect(
          "Felhasználási hely",
          "usedAtId",
          locationOptionsWithSite
        )}
        {filterSelect("Gyártó", "deviceBrand", filterOptions.brands)}
        {filterSelect("Lízingelve?", "isLeased", leasedOptions)}
        <div className="w-full flex flex-wrap justify-end gap-2">
          <button
            className="flex items-center gap-2 px-4 py-2 text-gray-600 hover:bg-yellow-600 hover:text-white rounded cursor-pointer"
            onClick={handleClearFilters}>
            🧹 Szűrők törlése
          </button>
          <button
            className="flex items-center gap-2 px-4 py-2 text-gray-600 hover:bg-green-600 hover:text-white rounded cursor-pointer"
            onClick={handleInventoryDownload}>
            <FileDown className="w-4 h-4" />
            XLS export
          </button>
          <button
            onClick={openCreate}
            className="flex items-center gap-2 px-4 py-2 text-gray-600 hover:bg-blue-600 hover:text-white rounded">
            <Plus className="w-4 h-4" />
            Új hozzáadása
          </button>
          {selectedCount > 0 && (
            <button
              className="flex items-center gap-2 px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700"
              onClick={() => handleSelectAction("delete", selectedRowsArray)}>
              <Trash2 className="w-4 h-4" />
              Kijelölt törlése ({selectedCount})
            </button>
          )}
        </div>
      </div>
    </div>
  );

  const handleSelectAction = async (actionType, rowsOrRow) => {
    const selected = Array.isArray(rowsOrRow) ? rowsOrRow : [rowsOrRow];
    const ids = selected.map((r) => r.id);

    if (actionType === "delete") {
      const confirmDelete = confirm(`Confirm delete:\n${ids.join(", ")}`);
      if (confirmDelete) {
        try {
          await ApiCall.inventory.delete(ids);
          fetchData();
          setToggleCleared(!toggleCleared);
          setSelectedRows([]);
          toast.success("Deleted successfully");
        } catch (err) {
          console.error("Delete failed:", err);
          toast.error("Delete failed");
        }
      }
    }

    if (actionType === "edit") {
      if (selected.length !== 1) {
        alert("Only one item can be edited at a time.");
        return;
      }
      openEdit(selected[0].id);
    }
  };

  const customStyles = {
    rows: {
      style: { fontSize: "11px !important", minHeight: "48px !important" },
    },
    headCells: {
      style: {
        fontSize: "12px !important",
        fontWeight: "bold !important",
        backgroundColor: "#f9f9f9 !important",
      },
    },
    cells: { style: { fontSize: "0.73rem !important" } },
  };

  const columns = useMemo(
    () => [
      {
        id: "site_name",
        name: "Telephely",
        selector: (row) => row.site_name,
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.site_name}>
            {row.site_name}
          </div>
        ),
        sortable: true,
      },
      {
        id: "department_name",
        name: "Szervezeti egység",
        selector: (row) => row.department_name,
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.department_name}>
            {row.department_name}
          </div>
        ),
        sortable: true,
      },
      {
        id: "name",
        name: "Eszköz név",
        selector: (row) => row.name,
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.name}>
            {row.name}
          </div>
        ),
        sortable: true,
      },
      {
        id: "item_status_name_hu",
        name: "Státusz",
        selector: (row) => row.item_status_name_hu || row.item_status_name,
        cell: (row) => (
          <span title={row.item_status_name_hu || row.item_status_name}>
            {row.item_status_name_hu || row.item_status_name || "—"}
          </span>
        ),
        sortable: false,
      },
      {
        id: "is_leased",
        name: "Lízingelve?",
        selector: (r) => r.is_leased,
        cell: (r) => (r.is_leased ? "✅" : "—"),
        sortable: true,
      },
      {
        id: "buyer_name",
        name: "Beszerző cég",
        selector: (r) => r.buyer_name,
        sortable: true,
      },
      {
        id: "asset_type_name",
        name: "Eszköz csoport",
        selector: (r) => r.asset_type_name,
        sortable: true,
      },
      {
        id: "inventory_number",
        name: "Leltári szám",
        selector: (r) => r.inventory_number,
        sortable: true,
      },
      {
        id: "purchase_date",
        name: "Beszerzés dátuma",
        selector: (row) => row.purchase_date,
        cell: (row) =>
          row.purchase_date ? (
            <div title={convertGetDateString(row.purchase_date)}>
              {convertGetDateString(row.purchase_date)}
            </div>
          ) : null,
        sortable: true,
      },
      // {
      //   id: "price",
      //   name: "Bruttó beszerzési ár",
      //   selector: (row) => (row.price ? currencyDisplayHandler(row.price) : ""),
      //   cell: (row) => (
      //     <div title={currencyDisplayHandler(row.price)}>
      //       {row.price ? currencyDisplayHandler(row.price) : ""}
      //     </div>
      //   ),
      //   sortable: true,
      // },
      // {
      //   id: "netto_price",
      //   name: "Nettó érték",
      //   selector: (row) =>
      //     row.netto_price ? currencyDisplayHandler(row.netto_price) : "",
      //   cell: (row) => (
      //     <div title={currencyDisplayHandler(row.netto_price)}>
      //       {row.netto_price ? currencyDisplayHandler(row.netto_price) : ""}
      //     </div>
      //   ),
      //   sortable: true,
      // },
      // {
      //   id: "serial_number",
      //   name: "Gyári szám",
      //   selector: (r) => r.serial_number,
      //   sortable: true,
      // },
      // {
      //   id: "placed_at_name",
      //   name: "Feltalálási hely",
      //   selector: (r) => r.placed_at_name,
      //   sortable: true,
      // },
      // {
      //   id: "used_at_name",
      //   name: "Felhasználási hely",
      //   selector: (r) => r.used_at_name,
      //   sortable: true,
      // },
      // {
      //   id: "device_brand",
      //   name: "Gyártó",
      //   selector: (r) => r.device_brand,
      //   sortable: true,
      // },
      // {
      //   id: "device_model",
      //   name: "Modell",
      //   selector: (r) => r.device_model,
      //   sortable: true,
      // },
      {
        name: "Műveletek",
        cell: (row) => (
          <div className="flex space-x-2">
            <button
              onClick={() => openEdit(row.id)}
              className="rounded bg-blue-500/75 px-3 py-1 text-sm text-white hover:bg-blue-600">
              <Pencil className="w-4 h-4" />
            </button>
            <button
              onClick={() => handleSelectAction("delete", row)}
              className="rounded bg-red-500/75 px-3 py-1 text-sm text-white hover:bg-red-600">
              <Trash2 className="w-4 h-4" />
            </button>
          </div>
        ),
      },
    ],
    []
  );
  const renderExpandedGroup = useCallback((data, onRowUpdated) => {
    return (
      <div ref={expandedRowRef} onClick={(e) => e.stopPropagation()}>
        <ExpandedRow id={data?.id} onRowUpdated={onRowUpdated} />
      </div>
    );
  }, []);

  const handleRowUpdated = () => fetchData();

  return (
    <>
      <DataTableWrapper
        tableTitle="Eszközök"
        data={data}
        columns={columns}
        customStyles={customStyles}
        progressPending={loading}
        paginationDefaultPage={page}
        defaultSortFieldId="name"
        conditionalRowStyles={conditionalRowStyles}
        paginationServer
        onChangePage={(p) => setPage(p)}
        progressComponent={<Loader />}
        paginationTotalRows={totalRecords}
        paginationPerPage={pageSize}
        onSort={(col, dir) => {
          setSortField(col.id);
          setSortDirection(dir);
          setPage(1);
        }}
        onChangeRowsPerPage={(size) => {
          setPageSize(size);
          setPage(1);
        }}
        onFilterClear={fetchData}
        subHeader
        subHeaderComponent={subHeaderComponent}
        subHeaderAlign="right"
        subHeaderWrap
        selectableRows
        onRowSelected={setSelectedRows}
        toggleCleared={toggleCleared}
        expandableRows
        expandableRowExpanded={(rowData) => rowData?.id === expandedRow?.id}
        onRowExpandToggled={(toggleState, rowData) => {
          if (toggleState) {
            if (expandedRow !== null && expandedRow._id !== rowData._id)
              setExpandedRow(null);
            setExpandedRow(rowData);
          } else {
            setExpandedRow(null);
          }
        }}
        renderExpandableRows={(rowData) =>
          renderExpandedGroup(rowData, handleRowUpdated)
        }
      />

      {/* SlideOver for Create / Edit */}
      <SlideOver
        open={formOpen}
        title={formMode === "create" ? "Új eszköz" : "Eszköz szerkesztése"}
        onClose={() => setFormOpen(false)}>
        {!formRefs || !formAsset ? (
          <div>Betöltés…</div>
        ) : (
          <InventoryElement
            asset={formAsset}
            references={formRefs}
            onRowUpdated={handleSave} // ✅ close + reload
            isCreateMode={formMode === "create"} // ✅ choose create vs update
          />
        )}
      </SlideOver>

      {/* <SlideOver
        open={formOpen}
        title={formMode === "create" ? "Új eszköz" : "Eszköz szerkesztése"}
        onClose={() => setFormOpen(false)}>
        {!formRefs ? (
          <div>Betöltés…</div>
        ) : formMode === "create" ? (
          <InventoryElementCreate
            asset={formAsset}
            references={formRefs}
            onCreated={handleSave}
          />
        ) : (
          <InventoryElement
            asset={formAsset}
            references={formRefs}
            onRowUpdated={() => {
              handleSave();
            }}
            isCreateMode={false}
          />
        )}
      </SlideOver> */}
    </>
  );
};

export default AssetsList;
