// app/admin/audit-log/list/page.tsx
"use client";

import { useEffect, useMemo, useState, useCallback } from "react";
import toast from "react-hot-toast";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import ExpandedAuditLog from "@/components/audit-log/ExpandedAuditLog";

export type AuditRow = {
  id: string;
  at: string; // timestamptz → string
  actor_id: string | null;
  username: string | null;
  action: string;
  object_type: string;
  object_id: string | null;
  route: string | null;
  ip: string | null;
  method: string | null;
  status_code: number | null;
  success: boolean | null;
  message: string | null;
};

export default function AuditLogListPage() {
  const [rows, setRows] = useState<AuditRow[]>([]);
  const [loading, setLoading] = useState(true);
  const [page, setPage] = useState(1);
  const [pageSize, setPageSize] = useState(25);
  const [total, setTotal] = useState(0);

  // filters
  const [search, setSearch] = useState(""); // q
  const [username, setUsername] = useState(""); // username
  const [method, setMethod] = useState<string>(""); // GET/POST/...
  const [success, setSuccess] = useState<string>(""); // "", "true", "false"
  const [dateFrom, setDateFrom] = useState<string>(""); // yyyy-mm-dd
  const [dateTo, setDateTo] = useState<string>("");

  const [expandedRow, setExpandedRow] = useState<AuditRow | null>(null);

  const queryString = useMemo(() => {
    const p = new URLSearchParams();
    p.set("page", String(page));
    p.set("pageSize", String(pageSize));

    if (search.trim()) p.set("q", search.trim());
    if (username.trim()) p.set("username", username.trim());
    if (method) p.set("method", method);
    if (success) p.set("success", success);
    if (dateFrom) p.set("dateFrom", dateFrom);
    if (dateTo) p.set("dateTo", dateTo);

    return `?${p.toString()}`;
  }, [page, pageSize, search, username, method, success, dateFrom, dateTo]);

  const fetchData = async () => {
    setLoading(true);
    try {
      const res = await ApiCall.auditLog.getList(queryString);
      const body = res.data?.body;
      setRows(body?.rows ?? []);
      setTotal(body?.pagination?.total ?? 0);
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Audit napló betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchData();
  }, [queryString]);

  const ellipsis = (text?: string | null) => {
    const v = String(text ?? "");
    return (
      <div
        title={v}
        style={{
          overflow: "hidden",
          whiteSpace: "nowrap",
          textOverflow: "ellipsis",
        }}>
        {v}
      </div>
    );
  };

  const columns = [
    {
      id: "at",
      name: "Időpont",
      selector: (r: AuditRow) => r.at,
      cell: (r: AuditRow) => ellipsis(r.at),
      sortable: false,
      width: "180px",
    },
    {
      id: "username",
      name: "Felhasználó",
      selector: (r: AuditRow) => r.username ?? "",
      cell: (r: AuditRow) => ellipsis(r.username ?? ""),
      sortable: false,
      width: "160px",
    },
    {
      id: "method",
      name: "Módszer",
      selector: (r: AuditRow) => r.method ?? "",
      cell: (r: AuditRow) => ellipsis(r.method ?? ""),
      sortable: false,
      width: "80px",
    },
    {
      id: "route",
      name: "Útvonal",
      selector: (r: AuditRow) => r.route ?? "",
      cell: (r: AuditRow) => ellipsis(r.route ?? ""),
      sortable: false,
      grow: 2,
    },
    {
      id: "status_code",
      name: "HTTP kód",
      selector: (r: AuditRow) => r.status_code ?? 0,
      cell: (r: AuditRow) => ellipsis(r.status_code?.toString() ?? ""),
      sortable: false,
      width: "100px",
    },
    {
      id: "success",
      name: "Sikeres?",
      selector: (r: AuditRow) => (r.success ? "Igen" : "Nem"),
      cell: (r: AuditRow) => ellipsis(r.success ? "Igen" : "Nem"),
      sortable: false,
      width: "90px",
    },
    {
      id: "message",
      name: "Üzenet",
      selector: (r: AuditRow) => r.message ?? "",
      cell: (r: AuditRow) => ellipsis(r.message ?? ""),
      sortable: false,
      grow: 2,
    },
  ];

  const conditionalRowStyles = [
    {
      when: (r: AuditRow) => r.status_code && r.status_code >= 400,
      style: {
        backgroundColor: "#FEF2F2",
        color: "#7F1D1D",
      },
    },
  ];

  const renderExpanded = useCallback(
    (row: AuditRow) => (
      <div className="p-3">
        <ExpandedAuditLog id={row.id} />
      </div>
    ),
    []
  );

  return (
    <main className="p-4 space-y-4">
      {/* Filters */}
      <div className="flex flex-wrap gap-2 items-center">
        <input
          className="border rounded px-3 py-2"
          placeholder="Keresés (útvonal, üzenet, akció)"
          value={search}
          onChange={(e) => {
            setSearch(e.target.value);
            setPage(1);
          }}
        />
        <input
          className="border rounded px-3 py-2"
          placeholder="Felhasználó"
          value={username}
          onChange={(e) => {
            setUsername(e.target.value);
            setPage(1);
          }}
        />
        <select
          className="border rounded px-3 py-2"
          value={method}
          onChange={(e) => {
            setMethod(e.target.value);
            setPage(1);
          }}>
          <option value="">Módszer: mind</option>
          <option value="GET">GET</option>
          <option value="POST">POST</option>
          <option value="PUT">PUT</option>
          <option value="PATCH">PATCH</option>
          <option value="DELETE">DELETE</option>
        </select>
        <select
          className="border rounded px-3 py-2"
          value={success}
          onChange={(e) => {
            setSuccess(e.target.value);
            setPage(1);
          }}>
          <option value="">Státusz: mind</option>
          <option value="true">Sikeres</option>
          <option value="false">Sikertelen</option>
        </select>

        <input
          type="date"
          className="border rounded px-3 py-2"
          value={dateFrom}
          onChange={(e) => {
            setDateFrom(e.target.value);
            setPage(1);
          }}
        />
        <span className="text-sm text-gray-500">–</span>
        <input
          type="date"
          className="border rounded px-3 py-2"
          value={dateTo}
          onChange={(e) => {
            setDateTo(e.target.value);
            setPage(1);
          }}
        />
      </div>

      {/* Table */}
      <DataTableWrapper
        tableTitle="Audit napló"
        data={rows}
        columns={columns as any}
        progressPending={loading}
        paginationServer
        paginationDefaultPage={page}
        paginationPerPage={pageSize}
        paginationTotalRows={total}
        onChangePage={(p) => setPage(p)}
        onChangeRowsPerPage={(s) => {
          setPageSize(s);
          setPage(1);
        }}
        // we keep sort fixed (ORDER BY at DESC in backend),
        // so no sortServer/onSort here
        expandableRows
        expandableRowExpanded={(r: AuditRow) => r?.id === expandedRow?.id}
        onRowExpandToggled={(open: boolean, row: AuditRow) => {
          if (open) {
            setExpandedRow(row);
          } else {
            setExpandedRow(null);
          }
        }}
        renderExpandableRows={(row: AuditRow) => renderExpanded(row)}
        conditionalRowStyles={conditionalRowStyles}
      />
    </main>
  );
}
