"use client";

import React, { useEffect, useMemo, useRef, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import type {
  CaseListItem,
  CaseFlowType,
  CaseStatus,
} from "@/api/routes/cases";
import { getTranslation } from "@/i18n";

import {
  Page,
  Card,
  CardTitle,
  CardMeta,
  Chip,
  ToolbarRow,
  TextInput,
  Select,
  Button,
  SecondaryButton,
  SectionTitle,
  EmptyState,
  Field,
  Divider,
} from "@/components/ui/CardBits";
import { Tabs } from "@/components/ui/Tabs";
import { Drawer } from "@/components/ui/Drawer";
import { useRouter } from "next/navigation";

// dnd-kit
import {
  DndContext,
  DragEndEvent,
  DragOverEvent,
  DragStartEvent,
  PointerSensor,
  KeyboardSensor,
  closestCorners,
  useSensor,
  useSensors,
  useDroppable,
} from "@dnd-kit/core";
import {
  SortableContext,
  useSortable,
  verticalListSortingStrategy,
  arrayMove,
  sortableKeyboardCoordinates,
} from "@dnd-kit/sortable";
import { CSS } from "@dnd-kit/utilities";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";

function fmt(iso?: string | null) {
  if (!iso) return "-";
  try {
    return new Date(iso).toLocaleString();
  } catch {
    return iso ?? "-";
  }
}

// --- i18n helpers (status/flow enum -> label)
const tStatus = (t: any, s?: string | null) =>
  s ? t(`case.status.${s}`, { defaultValue: String(s) }) : "–";

const tFlow = (t: any, s?: string | null) =>
  s ? t(`case.flow.${s}`, { defaultValue: String(s) }) : "–";

function statusTone(s: string) {
  switch (s) {
    case "needs_decision":
      return "warn";
    case "blocked":
      return "bad";
    case "in_progress":
      return "info";
    case "closed":
      return "good";
    case "new":
      return "neutral";
    default:
      return "neutral";
  }
}

function tagToneFromText(s?: string | null) {
  const t = (s ?? "").trim().toLowerCase();
  if (!t) return "neutral" as const;

  if (t.includes("panasz")) return "warn" as const;
  if (t.includes("klin")) return "info" as const; // klinikai
  if (t.includes("ingatlan")) return "good" as const;

  let h = 0;
  for (let i = 0; i < t.length; i++) h = (h * 31 + t.charCodeAt(i)) >>> 0;
  const tones = ["neutral", "info", "good", "warn", "bad"] as const;
  return tones[h % tones.length];
}

function isOverdue(iso?: string | null) {
  if (!iso) return false;
  const t = new Date(iso).getTime();
  return Number.isFinite(t) && t < Date.now();
}

function isDueSoon(iso?: string | null, hours = 72) {
  if (!iso) return false;
  const t = new Date(iso).getTime();
  const now = Date.now();
  return Number.isFinite(t) && t >= now && t <= now + hours * 3600 * 1000;
}

const STATUS_COLUMNS: Array<{ key: CaseStatus; hint?: string }> = [
  {
    key: "needs_decision",
    hint: "i18n:dashboard.cases.status_hint.needs_decision",
  },
  { key: "blocked", hint: "i18n:dashboard.cases.status_hint.blocked" },
  { key: "in_progress", hint: "i18n:dashboard.cases.status_hint.in_progress" },
  { key: "new", hint: "i18n:dashboard.cases.status_hint.new" },
  { key: "closed", hint: "i18n:dashboard.cases.status_hint.closed" },
];

// ---------- DnD helpers ----------
type BoardState = Record<CaseStatus, CaseListItem[]>;
const ALL_STATUSES = STATUS_COLUMNS.map((c) => c.key);

function buildBoard(items: CaseListItem[]): BoardState {
  const empty = Object.fromEntries(
    ALL_STATUSES.map((s) => [s, []])
  ) as BoardState;

  const score = (it: CaseListItem) => {
    const overdue = isOverdue(it.due_at) ? 0 : 1;
    const soon = isDueSoon(it.due_at) ? 0 : 1;
    const due = it.due_at
      ? new Date(it.due_at).getTime()
      : Number.POSITIVE_INFINITY;
    const upd = it.updated_at ? new Date(it.updated_at).getTime() : 0;
    return [overdue, soon, due, -upd];
  };

  const sorted = items.slice().sort((a, b) => {
    const A = score(a);
    const B = score(b);
    for (let i = 0; i < A.length; i++) {
      if (A[i] !== B[i]) return (A[i] as number) - (B[i] as number);
    }
    return 0;
  });

  for (const it of sorted) {
    const st = it.status as CaseStatus;
    if (empty[st]) empty[st].push(it);
  }

  for (const s of ALL_STATUSES) if (!empty[s]) empty[s] = [];
  return empty;
}

function findContainer(board: BoardState, id: string): CaseStatus | null {
  for (const s of ALL_STATUSES) {
    if (board[s].some((x) => x.id === id)) return s;
  }
  return null;
}

function DroppableColumn({
  id,
  children,
}: {
  id: string;
  children: React.ReactNode;
}) {
  const { setNodeRef, isOver } = useDroppable({ id });
  return (
    <div
      ref={setNodeRef}
      style={{
        minHeight: 60,
        outline: isOver ? "2px dashed rgba(100,116,139,.6)" : "none",
        outlineOffset: 6,
        borderRadius: 12,
      }}>
      {children}
    </div>
  );
}

// ---------- Sortable Card ----------
function SortableCaseCard({
  item,
  onOpen,
  t,
}: {
  item: CaseListItem;
  onOpen: () => void;
  t: any;
}) {
  const {
    attributes,
    listeners,
    setNodeRef,
    transform,
    transition,
    isDragging,
  } = useSortable({ id: item.id });

  const style: React.CSSProperties = {
    transform: CSS.Transform.toString(transform),
    transition,
    opacity: isDragging ? 0.65 : 1,
    cursor: isDragging ? "grabbing" : "grab",
  };

  const overdue = isOverdue(item.due_at);
  const soon = !overdue && isDueSoon(item.due_at);

  return (
    <div ref={setNodeRef} style={style} {...attributes} {...listeners}>
      <Card
        className="cursor-pointer hover:border-slate-300"
        onClick={(e) => {
          e.stopPropagation();
          onOpen();
        }}>
        <CardTitle>{item.title}</CardTitle>
        <CardMeta>
          {item.code ? `${t("dashboard.cases.code")}: ${item.code} · ` : null}
          {t("dashboard.cases.priority")}: {item.priority ?? "-"} ·{" "}
          {t("dashboard.cases.due")}: {fmt(item.due_at)}
        </CardMeta>

        <div
          style={{ display: "flex", gap: 8, flexWrap: "wrap", marginTop: 10 }}>
          <Chip tone={statusTone(item.status) as any}>
            {tStatus(t, item.status)}
          </Chip>

          {item.case_type ? (
            <Chip tone={tagToneFromText(item.case_type) as any}>
              {item.case_type}
            </Chip>
          ) : null}

          {item.flow_type ? (
            <Chip>{tFlow(t, String(item.flow_type))}</Chip>
          ) : null}

          {item.playbook_name ? (
            <Chip tone={tagToneFromText(item.playbook_name) as any}>
              {item.playbook_name}
            </Chip>
          ) : null}

          {overdue ? (
            <Chip tone="bad">{t("dashboard.cases.badges.overdue")}</Chip>
          ) : null}
          {soon ? (
            <Chip tone="warn">{t("dashboard.cases.badges.due_soon")}</Chip>
          ) : null}
          {item.archived_at ? (
            <Chip tone="warn">{t("dashboard.cases.badges.archived")}</Chip>
          ) : null}
        </div>
      </Card>
    </div>
  );
}

function CasesDataTable({
  items,
  onOpen,
  loading,
  t,
}: {
  items: CaseListItem[];
  onOpen: (c: CaseListItem) => void;
  loading: boolean;
  t: any;
}) {
  const ellipsis = (text?: string | null) => {
    const v = String(text ?? "");
    return (
      <div
        title={v}
        style={{
          overflow: "hidden",
          whiteSpace: "nowrap",
          textOverflow: "ellipsis",
        }}>
        {v || "–"}
      </div>
    );
  };

  const columns = [
    {
      id: "title",
      name: t("dashboard.cases.table.title"),
      selector: (r: CaseListItem) => r.title ?? "",
      cell: (r: CaseListItem) => ellipsis(r.title ?? ""),
      sortable: true,
      grow: 3,
    },
    {
      id: "code",
      name: t("dashboard.cases.table.code"),
      selector: (r: CaseListItem) => (r.code ?? "") as any,
      cell: (r: CaseListItem) => ellipsis(r.code ?? "–"),
      sortable: true,
      width: "140px",
    },
    {
      id: "status",
      name: t("dashboard.cases.table.status"),
      selector: (r: CaseListItem) => String(r.status ?? ""),
      cell: (r: CaseListItem) => (
        <Chip tone={statusTone(String(r.status)) as any}>
          {tStatus(t, r.status)}
        </Chip>
      ),
      sortable: true,
      width: "160px",
    },
    {
      id: "priority",
      name: t("dashboard.cases.table.priority"),
      selector: (r: CaseListItem) => Number(r.priority ?? 0) as any,
      cell: (r: CaseListItem) => ellipsis(String(r.priority ?? "–")),
      sortable: true,
      width: "110px",
    },
    {
      id: "due_at",
      name: t("dashboard.cases.table.due"),
      selector: (r: CaseListItem) => String(r.due_at ?? ""),
      cell: (r: CaseListItem) => {
        const overdue = isOverdue(r.due_at);
        const soon = !overdue && isDueSoon(r.due_at);

        return (
          <div className="flex items-center gap-2 whitespace-nowrap">
            <span className="tabular-nums">{fmt(r.due_at)}</span>

            {overdue ? (
              <Chip tone="bad">{t("dashboard.cases.badges.expired")}</Chip>
            ) : null}
            {soon ? (
              <Chip tone="warn">
                {t("dashboard.cases.badges.due_soon_short")}
              </Chip>
            ) : null}
            {r.archived_at ? (
              <Chip tone="warn">{t("dashboard.cases.badges.archived")}</Chip>
            ) : null}
          </div>
        );
      },
      sortable: true,
      width: "260px",
    },
    {
      id: "case_type",
      name: t("dashboard.cases.table.case_type"),
      selector: (r: CaseListItem) => String(r.case_type ?? ""),
      cell: (r: CaseListItem) =>
        r.case_type ? (
          <Chip tone={tagToneFromText(r.case_type) as any}>{r.case_type}</Chip>
        ) : (
          "–"
        ),
      sortable: true,
      width: "160px",
    },
    {
      id: "flow_type",
      name: t("dashboard.cases.table.flow_type"),
      selector: (r: CaseListItem) => String(r.flow_type ?? ""),
      cell: (r: CaseListItem) => ellipsis(tFlow(t, String(r.flow_type))),
      sortable: true,
      width: "140px",
    },
    {
      id: "playbook_name",
      name: t("dashboard.cases.table.playbook"),
      selector: (r: CaseListItem) => String(r.playbook_name ?? ""),
      cell: (r: CaseListItem) =>
        r.playbook_name ? (
          <Chip tone={tagToneFromText(r.playbook_name) as any}>
            {r.playbook_name}
          </Chip>
        ) : (
          "–"
        ),
      sortable: true,
      width: "180px",
    },
    {
      id: "updated_at",
      name: t("dashboard.cases.table.updated"),
      selector: (r: CaseListItem) => String(r.updated_at ?? ""),
      cell: (r: CaseListItem) => ellipsis(fmt(r.updated_at)),
      sortable: true,
      width: "180px",
    },
  ];

  return (
    <DataTableWrapper
      tableTitle={t("dashboard.cases.table_title")}
      data={items}
      columns={columns as any}
      progressPending={loading}
      pagination
      paginationServer={false}
      sortServer={false}
      onRowClicked={(row: CaseListItem) => onOpen(row)}
      pointerOnHover
      highlightOnHover
      striped
    />
  );
}

// forgatókönyv DTO (light)
type PlaybookListItem = {
  id: string;
  key: string;
  name: string;
  description?: string | null;
};

export default function DashboardCasesPage() {
  const router = useRouter();
  const { t } = getTranslation();

  const [items, setItems] = useState<CaseListItem[]>([]);
  const [board, setBoard] = useState<BoardState>(() => buildBoard([]));
  const [boardRef, setBoardRef] = useState(board);

  const [activeId, setActiveId] = useState<string | null>(null);
  const [loading, setLoading] = useState(false);

  // filters
  const [q, setQ] = useState("");
  type CaseStatusFilter = CaseStatus | "all";
  const [status, setStatus] = useState<CaseStatusFilter>("all");
  const [mine, setMine] = useState<0 | 1>(1);
  const [showArchived, setShowArchived] = useState(false);

  // NEW filters
  const [flowFilter, setFlowFilter] = useState<"all" | CaseFlowType>("all");
  const [caseTypeFilter, setCaseTypeFilter] = useState<string>("");

  // playbooks for create + filter
  const [playbooks, setPlaybooks] = useState<PlaybookListItem[]>([]);
  const [playbookFilter, setPlaybookFilter] = useState<string>("all");

  const [prefsLoaded, setPrefsLoaded] = useState(false);

  // saved views
  type SavedView =
    | "my_active"
    | "needs_decision"
    | "overdue"
    | "blocked"
    | "all_active";
  const [savedView, setSavedView] = useState<SavedView>("my_active");

  type DashboardCasesPrefs = {
    q: string;
    status: CaseStatus | "all";
    mine: 0 | 1;
    showArchived: boolean;
    flowFilter: "all" | CaseFlowType;
    caseTypeFilter: string;
    playbookFilter: string;
    savedView: SavedView;
    viewMode: "kanban" | "table";
  };

  // drawers
  const [filtersOpen, setFiltersOpen] = useState(false);
  const [createOpen, setCreateOpen] = useState(false);

  // quick create (dashboard-level)
  const [newTitle, setNewTitle] = useState("");
  const [newDesc, setNewDesc] = useState("");
  const [newDueAt, setNewDueAt] = useState("");
  const [newPriority, setNewPriority] = useState<number>(3);

  // NEW quick create fields
  const [newCaseType, setNewCaseType] = useState<string>("");
  const [newFlowType, setNewFlowType] = useState<CaseFlowType>("folyamatos");
  const [newPlaybookId, setNewPlaybookId] = useState<string>("");

  const skipNextSavedViewApplyRef = useRef(true);

  const [creating, setCreating] = useState(false);

  type DashboardFilters = {
    q: string;
    status: CaseStatus | "all";
    mine: 0 | 1;
    showArchived: boolean;

    flowFilter: "all" | CaseFlowType;
    caseTypeFilter: string;
    playbookFilter: string;
  };

  const [draft, setDraft] = useState<DashboardFilters>(() => ({
    q: "",
    status: "all",
    mine: 1,
    showArchived: false,
    flowFilter: "all",
    caseTypeFilter: "",
    playbookFilter: "all",
  }));

  const sensors = useSensors(
    useSensor(PointerSensor, { activationConstraint: { distance: 6 } }),
    useSensor(KeyboardSensor, { coordinateGetter: sortableKeyboardCoordinates })
  );

  type ViewMode = "kanban" | "table";
  const [viewMode, setViewMode] = useState<ViewMode>("kanban");

  useEffect(() => setBoardRef(board), [board]);

  async function saveMyDashboardView() {
    const payload = {
      q,
      status,
      mine,
      showArchived,
      flowFilter,
      caseTypeFilter,
      playbookFilter,
      savedView,
      viewMode,
    };

    try {
      await ApiCall.me.saveDashboardCases(payload);
      toast.success(t("dashboard.cases.toast.view_saved"));
    } catch (e: any) {
      toast.error(
        e?.response?.data?.message ?? t("dashboard.cases.toast.save_failed")
      );
    }
  }

  async function setAndPersistViewMode(nextMode: ViewMode) {
    setViewMode(nextMode);

    try {
      await ApiCall.me.saveDashboardCases({
        ...getCurrentFilters(),
        savedView,
        viewMode: nextMode,
      } as any);
    } catch (e: any) {
      toast.error(
        e?.response?.data?.message ?? t("dashboard.cases.toast.save_failed")
      );
    }
  }

  async function applyFilters(
    next: DashboardFilters,
    opts?: { closeDrawer?: boolean; persist?: boolean }
  ) {
    setQ(next.q);
    setStatus(next.status);
    setMine(next.mine);
    setShowArchived(next.showArchived);
    setFlowFilter(next.flowFilter);
    setCaseTypeFilter(next.caseTypeFilter);
    setPlaybookFilter(next.playbookFilter);

    setDraft(next);

    if (opts?.closeDrawer) setFiltersOpen(false);

    await load(next);

    if (opts?.persist !== false) {
      try {
        await ApiCall.me.saveDashboardCases({
          ...next,
          savedView,
          viewMode,
        });
      } catch {
        // non-blocking
      }
    }
  }

  useEffect(() => {
    if (!prefsLoaded) return;

    if (skipNextSavedViewApplyRef.current) {
      skipNextSavedViewApplyRef.current = false;
      return;
    }

    const base = getCurrentFilters();
    let next: DashboardFilters = base;

    if (savedView === "my_active") {
      next = { ...base, mine: 1, showArchived: false, status: "all" };
    }
    if (savedView === "all_active") {
      next = { ...base, mine: 0, showArchived: false, status: "all" };
    }
    if (savedView === "needs_decision") {
      next = {
        ...base,
        mine: 0,
        showArchived: false,
        status: "needs_decision",
      };
    }
    if (savedView === "blocked") {
      next = { ...base, mine: 0, showArchived: false, status: "blocked" };
    }
    if (savedView === "overdue") {
      next = { ...base, mine: 0, showArchived: false, status: "all" };
    }

    void applyFilters(next, { persist: true });
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [savedView, prefsLoaded]);

  async function loadPlaybooks() {
    try {
      const r = await ApiCall.playbooks?.list?.({ includeInactive: 0 } as any);
      const list = (r?.data?.playbooks ?? []) as PlaybookListItem[];
      setPlaybooks(list);
    } catch {
      setPlaybooks([]);
    }
  }

  function getCurrentFilters(): DashboardFilters {
    return {
      q,
      status,
      mine,
      showArchived,
      flowFilter,
      caseTypeFilter,
      playbookFilter,
    };
  }

  async function load(filters?: DashboardFilters) {
    setLoading(true);
    try {
      const f = filters ?? getCurrentFilters();

      const r = await ApiCall.cases.list({
        q: f.q.trim() || undefined,
        ...(f.status !== "all" ? { status: f.status } : {}),
        mine: f.mine,
        ...(f.showArchived ? { includeArchived: 1 } : {}),
      });

      let list = (r.data.cases ?? []) as CaseListItem[];

      if (f.flowFilter !== "all") {
        list = list.filter((x) => x.flow_type === f.flowFilter);
      }
      if (f.caseTypeFilter.trim()) {
        const needle = f.caseTypeFilter.trim().toLowerCase();
        list = list.filter((x) =>
          (x.case_type ?? "").toLowerCase().includes(needle)
        );
      }
      if (f.playbookFilter !== "all") {
        list = list.filter((x) => (x.playbook_id ?? "") === f.playbookFilter);
      }

      if (savedView === "overdue") {
        list = list.filter((x) => isOverdue(x.due_at));
      }

      setItems(list);
      setBoard(buildBoard(list));
    } catch (e: any) {
      toast.error(
        e?.response?.data?.message ?? t("dashboard.cases.toast.load_failed")
      );
    } finally {
      setLoading(false);
    }
  }

  useEffect(() => {
    void loadPlaybooks();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  useEffect(() => {
    if (!prefsLoaded) return;
    void load(getCurrentFilters());
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [prefsLoaded]);

  const counts = useMemo(() => {
    const c: Record<string, number> = {};
    for (const it of items) c[it.status] = (c[it.status] ?? 0) + 1;
    c.total = items.length;
    c.overdue = items.filter((x) => isOverdue(x.due_at)).length;
    c.dueSoon = items.filter((x) => isDueSoon(x.due_at)).length;

    // IMPORTANT: itt eddig "projekt/folyamatos" volt, de a filterek / enumok "transient/continuous"
    c.transient = items.filter((x) => x.flow_type === "transient").length;
    c.continuous = items.filter((x) => x.flow_type === "continuous").length;

    return c;
  }, [items]);

  const openCase = (c: CaseListItem) =>
    router.push(`/admin/cases/${c.code ?? c.id}`);

  async function persistMove(opts: {
    caseRefForRoute: string;
    toStatus: CaseStatus;
    orderInStatus: string[];
  }) {
    if (ApiCall.cases.boardMove) {
      await ApiCall.cases.boardMove(opts.caseRefForRoute, {
        toStatus: opts.toStatus,
        orderedIds: opts.orderInStatus,
      } as any);
      return;
    }

    await ApiCall.cases.update(opts.caseRefForRoute, {
      status: opts.toStatus,
    } as any);
  }

  function onDragStart(e: DragStartEvent) {
    setActiveId(String(e.active.id));
  }

  function onDragOver(e: DragOverEvent) {
    const { active, over } = e;
    if (!over) return;

    const activeCaseId = String(active.id);
    const overId = String(over.id);

    setBoard((prev) => {
      const from = findContainer(prev, activeCaseId);
      const to: CaseStatus | null = (ALL_STATUSES as string[]).includes(overId)
        ? (overId as CaseStatus)
        : findContainer(prev, overId);

      if (!from || !to) return prev;
      if (from === to) return prev;

      const fromItems = prev[from];
      const toItems = prev[to];

      const activeIndex = fromItems.findIndex((x) => x.id === activeCaseId);
      if (activeIndex < 0) return prev;

      const moving = fromItems[activeIndex];

      const nextFrom = fromItems.filter((x) => x.id !== activeCaseId);
      let nextTo = toItems.slice();

      const overIndex = (ALL_STATUSES as string[]).includes(overId)
        ? -1
        : nextTo.findIndex((x) => x.id === overId);

      const insertAt = overIndex >= 0 ? overIndex : nextTo.length;
      nextTo.splice(insertAt, 0, { ...moving, status: to });

      return { ...prev, [from]: nextFrom, [to]: nextTo };
    });
  }

  function onDragEnd(e: DragEndEvent) {
    const { active, over } = e;
    setActiveId(null);
    if (!over) return;

    const activeCaseId = String(active.id);
    const overId = String(over.id);

    setBoard((prev) => {
      const from = findContainer(prev, activeCaseId);
      const to: CaseStatus | null = (ALL_STATUSES as string[]).includes(overId)
        ? (overId as CaseStatus)
        : findContainer(prev, overId);

      if (!from || !to) return prev;

      if (from === to) {
        const arr = prev[to];
        const oldIndex = arr.findIndex((x) => x.id === activeCaseId);
        const newIndex = arr.findIndex((x) => x.id === overId);
        if (oldIndex < 0 || newIndex < 0) return prev;
        const next = arrayMove(arr, oldIndex, newIndex);
        return { ...prev, [to]: next };
      }

      const nextTo = prev[to].map((x) =>
        x.id === activeCaseId ? { ...x, status: to } : x
      );
      return { ...prev, [to]: nextTo };
    });
  }

  async function onDragEndWithPersist(e: DragEndEvent) {
    onDragEnd(e);

    const caseId = String(e.active.id);
    const to = findContainer(boardRef, caseId);
    if (!to) return;

    const orderIds = boardRef[to].map((x) => x.id);
    const moved = items.find((x) => x.id === caseId);
    const ref = moved?.code ?? moved?.id ?? caseId;

    try {
      await persistMove({
        caseRefForRoute: ref,
        toStatus: to,
        orderInStatus: orderIds,
      });
      toast.success(t("dashboard.cases.toast.moved"));
    } catch {
      toast.error(t("dashboard.cases.toast.dnd_save_failed"));
      await load();
    }
  }

  useEffect(() => {
    (async () => {
      try {
        const r = await ApiCall.me.get();
        const p = r.data.preferences?.dashboardCases ?? {};
        setViewMode((p.viewMode as any) ?? "kanban");

        const next: DashboardFilters = {
          q: String(p.q ?? ""),
          status: ((p.status as any) ?? "all") as any,
          mine: (Number(p.mine ?? 1) as any) ?? 1,
          showArchived: !!p.showArchived,
          flowFilter: ((p.flowFilter as any) ?? "all") as any,
          caseTypeFilter: String(p.caseTypeFilter ?? ""),
          playbookFilter: String(p.playbookFilter ?? "all"),
        };

        setDraft(next);

        setQ(next.q);
        setStatus(next.status);
        setMine(next.mine);
        setShowArchived(next.showArchived);
        setFlowFilter(next.flowFilter as any);
        setCaseTypeFilter(next.caseTypeFilter);
        setPlaybookFilter(next.playbookFilter);

        skipNextSavedViewApplyRef.current = true;
        setSavedView((p.savedView as any) ?? "my_active");
      } catch {
        // non-blocking
      } finally {
        setPrefsLoaded(true);
      }
    })();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  async function onCreate() {
    const title = newTitle.trim();
    if (!title) return;

    setCreating(true);
    try {
      const r = await ApiCall.cases.create({
        title,
        description: newDesc.trim() || undefined,
        dueAt: newDueAt ? new Date(newDueAt).toISOString() : undefined,
        priority: newPriority,
        caseType: newCaseType.trim() || undefined,
        flowType: newFlowType,
        playbookId: newPlaybookId.trim() || undefined,
      } as any);

      setNewTitle("");
      setNewDesc("");
      setNewDueAt("");
      setNewPriority(3);
      setNewCaseType("");
      setNewFlowType("folyamatos");
      setNewPlaybookId("");

      setCreateOpen(false);
      toast.success(t("dashboard.cases.toast.created"));
      router.push(`/admin/cases/${r.data.code ?? r.data.id}`);
    } catch (e: any) {
      toast.error(
        e?.response?.data?.message ?? t("dashboard.cases.toast.create_failed")
      );
    } finally {
      setCreating(false);
    }
  }

  return (
    <Page
      title={t("dashboard.cases.page_title")}
      subtitle={
        viewMode === "kanban"
          ? t("dashboard.cases.subtitle_kanban")
          : t("dashboard.cases.subtitle_table")
      }
      actions={
        <>
          <Button onClick={() => setCreateOpen(true)}>
            {t("dashboard.cases.actions.new_case")}
          </Button>
          <SecondaryButton onClick={() => void saveMyDashboardView()}>
            {t("dashboard.cases.actions.save_view")}
          </SecondaryButton>
        </>
      }>
      <SectionTitle>{t("dashboard.cases.sections.overview")}</SectionTitle>

      <div
        className="grid gap-3"
        style={{ gridTemplateColumns: "repeat(auto-fit, minmax(220px, 1fr))" }}>
        <Card>
          <div className="text-sm opacity-70">
            {t("dashboard.cases.kpi.total")}
          </div>
          <div className="mt-1 text-3xl font-extrabold">
            {counts.total ?? 0}
          </div>
          <div className="mt-2 flex gap-2 flex-wrap">
            <Chip tone="warn">
              {tStatus(t, "needs_decision")}: {counts.needs_decision ?? 0}
            </Chip>
            <Chip tone="bad">
              {tStatus(t, "blocked")}: {counts.blocked ?? 0}
            </Chip>
          </div>
        </Card>

        <Card>
          <div className="text-sm opacity-70">
            {t("dashboard.cases.kpi.deadlines")}
          </div>
          <div className="mt-2 flex gap-2 flex-wrap">
            <Chip tone="bad">
              {t("dashboard.cases.kpi.expired")}: {counts.overdue ?? 0}
            </Chip>
            <Chip tone="warn">
              {t("dashboard.cases.kpi.within_72h")}: {counts.dueSoon ?? 0}
            </Chip>
          </div>
        </Card>

        <Card>
          <div className="text-sm opacity-70">
            {t("dashboard.cases.kpi.flow")}
          </div>
          <div className="mt-2 flex gap-2 flex-wrap">
            <Chip>
              {tFlow(t, "transient")}: {counts.transient ?? 0}
            </Chip>
            <Chip>
              {tFlow(t, "continuous")}: {counts.continuous ?? 0}
            </Chip>
          </div>
        </Card>
      </div>

      <Divider />

      <SectionTitle>{t("dashboard.cases.sections.views")}</SectionTitle>
      <div className="mb-4 flex flex-wrap items-center justify-between gap-3">
        <Tabs
          value={savedView}
          onChange={setSavedView as any}
          tabs={[
            { id: "my_active", label: t("dashboard.cases.views.my_active") },
            { id: "needs_decision", label: tStatus(t, "needs_decision") },
            { id: "overdue", label: t("dashboard.cases.views.overdue") },
            { id: "blocked", label: tStatus(t, "blocked") },
            { id: "all_active", label: t("dashboard.cases.views.all_active") },
          ]}
        />

        <div className="flex items-center gap-2">
          <SecondaryButton
            onClick={() => void setAndPersistViewMode("kanban")}
            disabled={viewMode === "kanban"}
            className={`${
              viewMode === "kanban" ? "bg-green-500 text-white" : ""
            } disabled:text-gray-600`}>
            {t("dashboard.cases.view_mode.kanban")}
          </SecondaryButton>

          <SecondaryButton
            onClick={() => void setAndPersistViewMode("table")}
            disabled={viewMode === "table"}
            className={`${
              viewMode === "table" ? "bg-green-500 text-white" : ""
            } disabled:text-gray-600`}>
            {t("dashboard.cases.view_mode.table")}
          </SecondaryButton>

          <SecondaryButton
            onClick={() => {
              setDraft(getCurrentFilters());
              setFiltersOpen(true);
            }}>
            {t("dashboard.cases.actions.filters")}
          </SecondaryButton>

          <Button onClick={() => void load()} disabled={loading}>
            {loading ? "..." : t("dashboard.cases.actions.refresh")}
          </Button>
        </div>
      </div>

      <Drawer
        open={filtersOpen}
        onClose={() => setFiltersOpen(false)}
        title={t("dashboard.cases.filters.title")}
        widthClass="max-w-2xl">
        <div className="grid gap-3">
          <Field label={t("dashboard.cases.filters.search.label")}>
            <TextInput
              value={draft.q}
              onChange={(e) => setDraft((p) => ({ ...p, q: e.target.value }))}
              placeholder={t("dashboard.cases.filters.search.placeholder")}
            />
          </Field>

          <div className="grid gap-3 md:grid-cols-2">
            <Field label={t("dashboard.cases.filters.status.label")}>
              <Select
                value={draft.status}
                onChange={(e) =>
                  setDraft((p) => ({ ...p, status: e.target.value as any }))
                }>
                <option value="all">{t("dashboard.common.all")}</option>
                <option value="new">{tStatus(t, "new")}</option>
                <option value="in_progress">{tStatus(t, "in_progress")}</option>
                <option value="needs_decision">
                  {tStatus(t, "needs_decision")}
                </option>
                <option value="blocked">{tStatus(t, "blocked")}</option>
                <option value="closed">{tStatus(t, "closed")}</option>
              </Select>
            </Field>

            <Field label={t("dashboard.cases.filters.scope.label")}>
              <Select
                value={draft.mine}
                onChange={(e) =>
                  setDraft((p) => ({
                    ...p,
                    mine: Number(e.target.value) as any,
                  }))
                }>
                <option value={0}>
                  {t("dashboard.cases.filters.scope.all")}
                </option>
                <option value={1}>
                  {t("dashboard.cases.filters.scope.mine")}
                </option>
              </Select>
            </Field>

            <Field label={t("dashboard.cases.filters.archived.label")}>
              <Select
                value={draft.showArchived ? "1" : "0"}
                onChange={(e) =>
                  setDraft((p) => ({
                    ...p,
                    showArchived: e.target.value === "1",
                  }))
                }>
                <option value="0">
                  {t("dashboard.cases.filters.archived.only_active")}
                </option>
                <option value="1">
                  {t("dashboard.cases.filters.archived.with_archived")}
                </option>
              </Select>
            </Field>

            <Field label={t("dashboard.cases.filters.flow.label")}>
              <Select
                value={draft.flowFilter}
                onChange={(e) =>
                  setDraft((p) => ({ ...p, flowFilter: e.target.value as any }))
                }>
                <option value="all">{t("dashboard.common.all")}</option>
                <option value="transient">{tFlow(t, "transient")}</option>
                <option value="continuous">{tFlow(t, "continuous")}</option>
              </Select>
            </Field>

            <Field label={t("dashboard.cases.filters.case_type.label")}>
              <TextInput
                value={draft.caseTypeFilter}
                onChange={(e) =>
                  setDraft((p) => ({ ...p, caseTypeFilter: e.target.value }))
                }
                placeholder={t("dashboard.cases.filters.case_type.placeholder")}
              />
            </Field>

            <Field label={t("dashboard.cases.filters.playbook.label")}>
              <Select
                value={draft.playbookFilter}
                onChange={(e) =>
                  setDraft((p) => ({ ...p, playbookFilter: e.target.value }))
                }>
                <option value="all">{t("dashboard.common.all")}</option>
                {playbooks.map((p) => (
                  <option key={p.id} value={p.id}>
                    {p.name}
                  </option>
                ))}
              </Select>
            </Field>
          </div>

          <ToolbarRow>
            <Button
              onClick={() => void applyFilters(draft, { closeDrawer: true })}
              disabled={loading}>
              {loading ? "..." : t("dashboard.common.apply")}
            </Button>

            <SecondaryButton
              onClick={() => {
                const clean: DashboardFilters = {
                  q: "",
                  status: "all",
                  mine: 1,
                  showArchived: false,
                  flowFilter: "all",
                  caseTypeFilter: "",
                  playbookFilter: "all",
                };
                setDraft(clean);
                void applyFilters(clean, { closeDrawer: true });
              }}>
              {t("dashboard.common.clear_filters")}
            </SecondaryButton>
          </ToolbarRow>
        </div>
      </Drawer>

      <Divider />

      {items.length ? (
        <>
          {viewMode === "kanban" ? (
            <DndContext
              sensors={sensors}
              collisionDetection={closestCorners}
              onDragStart={onDragStart}
              onDragOver={onDragOver}
              onDragEnd={onDragEndWithPersist}>
              <div
                className="grid gap-3"
                style={{
                  gridTemplateColumns: "repeat(auto-fit, minmax(280px, 1fr))",
                  alignItems: "start",
                }}>
                {STATUS_COLUMNS.map((col) => {
                  const list = board[col.key] ?? [];
                  return (
                    <div key={col.key}>
                      <Card>
                        <div
                          style={{
                            display: "flex",
                            justifyContent: "space-between",
                            gap: 12,
                            alignItems: "center",
                            marginBottom: 10,
                          }}>
                          <div>
                            <div className="text-sm font-extrabold">
                              {tStatus(t, col.key)}
                            </div>

                            {col.hint ? (
                              <div className="text-xs opacity-60">
                                {t((col.hint || "").replace("i18n:", ""))}
                              </div>
                            ) : null}
                          </div>

                          <Chip tone={statusTone(col.key) as any} dot={false}>
                            {list.length}
                          </Chip>
                        </div>

                        <DroppableColumn id={col.key}>
                          <SortableContext
                            items={list.map((x) => x.id)}
                            strategy={verticalListSortingStrategy}>
                            <div className="grid gap-2">
                              {list.length ? (
                                list.map((c) => (
                                  <SortableCaseCard
                                    key={c.id}
                                    item={c}
                                    onOpen={() => openCase(c)}
                                    t={t}
                                  />
                                ))
                              ) : (
                                <div className="text-xs opacity-70">
                                  {t("dashboard.cases.kanban.drop_here")}
                                </div>
                              )}
                            </div>
                          </SortableContext>
                        </DroppableColumn>

                        {activeId ? (
                          <div className="mt-3 text-xs opacity-60">
                            {t("dashboard.cases.kanban.tip")}
                          </div>
                        ) : null}
                      </Card>
                    </div>
                  );
                })}
              </div>
            </DndContext>
          ) : (
            <CasesDataTable
              items={items}
              onOpen={openCase}
              loading={loading}
              t={t}
            />
          )}
        </>
      ) : (
        <EmptyState
          title={
            loading
              ? t("dashboard.common.loading")
              : t("dashboard.cases.empty.title")
          }
          hint={t("dashboard.cases.empty.hint")}
        />
      )}

      <Divider />

      <Drawer
        open={createOpen}
        onClose={() => setCreateOpen(false)}
        title={t("dashboard.cases.create.title")}>
        <div className="grid gap-3">
          <Field
            label={t("dashboard.cases.create.fields.title.label")}
            hint={t("dashboard.cases.create.fields.title.hint")}>
            <TextInput
              value={newTitle}
              onChange={(e) => setNewTitle(e.target.value)}
              placeholder={t("dashboard.cases.create.fields.title.placeholder")}
            />
          </Field>

          <Field
            label={t("dashboard.cases.create.fields.description.label")}
            hint={t("dashboard.cases.create.fields.description.hint")}>
            <TextInput
              value={newDesc}
              onChange={(e) => setNewDesc(e.target.value)}
              placeholder={t(
                "dashboard.cases.create.fields.description.placeholder"
              )}
            />
          </Field>

          <div className="grid gap-3 md:grid-cols-2">
            <Field label={t("dashboard.cases.create.fields.due.label")}>
              <input
                type="datetime-local"
                value={newDueAt}
                onChange={(e) => setNewDueAt(e.target.value)}
                className="w-full rounded-xl border border-slate-200 bg-white px-3 py-2 text-sm shadow-sm"
              />
            </Field>

            <Field label={t("dashboard.cases.create.fields.priority.label")}>
              <Select
                value={newPriority}
                onChange={(e) => setNewPriority(Number(e.target.value))}>
                {[1, 2, 3, 4, 5].map((p) => (
                  <option key={p} value={p}>
                    {p}
                  </option>
                ))}
              </Select>
            </Field>
          </div>

          <Divider />

          <div className="grid gap-3 md:grid-cols-2">
            <Field
              label={t("dashboard.cases.create.fields.case_type.label")}
              hint={t("dashboard.cases.create.fields.case_type.hint")}>
              <TextInput
                value={newCaseType}
                onChange={(e) => setNewCaseType(e.target.value)}
                placeholder={t(
                  "dashboard.cases.create.fields.case_type.placeholder"
                )}
              />
            </Field>

            <Field label={t("dashboard.cases.create.fields.flow_type.label")}>
              <Select
                value={newFlowType}
                onChange={(e) =>
                  setNewFlowType(e.target.value as CaseFlowType)
                }>
                <option value="transient">{tFlow(t, "transient")}</option>
                <option value="continuous">{tFlow(t, "continuous")}</option>
              </Select>
            </Field>
          </div>

          <Field
            label={t("dashboard.cases.create.fields.playbook.label")}
            hint={t("dashboard.cases.create.fields.playbook.hint")}>
            <Select
              value={newPlaybookId}
              onChange={(e) => setNewPlaybookId(e.target.value)}>
              <option value="">
                {t("dashboard.cases.create.fields.playbook.none")}
              </option>
              {playbooks.map((p) => (
                <option key={p.id} value={p.id}>
                  {p.name}
                </option>
              ))}
            </Select>
          </Field>

          <ToolbarRow>
            <Button
              onClick={() => void onCreate()}
              disabled={creating || !newTitle.trim()}>
              {creating ? "..." : t("dashboard.common.create")}
            </Button>
            <SecondaryButton
              onClick={() => setCreateOpen(false)}
              disabled={creating}>
              {t("dashboard.common.cancel")}
            </SecondaryButton>
          </ToolbarRow>

          <div className="text-xs opacity-70">
            {t("dashboard.cases.create.tip")}
          </div>
        </div>
      </Drawer>
    </Page>
  );
}

// "use client";

// import React, { useEffect, useMemo, useRef, useState } from "react";
// import toast from "react-hot-toast";
// import ApiCall from "@/api/api-calls";
// import type {
//   CaseListItem,
//   CaseFlowType,
//   CaseStatus,
// } from "@/api/routes/cases";
// import { getTranslation } from "@/i18n";

// import {
//   Page,
//   Card,
//   CardTitle,
//   CardMeta,
//   Chip,
//   ToolbarRow,
//   TextInput,
//   Select,
//   Button,
//   SecondaryButton,
//   SectionTitle,
//   EmptyState,
//   Field,
//   Divider,
// } from "@/components/ui/CardBits";
// import { Tabs } from "@/components/ui/Tabs";
// import { Drawer } from "@/components/ui/Drawer";
// import { useRouter } from "next/navigation";

// // dnd-kit
// import {
//   DndContext,
//   DragEndEvent,
//   DragOverEvent,
//   DragStartEvent,
//   PointerSensor,
//   KeyboardSensor,
//   closestCorners,
//   useSensor,
//   useSensors,
//   useDroppable,
// } from "@dnd-kit/core";
// import {
//   SortableContext,
//   useSortable,
//   verticalListSortingStrategy,
//   arrayMove,
//   sortableKeyboardCoordinates,
// } from "@dnd-kit/sortable";
// import { CSS } from "@dnd-kit/utilities";
// import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";

// function fmt(iso?: string | null) {
//   if (!iso) return "-";
//   try {
//     return new Date(iso).toLocaleString();
//   } catch {
//     return iso ?? "-";
//   }
// }
// const tStatus = (t: any, s?: string | null) =>
//   s ? t(`case.status.${s}`, { defaultValue: s }) : "–";

// const tFlow = (t: any, s?: string | null) =>
//   s ? t(`case.flow.${s}`, { defaultValue: s }) : "–";

// function statusTone(s: string) {
//   switch (s) {
//     case "needs_decision":
//       return "warn";
//     case "blocked":
//       return "bad";
//     case "in_progress":
//       return "info";
//     case "closed":
//       return "good";
//     case "new":
//       return "neutral";
//     default:
//       return "neutral";
//   }
// }
// function tagToneFromText(s?: string | null) {
//   const t = (s ?? "").trim().toLowerCase();
//   if (!t) return "neutral" as const;

//   // pár hasznos fix mapping (amit akartok, bővíthető)
//   if (t.includes("panasz")) return "warn" as const;
//   if (t.includes("klin")) return "info" as const; // klinikai
//   if (t.includes("ingatlan")) return "good" as const;

//   // fallback: egyszerű hash -> tone (deterministic)
//   let h = 0;
//   for (let i = 0; i < t.length; i++) h = (h * 31 + t.charCodeAt(i)) >>> 0;
//   const tones = ["neutral", "info", "good", "warn", "bad"] as const;
//   return tones[h % tones.length];
// }

// function isOverdue(iso?: string | null) {
//   if (!iso) return false;
//   const t = new Date(iso).getTime();
//   return Number.isFinite(t) && t < Date.now();
// }

// function isDueSoon(iso?: string | null, hours = 72) {
//   if (!iso) return false;
//   const t = new Date(iso).getTime();
//   const now = Date.now();
//   return Number.isFinite(t) && t >= now && t <= now + hours * 3600 * 1000;
// }

// const STATUS_COLUMNS: Array<{
//   key: CaseStatus;
//   label: string;
//   hint?: string;
// }> = [
//   {
//     key: "needs_decision",
//     label: "Döntés szükséges",
//     hint: "vezetői döntésre vár",
//   },
//   { key: "blocked", label: "Elakadás", hint: "elakadt / akadály" },
//   { key: "in_progress", label: "Folyamatban", hint: "folyamatban" },
//   { key: "new", label: "Új", hint: "új ügy" },
//   { key: "closed", label: "Lezárva", hint: "lezárt" },
// ];

// // ---------- DnD helpers ----------
// type BoardState = Record<CaseStatus, CaseListItem[]>;
// const ALL_STATUSES = STATUS_COLUMNS.map((c) => c.key);

// function buildBoard(items: CaseListItem[]): BoardState {
//   const empty = Object.fromEntries(
//     ALL_STATUSES.map((s) => [s, []])
//   ) as BoardState;

//   // default sorting inside status: overdue/soon first, then dueAt, then updated
//   const score = (it: CaseListItem) => {
//     const overdue = isOverdue(it.due_at) ? 0 : 1;
//     const soon = isDueSoon(it.due_at) ? 0 : 1;
//     const due = it.due_at
//       ? new Date(it.due_at).getTime()
//       : Number.POSITIVE_INFINITY;
//     const upd = it.updated_at ? new Date(it.updated_at).getTime() : 0;
//     return [overdue, soon, due, -upd];
//   };

//   const sorted = items.slice().sort((a, b) => {
//     const A = score(a);
//     const B = score(b);
//     for (let i = 0; i < A.length; i++) {
//       if (A[i] !== B[i]) return (A[i] as number) - (B[i] as number);
//     }
//     return 0;
//   });

//   for (const it of sorted) {
//     const st = it.status as CaseStatus;
//     if (empty[st]) empty[st].push(it);
//   }

//   // ensure all statuses exist
//   for (const s of ALL_STATUSES) if (!empty[s]) empty[s] = [];
//   return empty;
// }

// function findContainer(board: BoardState, id: string): CaseStatus | null {
//   for (const s of ALL_STATUSES) {
//     if (board[s].some((x) => x.id === id)) return s;
//   }
//   return null;
// }

// function DroppableColumn({
//   id,
//   children,
// }: {
//   id: string;
//   children: React.ReactNode;
// }) {
//   const { setNodeRef, isOver } = useDroppable({ id });
//   return (
//     <div
//       ref={setNodeRef}
//       style={{
//         minHeight: 60,
//         outline: isOver ? "2px dashed rgba(100,116,139,.6)" : "none",
//         outlineOffset: 6,
//         borderRadius: 12,
//       }}>
//       {children}
//     </div>
//   );
// }

// // ---------- Sortable Card ----------
// function SortableCaseCard({
//   item,
//   onOpen,
//   t,
// }: {
//   item: CaseListItem;
//   onOpen: () => void;
//   t: any;
// }) {
//   const {
//     attributes,
//     listeners,
//     setNodeRef,
//     transform,
//     transition,
//     isDragging,
//   } = useSortable({ id: item.id });

//   const style: React.CSSProperties = {
//     transform: CSS.Transform.toString(transform),
//     transition,
//     opacity: isDragging ? 0.65 : 1,
//     cursor: isDragging ? "grabbing" : "grab",
//   };

//   const overdue = isOverdue(item.due_at);
//   const soon = !overdue && isDueSoon(item.due_at);

//   return (
//     <div ref={setNodeRef} style={style} {...attributes} {...listeners}>
//       <Card
//         className="cursor-pointer hover:border-slate-300"
//         onClick={(e) => {
//           e.stopPropagation();
//           onOpen();
//         }}>
//         <CardTitle>{item.title}</CardTitle>
//         <CardMeta>
//           {item.code ? `kód: ${item.code} · ` : null}
//           Prioritás: {item.priority ?? "-"} · Határidő: {fmt(item.due_at)}
//         </CardMeta>

//         <div
//           style={{ display: "flex", gap: 8, flexWrap: "wrap", marginTop: 10 }}>
//           <Chip tone={statusTone(item.status) as any}>
//             {tStatus(t, item.status)}
//           </Chip>
//           {/* {item.case_type ? <Chip tone="info">{item.case_type}</Chip> : null} */}
//           {item.case_type ? (
//             <Chip tone={tagToneFromText(item.case_type) as any}>
//               {item.case_type}
//             </Chip>
//           ) : null}

//           {item.flow_type ? <Chip>{String(item.flow_type)}</Chip> : null}
//           {item.playbook_name ? (
//             <Chip tone={tagToneFromText(item.playbook_name) as any}>
//               {item.playbook_name}
//             </Chip>
//           ) : null}

//           {overdue ? <Chip tone="bad">Határidőn túli</Chip> : null}
//           {soon ? <Chip tone="warn">Közelgő határidő</Chip> : null}
//           {item.archived_at ? <Chip tone="warn">Archivált</Chip> : null}
//         </div>
//       </Card>
//     </div>
//   );
// }

// function CasesDataTable({
//   items,
//   onOpen,
//   loading,
//   t,
// }: {
//   items: CaseListItem[];
//   onOpen: (c: CaseListItem) => void;
//   loading: boolean;
//   t: any;
// }) {
//   // kis segéd: legyen ellipsis kompatibilis cell, mint a másik oldalon
//   const ellipsis = (text?: string | null) => {
//     const v = String(text ?? "");
//     return (
//       <div
//         title={v}
//         style={{
//           overflow: "hidden",
//           whiteSpace: "nowrap",
//           textOverflow: "ellipsis",
//         }}>
//         {v || "–"}
//       </div>
//     );
//   };

//   const columns = [
//     {
//       id: "title",
//       name: "Cím",
//       selector: (r: CaseListItem) => r.title ?? "",
//       cell: (r: CaseListItem) => ellipsis(r.title ?? ""),
//       sortable: true,
//       grow: 3,
//       // minWidth: "260px",
//     },
//     {
//       id: "code",
//       name: "Kód",
//       selector: (r: CaseListItem) => (r.code ?? "") as any,
//       cell: (r: CaseListItem) => ellipsis(r.code ?? "–"),
//       sortable: true,
//       width: "140px",
//     },
//     {
//       id: "status",
//       name: "Státusz",
//       selector: (r: CaseListItem) => String(r.status ?? ""),
//       cell: (r: CaseListItem) => (
//         <Chip tone={statusTone(String(r.status)) as any}>
//           {tStatus(t, r.status)}
//           {/* {String(r.status)} */}
//         </Chip>
//       ),
//       sortable: true,
//       width: "160px",
//     },
//     {
//       id: "priority",
//       name: "Prioritás",
//       selector: (r: CaseListItem) => Number(r.priority ?? 0) as any,
//       cell: (r: CaseListItem) => ellipsis(String(r.priority ?? "–")),
//       sortable: true,
//       width: "110px",
//     },
//     {
//       id: "due_at",
//       name: "Határidő",
//       selector: (r: CaseListItem) => String(r.due_at ?? ""),
//       cell: (r: CaseListItem) => {
//         const overdue = isOverdue(r.due_at);
//         const soon = !overdue && isDueSoon(r.due_at);

//         return (
//           <div className="flex items-center gap-2 whitespace-nowrap">
//             <span className="tabular-nums">{fmt(r.due_at)}</span>

//             {overdue ? <Chip tone="bad">Lejárt</Chip> : null}
//             {soon ? <Chip tone="warn">Közelgő</Chip> : null}
//             {r.archived_at ? <Chip tone="warn">Archivált</Chip> : null}
//           </div>
//         );
//       },
//       sortable: true,
//       width: "260px",
//     },

//     {
//       id: "case_type",
//       name: "Ügytípus",
//       selector: (r: CaseListItem) => String(r.case_type ?? ""),
//       cell: (r: CaseListItem) =>
//         r.case_type ? (
//           <Chip tone={tagToneFromText(r.case_type) as any}>{r.case_type}</Chip>
//         ) : (
//           "–"
//         ),
//       sortable: true,
//       width: "160px",
//     },
//     {
//       id: "flow_type",
//       name: "Típus",
//       selector: (r: CaseListItem) => String(r.flow_type ?? ""),
//       cell: (r: CaseListItem) =>
//         ellipsis(
//           String(
//             r.flow_type && r.flow_type === "transient"
//               ? "Projekt"
//               : r.flow_type && r.flow_type === "continuous"
//               ? "Folyamatos"
//               : "–"
//           )
//         ),
//       sortable: true,
//       width: "140px",
//     },
//     {
//       id: "playbook_name",
//       name: "Forgatókönyv",
//       selector: (r: CaseListItem) => String(r.playbook_name ?? ""),
//       cell: (r: CaseListItem) =>
//         r.playbook_name ? (
//           <Chip tone={tagToneFromText(r.playbook_name) as any}>
//             {r.playbook_name}
//           </Chip>
//         ) : (
//           "–"
//         ),
//       sortable: true,
//       width: "180px",
//     },
//     {
//       id: "updated_at",
//       name: "Frissítve",
//       selector: (r: CaseListItem) => String(r.updated_at ?? ""),
//       cell: (r: CaseListItem) => ellipsis(fmt(r.updated_at)),
//       sortable: true,
//       width: "180px",
//     },
//   ];

//   return (
//     <DataTableWrapper
//       tableTitle="Ügyek"
//       data={items}
//       columns={columns as any}
//       progressPending={loading}
//       pagination
//       paginationServer={false}
//       sortServer={false}
//       // kattintásra nyitás (RDT támogatja)
//       onRowClicked={(row: CaseListItem) => onOpen(row)}
//       pointerOnHover
//       highlightOnHover
//       striped
//     />
//   );
// }

// // forgatókönyv DTO (light)
// type PlaybookListItem = {
//   id: string;
//   key: string;
//   name: string;
//   description?: string | null;
// };

// export default function DashboardCasesPage() {
//   const router = useRouter();
//   const { t } = getTranslation();
//   const [items, setItems] = useState<CaseListItem[]>([]);
//   const [board, setBoard] = useState<BoardState>(() => buildBoard([]));
//   const [boardRef, setBoardRef] = useState(board);

//   const [activeId, setActiveId] = useState<string | null>(null);
//   const [loading, setLoading] = useState(false);

//   // filters
//   const [q, setQ] = useState("");
//   type CaseStatusFilter = CaseStatus | "all";
//   const [status, setStatus] = useState<CaseStatusFilter>("all");
//   const [mine, setMine] = useState<0 | 1>(1);
//   const [showArchived, setShowArchived] = useState(false);

//   // NEW filters
//   const [flowFilter, setFlowFilter] = useState<"all" | CaseFlowType>("all");
//   const [caseTypeFilter, setCaseTypeFilter] = useState<string>("");

//   // playbooks for create + filter
//   const [playbooks, setPlaybooks] = useState<PlaybookListItem[]>([]);
//   const [playbookFilter, setPlaybookFilter] = useState<string>("all");

//   const [prefsLoaded, setPrefsLoaded] = useState(false);
//   // saved views
//   type SavedView =
//     | "my_active"
//     | "needs_decision"
//     | "overdue"
//     | "blocked"
//     | "all_active";
//   const [savedView, setSavedView] = useState<SavedView>("my_active");
//   type DashboardCasesPrefs = {
//     q: string;
//     status: CaseStatus | "all";
//     mine: 0 | 1;
//     showArchived: boolean;
//     flowFilter: "all" | CaseFlowType;
//     caseTypeFilter: string;
//     playbookFilter: string;
//     savedView: SavedView;

//     viewMode: "kanban" | "table"; // 👈 új
//   };

//   // drawers
//   const [filtersOpen, setFiltersOpen] = useState(false);
//   const [createOpen, setCreateOpen] = useState(false);

//   // quick create (dashboard-level)
//   const [newTitle, setNewTitle] = useState("");
//   const [newDesc, setNewDesc] = useState("");
//   const [newDueAt, setNewDueAt] = useState("");
//   const [newPriority, setNewPriority] = useState<number>(3);

//   // NEW quick create fields
//   const [newCaseType, setNewCaseType] = useState<string>("");
//   const [newFlowType, setNewFlowType] = useState<CaseFlowType>("folyamatos");
//   const [newPlaybookId, setNewPlaybookId] = useState<string>("");

//   const skipNextSavedViewApplyRef = useRef(true);

//   const [creating, setCreating] = useState(false);
//   type DashboardFilters = {
//     q: string;
//     status: CaseStatus | "all";
//     mine: 0 | 1;
//     showArchived: boolean;

//     flowFilter: "all" | CaseFlowType;
//     caseTypeFilter: string;
//     playbookFilter: string; // "all" | playbook_id
//   };
//   type DashboardFiltersDraft = DashboardFilters;
//   const [draft, setDraft] = useState<DashboardFilters>(() => ({
//     q: "",
//     status: "all",
//     mine: 1,
//     showArchived: false,
//     flowFilter: "all",
//     caseTypeFilter: "",
//     playbookFilter: "all",
//   }));
//   const sensors = useSensors(
//     useSensor(PointerSensor, { activationConstraint: { distance: 6 } }),
//     useSensor(KeyboardSensor, { coordinateGetter: sortableKeyboardCoordinates })
//   );
//   type ViewMode = "kanban" | "table";
//   const [viewMode, setViewMode] = useState<ViewMode>("kanban");

//   useEffect(() => setBoardRef(board), [board]);
//   async function saveMyDashboardView() {
//     const payload = {
//       q,
//       status,
//       mine,
//       showArchived,
//       flowFilter,
//       caseTypeFilter,
//       playbookFilter,
//       savedView,
//       viewMode,
//     };

//     try {
//       await ApiCall.me.saveDashboardCases(payload);
//       toast.success("Nézet elmentve (csak neked)");
//     } catch (e: any) {
//       toast.error(e?.response?.data?.message ?? "Mentés sikertelen");
//     }
//   }
//   async function setAndPersistViewMode(nextMode: ViewMode) {
//     setViewMode(nextMode);

//     try {
//       await ApiCall.me.saveDashboardCases({
//         ...getCurrentFilters(),
//         savedView,
//         viewMode: nextMode,
//       } as any);
//     } catch (e: any) {
//       toast.error(e?.response?.data?.message ?? "Nézet mentése sikertelen");
//     }
//   }

//   async function applyFilters(
//     next: DashboardFilters,
//     opts?: { closeDrawer?: boolean; persist?: boolean }
//   ) {
//     // main state (applied)
//     setQ(next.q);
//     setStatus(next.status);
//     setMine(next.mine);
//     setShowArchived(next.showArchived);
//     setFlowFilter(next.flowFilter);
//     setCaseTypeFilter(next.caseTypeFilter);
//     setPlaybookFilter(next.playbookFilter);

//     // ✅ drawer draft mindig kövesse
//     setDraft(next);

//     if (opts?.closeDrawer) setFiltersOpen(false);

//     // ✅ load snapshot alapján
//     await load(next);

//     // ✅ autosave (default: true az Apply-nál, de kapcsolható)
//     if (opts?.persist !== false) {
//       try {
//         await ApiCall.me.saveDashboardCases({
//           ...next,
//           savedView,
//           viewMode,
//         });
//       } catch {
//         // non-blocking; ne ölje meg az UX-et
//       }
//     }
//   }

//   useEffect(() => {
//     if (!prefsLoaded) return;

//     // ✅ amikor prefs-ből beállítjuk a savedView-t, NE írja felül rögtön
//     if (skipNextSavedViewApplyRef.current) {
//       skipNextSavedViewApplyRef.current = false;
//       return;
//     }

//     const base = getCurrentFilters();
//     let next: DashboardFilters = base;

//     if (savedView === "my_active") {
//       next = { ...base, mine: 1, showArchived: false, status: "all" };
//     }
//     if (savedView === "all_active") {
//       next = { ...base, mine: 0, showArchived: false, status: "all" };
//     }
//     if (savedView === "needs_decision") {
//       next = {
//         ...base,
//         mine: 0,
//         showArchived: false,
//         status: "needs_decision",
//       };
//     }
//     if (savedView === "blocked") {
//       next = { ...base, mine: 0, showArchived: false, status: "blocked" };
//     }
//     if (savedView === "overdue") {
//       next = { ...base, mine: 0, showArchived: false, status: "all" };
//     }

//     void applyFilters(next, { persist: true }); // ✅ preset váltás is menthető
//     // eslint-disable-next-line react-hooks/exhaustive-deps
//   }, [savedView, prefsLoaded]);

//   async function loadPlaybooks() {
//     try {
//       const r = await ApiCall.playbooks?.list?.({ includeInactive: 0 } as any);
//       const list = (r?.data?.playbooks ?? []) as PlaybookListItem[];
//       setPlaybooks(list);
//     } catch {
//       // non-blocking
//       setPlaybooks([]);
//     }
//   }

//   function getCurrentFilters(): DashboardFilters {
//     return {
//       q,
//       status,
//       mine,
//       showArchived,
//       flowFilter,
//       caseTypeFilter,
//       playbookFilter,
//     };
//   }

//   async function load(filters?: DashboardFilters) {
//     setLoading(true);
//     try {
//       const f = filters ?? getCurrentFilters();

//       const r = await ApiCall.cases.list({
//         q: f.q.trim() || undefined,
//         ...(f.status !== "all" ? { status: f.status } : {}),
//         mine: f.mine,
//         ...(f.showArchived ? { includeArchived: 1 } : {}),
//       });

//       let list = (r.data.cases ?? []) as CaseListItem[];

//       // client-side extra filters
//       if (f.flowFilter !== "all") {
//         list = list.filter((x) => x.flow_type === f.flowFilter);
//       }
//       if (f.caseTypeFilter.trim()) {
//         const needle = f.caseTypeFilter.trim().toLowerCase();
//         list = list.filter((x) =>
//           (x.case_type ?? "").toLowerCase().includes(needle)
//         );
//       }
//       if (f.playbookFilter !== "all") {
//         list = list.filter((x) => (x.playbook_id ?? "") === f.playbookFilter);
//       }

//       // overdue saved view is computed client-side
//       if (savedView === "overdue") {
//         list = list.filter((x) => isOverdue(x.due_at));
//       }

//       setItems(list);
//       setBoard(buildBoard(list));
//     } catch (e: any) {
//       toast.error(e?.response?.data?.message ?? "Ügyek betöltése sikertelen.");
//     } finally {
//       setLoading(false);
//     }
//   }

//   useEffect(() => {
//     void loadPlaybooks();
//     // eslint-disable-next-line react-hooks/exhaustive-deps
//   }, []);

//   useEffect(() => {
//     if (!prefsLoaded) return;
//     void load(getCurrentFilters());
//     // eslint-disable-next-line react-hooks/exhaustive-deps
//   }, [prefsLoaded]);

//   const counts = useMemo(() => {
//     const c: Record<string, number> = {};
//     for (const it of items) c[it.status] = (c[it.status] ?? 0) + 1;
//     c.total = items.length;
//     c.overdue = items.filter((x) => isOverdue(x.due_at)).length;
//     c.dueSoon = items.filter((x) => isDueSoon(x.due_at)).length;

//     c.transient = items.filter((x) => x.flow_type === "projekt").length;
//     c.continuous = items.filter((x) => x.flow_type === "folyamatos").length;

//     return c;
//   }, [items]);

//   const openCase = (c: CaseListItem) =>
//     router.push(`/admin/cases/${c.code ?? c.id}`);

//   // ---------------------------------------
//   // Persist move (optional, best-effort)
//   // ---------------------------------------
//   async function persistMove(opts: {
//     caseRefForRoute: string; // code OR id
//     toStatus: CaseStatus;
//     orderInStatus: string[];
//   }) {
//     // if you have this route, use it:
//     if (ApiCall.cases.boardMove) {
//       await ApiCall.cases.boardMove(opts.caseRefForRoute, {
//         toStatus: opts.toStatus,
//         orderedIds: opts.orderInStatus,
//       } as any);
//       return;
//     }

//     // fallback: at least patch status
//     await ApiCall.cases.update(opts.caseRefForRoute, {
//       status: opts.toStatus,
//     } as any);
//   }

//   function onDragStart(e: DragStartEvent) {
//     setActiveId(String(e.active.id));
//   }

//   function onDragOver(e: DragOverEvent) {
//     const { active, over } = e;
//     if (!over) return;

//     const activeCaseId = String(active.id);
//     const overId = String(over.id);

//     setBoard((prev) => {
//       const from = findContainer(prev, activeCaseId);
//       const to: CaseStatus | null = (ALL_STATUSES as string[]).includes(overId)
//         ? (overId as CaseStatus)
//         : findContainer(prev, overId);

//       if (!from || !to) return prev;
//       if (from === to) return prev;

//       const fromItems = prev[from];
//       const toItems = prev[to];

//       const activeIndex = fromItems.findIndex((x) => x.id === activeCaseId);
//       if (activeIndex < 0) return prev;

//       const moving = fromItems[activeIndex];

//       const nextFrom = fromItems.filter((x) => x.id !== activeCaseId);
//       let nextTo = toItems.slice();

//       const overIndex = (ALL_STATUSES as string[]).includes(overId)
//         ? -1
//         : nextTo.findIndex((x) => x.id === overId);

//       const insertAt = overIndex >= 0 ? overIndex : nextTo.length;
//       nextTo.splice(insertAt, 0, { ...moving, status: to });

//       return { ...prev, [from]: nextFrom, [to]: nextTo };
//     });
//   }

//   function onDragEnd(e: DragEndEvent) {
//     const { active, over } = e;
//     setActiveId(null);
//     if (!over) return;

//     const activeCaseId = String(active.id);
//     const overId = String(over.id);

//     setBoard((prev) => {
//       const from = findContainer(prev, activeCaseId);
//       const to: CaseStatus | null = (ALL_STATUSES as string[]).includes(overId)
//         ? (overId as CaseStatus)
//         : findContainer(prev, overId);

//       if (!from || !to) return prev;

//       if (from === to) {
//         const arr = prev[to];
//         const oldIndex = arr.findIndex((x) => x.id === activeCaseId);
//         const newIndex = arr.findIndex((x) => x.id === overId);
//         if (oldIndex < 0 || newIndex < 0) return prev;
//         const next = arrayMove(arr, oldIndex, newIndex);
//         return { ...prev, [to]: next };
//       }

//       const nextTo = prev[to].map((x) =>
//         x.id === activeCaseId ? { ...x, status: to } : x
//       );
//       return { ...prev, [to]: nextTo };
//     });
//   }

//   async function onDragEndWithPersist(e: DragEndEvent) {
//     onDragEnd(e);

//     const caseId = String(e.active.id);
//     const to = findContainer(boardRef, caseId);
//     if (!to) return;

//     const orderIds = boardRef[to].map((x) => x.id);
//     const moved = items.find((x) => x.id === caseId);
//     const ref = moved?.code ?? moved?.id ?? caseId;

//     try {
//       await persistMove({
//         caseRefForRoute: ref,
//         toStatus: to,
//         orderInStatus: orderIds,
//       });
//       toast.success("Áthelyezve");
//     } catch {
//       toast.error("Mentés sikertelen (drag&drop). Frissítek…");
//       await load();
//     }
//   }
//   useEffect(() => {
//     (async () => {
//       try {
//         const r = await ApiCall.me.get();
//         const p = r.data.preferences?.dashboardCases ?? {};
//         setViewMode((p.viewMode as any) ?? "kanban");
//         const next: DashboardFilters = {
//           q: String(p.q ?? ""),
//           status: ((p.status as any) ?? "all") as any,
//           mine: (Number(p.mine ?? 1) as any) ?? 1,
//           showArchived: !!p.showArchived,
//           flowFilter: ((p.flowFilter as any) ?? "all") as any,
//           caseTypeFilter: String(p.caseTypeFilter ?? ""),
//           playbookFilter: String(p.playbookFilter ?? "all"),
//         };

//         // ✅ fontos: draft + applied ugyanaz
//         setDraft(next);

//         setQ(next.q);
//         setStatus(next.status);
//         setMine(next.mine);
//         setShowArchived(next.showArchived);
//         setFlowFilter(next.flowFilter as any);
//         setCaseTypeFilter(next.caseTypeFilter);
//         setPlaybookFilter(next.playbookFilter);

//         // ✅ savedView beállítás, de ne triggelje rögtön a preset-apply-t
//         skipNextSavedViewApplyRef.current = true;
//         setSavedView((p.savedView as any) ?? "my_active");
//       } catch {
//         // non-blocking
//       } finally {
//         setPrefsLoaded(true);
//       }
//     })();
//     // eslint-disable-next-line react-hooks/exhaustive-deps
//   }, []);

//   async function onCreate() {
//     const t = newTitle.trim();
//     if (!t) return;

//     setCreating(true);
//     try {
//       const r = await ApiCall.cases.create({
//         title: t,
//         description: newDesc.trim() || undefined,
//         dueAt: newDueAt ? new Date(newDueAt).toISOString() : undefined,
//         priority: newPriority,
//         caseType: newCaseType.trim() || undefined,
//         flowType: newFlowType,
//         playbookId: newPlaybookId.trim() || undefined,
//       } as any);

//       // reset
//       setNewTitle("");
//       setNewDesc("");
//       setNewDueAt("");
//       setNewPriority(3);
//       setNewCaseType("");
//       setNewFlowType("folyamatos");
//       setNewPlaybookId("");

//       setCreateOpen(false);
//       toast.success("Ügy létrehozva");
//       router.push(`/admin/cases/${r.data.code ?? r.data.id}`);
//     } catch (e: any) {
//       toast.error(e?.response?.data?.message ?? "Ügy létrehozása sikertelen.");
//     } finally {
//       setCreating(false);
//     }
//   }

//   return (
//     <Page
//       title="Ügyek - Vezérlőpult"
//       subtitle={
//         viewMode === "kanban"
//           ? "Húzd a kártyákat: státuszváltás + sorrend oszlopon belül."
//           : "Táblázat nézet: kattints egy sorra az ügy megnyitásához."
//       }
//       actions={
//         <>
//           {/* <SecondaryButton
//             onClick={() => {
//               setDraft(getCurrentFilters());
//               setFiltersOpen(true);
//             }}>
//             Szűrők
//           </SecondaryButton>

//           <SecondaryButton onClick={() => void load()} disabled={loading}>
//             {loading ? "..." : "Frissít"}
//           </SecondaryButton> */}
//           <Button onClick={() => setCreateOpen(true)}>+ Új ügy</Button>
//           <SecondaryButton onClick={() => void saveMyDashboardView()}>
//             Nézet mentése
//           </SecondaryButton>
//         </>
//       }>
//       {/* KPI */}
//       <SectionTitle>Áttekintés</SectionTitle>
//       <div
//         className="grid gap-3"
//         style={{ gridTemplateColumns: "repeat(auto-fit, minmax(220px, 1fr))" }}>
//         <Card>
//           <div className="text-sm opacity-70">Összes ügy</div>
//           <div className="mt-1 text-3xl font-extrabold">
//             {counts.total ?? 0}
//           </div>
//           <div className="mt-2 flex gap-2 flex-wrap">
//             <Chip tone="warn">
//               needs_decision: {counts.needs_decision ?? 0}
//             </Chip>
//             <Chip tone="bad">blocked: {counts.blocked ?? 0}</Chip>
//           </div>
//         </Card>

//         <Card>
//           <div className="text-sm opacity-70">Határidők</div>
//           <div className="mt-2 flex gap-2 flex-wrap">
//             <Chip tone="bad">lejárt: {counts.overdue ?? 0}</Chip>
//             <Chip tone="warn">72 órán belül: {counts.dueSoon ?? 0}</Chip>
//           </div>
//         </Card>

//         <Card>
//           <div className="text-sm opacity-70">Típus</div>
//           <div className="mt-2 flex gap-2 flex-wrap">
//             <Chip>projekt: {counts.transient ?? 0}</Chip>
//             <Chip>folyamatos: {counts.continuous ?? 0}</Chip>
//           </div>
//         </Card>
//       </div>

//       <Divider />

//       {/* Saved views */}
//       <SectionTitle>Nézetek</SectionTitle>
//       <div className="mb-4 flex flex-wrap items-center justify-between gap-3">
//         <Tabs
//           value={savedView}
//           onChange={setSavedView as any}
//           tabs={[
//             { id: "my_active", label: "Saját (aktív)" },
//             { id: "needs_decision", label: "Döntés szükséges" },
//             { id: "overdue", label: "Határidőn túl" },
//             { id: "blocked", label: "Elakadt" },
//             { id: "all_active", label: "Összes (aktív)" },
//           ]}
//         />
//         <div className="flex items-center gap-2">
//           <SecondaryButton
//             onClick={() => void setAndPersistViewMode("kanban")}
//             // opcionális "active" jelzés: egyszerűen halványítsuk a nem aktívat
//             disabled={viewMode === "kanban"}
//             className={`${
//               viewMode === "kanban" ? "bg-green-500 text-white" : ""
//             } disabled:text-gray-600`}>
//             Kártyák
//           </SecondaryButton>

//           <SecondaryButton
//             onClick={() => void setAndPersistViewMode("table")}
//             disabled={viewMode === "table"}
//             className={`${
//               viewMode === "table" ? "bg-green-500 text-white" : ""
//             } disabled:text-gray-600`}>
//             Táblázat
//           </SecondaryButton>

//           <SecondaryButton
//             onClick={() => {
//               setDraft(getCurrentFilters());
//               setFiltersOpen(true);
//             }}>
//             Szűrők
//           </SecondaryButton>

//           <Button onClick={() => void load()} disabled={loading}>
//             {loading ? "..." : "Frissít"}
//           </Button>
//         </div>
//         {/* <div className="flex items-center gap-2">
//           <SecondaryButton
//             onClick={() => {
//               setDraft(getCurrentFilters());
//               setFiltersOpen(true);
//             }}>
//             Szűrők
//           </SecondaryButton>

//           <Button onClick={() => void load()} disabled={loading}>
//             {loading ? "..." : "Frissít"}
//           </Button>
//         </div> */}
//       </div>

//       <Drawer
//         open={filtersOpen}
//         onClose={() => setFiltersOpen(false)}
//         title="Szűrők"
//         widthClass="max-w-2xl">
//         <div className="grid gap-3">
//           <Field label="Keresés">
//             <TextInput
//               value={draft.q}
//               onChange={(e) => setDraft((p) => ({ ...p, q: e.target.value }))}
//               placeholder="Keresés cím/kód alapján…"
//             />
//           </Field>

//           <div className="grid gap-3 md:grid-cols-2">
//             <Field label="Státusz">
//               <Select
//                 value={draft.status}
//                 onChange={(e) =>
//                   setDraft((p) => ({ ...p, status: e.target.value as any }))
//                 }>
//                 <option value="all">Összes</option>
//                 <option value="new">Új ügy</option>
//                 <option value="in_progress">Folyamatban</option>
//                 <option value="needs_decision">Döntés szükséges</option>
//                 <option value="blocked">Elakadt</option>
//                 <option value="closed">Lezárt</option>
//               </Select>
//             </Field>

//             <Field label="Nézet">
//               <Select
//                 value={draft.mine}
//                 onChange={(e) =>
//                   setDraft((p) => ({
//                     ...p,
//                     mine: Number(e.target.value) as any,
//                   }))
//                 }>
//                 <option value={0}>Minden (jogosultság szerint)</option>
//                 <option value={1}>Csak az enyém</option>
//               </Select>
//             </Field>

//             <Field label="Archivált">
//               <Select
//                 value={draft.showArchived ? "1" : "0"}
//                 onChange={(e) =>
//                   setDraft((p) => ({
//                     ...p,
//                     showArchived: e.target.value === "1",
//                   }))
//                 }>
//                 <option value="0">Csak aktív</option>
//                 <option value="1">Archiváltal együtt</option>
//               </Select>
//             </Field>

//             <Field label="Típus (projekt/folyamatos)">
//               <Select
//                 value={draft.flowFilter}
//                 onChange={(e) =>
//                   setDraft((p) => ({ ...p, flowFilter: e.target.value as any }))
//                 }>
//                 <option value="all">Összes</option>
//                 <option value="transient">Projekt</option>
//                 <option value="continuous">Folyamatos</option>
//               </Select>
//             </Field>

//             <Field label="Ügytípus (case_type) tartalmazza">
//               <TextInput
//                 value={draft.caseTypeFilter}
//                 onChange={(e) =>
//                   setDraft((p) => ({ ...p, caseTypeFilter: e.target.value }))
//                 }
//                 placeholder="pl. panasz, ingatlan, klinikai…"
//               />
//             </Field>

//             <Field label="Forgatókönyv">
//               <Select
//                 value={draft.playbookFilter}
//                 onChange={(e) =>
//                   setDraft((p) => ({ ...p, playbookFilter: e.target.value }))
//                 }>
//                 <option value="all">Összes</option>
//                 {playbooks.map((p) => (
//                   <option key={p.id} value={p.id}>
//                     {p.name}
//                   </option>
//                 ))}
//               </Select>
//             </Field>
//           </div>

//           <ToolbarRow>
//             <Button
//               onClick={() => {
//                 void applyFilters(draft, { closeDrawer: true });
//               }}
//               disabled={loading}>
//               {loading ? "..." : "Alkalmaz"}
//             </Button>
//             <SecondaryButton
//               onClick={() => {
//                 const clean: DashboardFilters = {
//                   q: "",
//                   status: "all",
//                   mine: 1,
//                   showArchived: false,
//                   flowFilter: "all",
//                   caseTypeFilter: "",
//                   playbookFilter: "all",
//                 };
//                 setDraft(clean);
//                 void applyFilters(clean, { closeDrawer: true });
//               }}>
//               Szűrők törlése
//             </SecondaryButton>
//           </ToolbarRow>

//           {/* <div className="text-xs opacity-70">
//             Tipp: az “Overdue” nézet a határidős ügyeket kliensoldalon szűri
//             (nem kell backend paraméter).
//           </div> */}
//         </div>
//       </Drawer>

//       <Divider />

//       {/* Kanban */}
//       {/* <SectionTitle>Kanban</SectionTitle> */}

//       {/* <div className="flex items-center gap-2">
//         <SecondaryButton
//           onClick={() => {
//             setViewMode("kanban");
//             void ApiCall.me.saveDashboardCases({
//               ...getCurrentFilters(),
//               savedView,
//               viewMode: "kanban",
//             } as any);
//           }}>
//           Kártyák
//         </SecondaryButton>

//         <SecondaryButton
//           onClick={() => {
//             setViewMode("table");
//             void ApiCall.me.saveDashboardCases({
//               ...getCurrentFilters(),
//               savedView,
//               viewMode: "table",
//             } as any);
//           }}>
//           Táblázat
//         </SecondaryButton>
//       </div>
//       <Divider /> */}

//       {/* Cases list */}
//       {/* <SectionTitle>
//         {viewMode === "kanban" ? "Kanban" : "Táblázat"}
//       </SectionTitle> */}

//       {items.length ? (
//         <>
//           {viewMode === "kanban" ? (
//             <DndContext
//               sensors={sensors}
//               collisionDetection={closestCorners}
//               onDragStart={onDragStart}
//               onDragOver={onDragOver}
//               onDragEnd={onDragEndWithPersist}>
//               <div
//                 className="grid gap-3"
//                 style={{
//                   gridTemplateColumns: "repeat(auto-fit, minmax(280px, 1fr))",
//                   alignItems: "start",
//                 }}>
//                 {STATUS_COLUMNS.map((col) => {
//                   const list = board[col.key] ?? [];
//                   return (
//                     <div key={col.key}>
//                       <Card>
//                         <div
//                           style={{
//                             display: "flex",
//                             justifyContent: "space-between",
//                             gap: 12,
//                             alignItems: "center",
//                             marginBottom: 10,
//                           }}>
//                           <div>
//                             <div className="text-sm font-extrabold">
//                               {col.label}
//                             </div>
//                             {col.hint ? (
//                               <div className="text-xs opacity-60">
//                                 {col.hint}
//                               </div>
//                             ) : null}
//                           </div>
//                           <Chip tone={statusTone(col.key) as any} dot={false}>
//                             {list.length}
//                           </Chip>
//                         </div>

//                         <DroppableColumn id={col.key}>
//                           <SortableContext
//                             items={list.map((x) => x.id)}
//                             strategy={verticalListSortingStrategy}>
//                             <div className="grid gap-2">
//                               {list.length ? (
//                                 list.map((c) => (
//                                   <SortableCaseCard
//                                     key={c.id}
//                                     item={c}
//                                     onOpen={() => openCase(c)}
//                                     t={t}
//                                   />
//                                 ))
//                               ) : (
//                                 <div className="text-xs opacity-70">
//                                   Húzz ide egy kártyát…
//                                 </div>
//                               )}
//                             </div>
//                           </SortableContext>
//                         </DroppableColumn>

//                         {activeId ? (
//                           <div className="mt-3 text-xs opacity-60">
//                             Tipp: húzd át másik oszlopba a státuszváltáshoz.
//                           </div>
//                         ) : null}
//                       </Card>
//                     </div>
//                   );
//                 })}
//               </div>
//             </DndContext>
//           ) : (
//             <CasesDataTable
//               items={items}
//               onOpen={openCase}
//               loading={loading}
//               t={t}
//             />
//             // <CasesTable items={items} onOpen={openCase} />
//           )}
//         </>
//       ) : (
//         <EmptyState
//           title={loading ? "Betöltés…" : "Nincs ügy"}
//           hint="Próbálj szűrőket módosítani, vagy hozz létre egy újat."
//         />
//       )}

//       <Divider />

//       {/* Quick Create (as drawer, not permanent block) */}
//       <Drawer
//         open={createOpen}
//         onClose={() => setCreateOpen(false)}
//         title="Új ügy (gyors)">
//         <div className="grid gap-3">
//           <Field label="Cím" hint="kötelező">
//             <TextInput
//               value={newTitle}
//               onChange={(e) => setNewTitle(e.target.value)}
//               placeholder="Ügy címe"
//             />
//           </Field>

//           <Field label="Leírás" hint="opcionális">
//             <TextInput
//               value={newDesc}
//               onChange={(e) => setNewDesc(e.target.value)}
//               placeholder="Rövid leírás"
//             />
//           </Field>

//           <div className="grid gap-3 md:grid-cols-2">
//             <Field label="Határidő (opcionális)">
//               <input
//                 type="datetime-local"
//                 value={newDueAt}
//                 onChange={(e) => setNewDueAt(e.target.value)}
//                 className="w-full rounded-xl border border-slate-200 bg-white px-3 py-2 text-sm shadow-sm"
//               />
//             </Field>

//             <Field label="Prioritás">
//               <Select
//                 value={newPriority}
//                 onChange={(e) => setNewPriority(Number(e.target.value))}>
//                 {[1, 2, 3, 4, 5].map((p) => (
//                   <option key={p} value={p}>
//                     {p}
//                   </option>
//                 ))}
//               </Select>
//             </Field>
//           </div>

//           <Divider />

//           <div className="grid gap-3 md:grid-cols-2">
//             <Field label="Ügytípus (case_type)" hint="címke, nem workflow">
//               <TextInput
//                 value={newCaseType}
//                 onChange={(e) => setNewCaseType(e.target.value)}
//                 placeholder="pl. panasz, klinikai, ingatlan…"
//               />
//             </Field>

//             <Field label="Jelleg (flow_type)">
//               <Select
//                 value={newFlowType}
//                 onChange={(e) =>
//                   setNewFlowType(e.target.value as CaseFlowType)
//                 }>
//                 <option value="transient">Projekt</option>
//                 <option value="continuous">Folyamatos</option>
//               </Select>
//             </Field>
//           </div>

//           <Field label="Forgatókönyv" hint="irányt ad, nem kényszerít">
//             <Select
//               value={newPlaybookId}
//               onChange={(e) => setNewPlaybookId(e.target.value)}>
//               <option value="">(nincs forgatókönyv)</option>
//               {playbooks.map((p) => (
//                 <option key={p.id} value={p.id}>
//                   {p.name}
//                 </option>
//               ))}
//             </Select>
//           </Field>

//           <ToolbarRow>
//             <Button
//               onClick={() => void onCreate()}
//               disabled={creating || !newTitle.trim()}>
//               {creating ? "..." : "Létrehoz"}
//             </Button>
//             <SecondaryButton
//               onClick={() => setCreateOpen(false)}
//               disabled={creating}>
//               Mégse
//             </SecondaryButton>
//           </ToolbarRow>

//           <div className="text-xs opacity-70">
//             Tipp: a forgatókönyv csak “guideline” — a valós munka a feladatok +
//             meetingek + döntések.
//           </div>
//         </div>
//       </Drawer>
//     </Page>
//   );
// }
