"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import dynamic from "next/dynamic";
import toast from "react-hot-toast";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";

const ExpandedDocTemplateKind = dynamic(
  () => import("@/components/document-template-kinds/ExpandedDocTemplateKind"),
  {
    ssr: false,
    loading: () => <div className="p-3 text-sm text-gray-500">Betöltés…</div>,
  }
);

export type DocTemplateKindRow = {
  id: string;
  code: string;
  label: string;
  description: string | null;
  is_active: boolean;
  sort_order: number;
  created_at?: string | null;
  updated_at?: string | null;
};

export default function DocTemplateKindListPage() {
  const router = useRouter();
  const [rows, setRows] = useState<DocTemplateKindRow[]>([]);
  const [loading, setLoading] = useState(true);

  const [search, setSearch] = useState("");
  const [onlyActive, setOnlyActive] = useState<"all" | "yes" | "no">("all");
  const [expandedRow, setExpandedRow] = useState<DocTemplateKindRow | null>(
    null
  );
  const [selectedIds, setSelectedIds] = useState<string[]>([]);
  const [clearSelectionToggle, setClearSelectionToggle] = useState(false);

  const fetchData = useCallback(async () => {
    setLoading(true);
    try {
      const res = await ApiCall.docTemplateKinds.getList({
        search: search.trim() || undefined,
        onlyActive,
      });
      setRows(res.data ?? []);
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Sablon típus lista betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  }, [search, onlyActive]);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  const ellipsis = (text?: string | null) => {
    const v = String(text ?? "");
    return (
      <div
        title={v}
        style={{
          overflow: "hidden",
          whiteSpace: "nowrap",
          textOverflow: "ellipsis",
        }}>
        {v}
      </div>
    );
  };
  const handleSelectedRowsChange = (state: {
    selectedRows: DocTemplateKindRow[];
  }) => {
    setSelectedIds(state.selectedRows.map((r) => r.id));
  };
  const columns = useMemo(
    () => [
      {
        id: "code",
        name: "Kód",
        selector: (r: DocTemplateKindRow) => r.code,
        cell: (r: DocTemplateKindRow) => ellipsis(r.code),
        sortable: true,
        width: "160px",
      },
      {
        id: "label",
        name: "Megnevezés",
        selector: (r: DocTemplateKindRow) => r.label,
        cell: (r: DocTemplateKindRow) => ellipsis(r.label),
        sortable: true,
        width: "220px",
      },
      {
        id: "description",
        name: "Leírás",
        selector: (r: DocTemplateKindRow) => r.description ?? "",
        cell: (r: DocTemplateKindRow) => ellipsis(r.description),
        sortable: false,
      },
      {
        id: "sort_order",
        name: "Sorrend",
        selector: (r: DocTemplateKindRow) => r.sort_order,
        cell: (r: DocTemplateKindRow) => r.sort_order,
        sortable: true,
        width: "100px",
      },
      {
        id: "is_active",
        name: "Aktív",
        selector: (r: DocTemplateKindRow) => (r.is_active ? "Igen" : "Nem"),
        cell: (r: DocTemplateKindRow) => (
          <span className={r.is_active ? "text-emerald-600" : "text-gray-400"}>
            {r.is_active ? "Igen" : "Nem"}
          </span>
        ),
        sortable: true,
        width: "100px",
      },
      {
        id: "updated_at",
        name: "Módosítva",
        selector: (r: DocTemplateKindRow) => r.updated_at ?? "",
        cell: (r: DocTemplateKindRow) => (
          <div style={{ textAlign: "right" }}>
            {ellipsis(r.updated_at ?? "")}
          </div>
        ),
        sortable: true,
        width: "180px",
      },
    ],
    []
  );

  return (
    <main className="p-4 space-y-4">
      <div className="flex flex-wrap gap-2 items-center">
        <input
          className="border rounded px-3 py-2"
          placeholder="Keresés (kód, megnevezés)"
          value={search}
          onChange={(e) => setSearch(e.target.value)}
        />

        <select
          className="border rounded px-3 py-2"
          value={onlyActive}
          onChange={(e) => setOnlyActive(e.target.value as any)}>
          <option value="all">Aktív: mind</option>
          <option value="yes">Csak aktív</option>
          <option value="no">Csak inaktív</option>
        </select>

        <div className="flex gap-2 ml-auto">
          {selectedIds.length > 0 && (
            <button
              onClick={async () => {
                if (
                  !confirm(
                    `Biztosan törölni szeretnéd a kijelölt ${selectedIds.length} típust?`
                  )
                ) {
                  return;
                }

                try {
                  const res = await ApiCall.docTemplateKinds.removeMany(
                    selectedIds
                  );
                  const { deleted, blocked, error } = res.data || {};
                  if (deleted?.length) {
                    toast.success(`Törölve: ${deleted.length} sablon típus`);
                  }
                  if (blocked?.length) {
                    // prefer backend message if present
                    toast.error(
                      error ||
                        `Nem törölhető: ${blocked.length} típus (még használatban vannak)`
                    );
                  }
                  setSelectedIds([]);
                  setClearSelectionToggle((prev) => !prev);
                  fetchData();
                } catch (e: any) {
                  console.error(e);
                  toast.error(
                    e?.response?.data?.error ?? "Tömeges törlés sikertelen"
                  );
                }
              }}
              className="px-3 py-2 bg-red-600/80 hover:bg-red-600 text-white rounded">
              Kijelöltek törlése ({selectedIds.length})
            </button>
          )}
          <button
            onClick={() => router.push("/admin/document-template-kinds/create")}
            className="px-3 py-2 bg-emerald-600/80 hover:bg-emerald-600 text-white rounded">
            Új sablon típus
          </button>
        </div>
      </div>

      <DataTableWrapper
        keyField="id"
        tableTitle="Dokumentum sablon típusok"
        data={rows}
        columns={columns as any}
        progressPending={loading}
        pagination
        paginationPerPage={25}
        paginationRowsPerPageOptions={[25, 50, 100]}
        defaultSortFieldId="sort_order"
        expandableRows
        expandableRowExpanded={(r: DocTemplateKindRow) =>
          r?.id === expandedRow?.id
        }
        onRowExpandToggled={(open: boolean, row: DocTemplateKindRow) => {
          if (open) {
            setExpandedRow(row);
          } else {
            setExpandedRow(null);
          }
        }}
        renderExpandableRows={(row: DocTemplateKindRow) => (
          <ExpandedDocTemplateKind
            id={row.id}
            onSaved={fetchData}
            onDeleted={fetchData}
          />
        )}
        selectableRows
        onSelectedRowsChange={handleSelectedRowsChange}
        clearSelectedRows={clearSelectionToggle}
      />
    </main>
  );
}
