"use client";

import { useCallback, useEffect, useMemo, useState, useRef } from "react";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import Loader from "@/components/shared/loader/global-loading";
import { TextField } from "@mui/material";
import Autocomplete from "@mui/material/Autocomplete";
import toast from "react-hot-toast";
import { Pencil, Trash2 } from "lucide-react";
import Link from "next/link";
import useTableDebounce from "@/hooks/useTableDebounce";
import ExpandedDocumentTypeRow from "@/components/documents/document-types/ExpandedDocumentTypeRow";

const DocumentTypesList = () => {
  const [data, setData] = useState([]);
  const [page, setPage] = useState(1);
  const [loading, setLoading] = useState(true);
  const [pageSize, setPageSize] = useState(25);
  const [sortField, setSortField] = useState("name");
  const [sortDirection, setSortDirection] = useState("asc");
  const [totalRecords, setTotalRecords] = useState(0);
  const [inputValue, setInputValue] = useState("");
  const [selectedRows, setSelectedRows] = useState([]);
  const [toggleCleared, setToggleCleared] = useState(false);
  const debouncedSearchTerm = useTableDebounce(inputValue, 500);
  const [showIdColumn, setShowColumnId] = useState(false);
  const [expandedRow, setExpandedRow] = useState(null);
  const expandedRowRef = useRef(null);
  const query = useMemo(() => {
    return `?page=${page}&pageSize=${pageSize}&sortField=${sortField}&sortDirection=${sortDirection}&searchTerm=${debouncedSearchTerm.trim()}`;
  }, [page, pageSize, sortField, sortDirection, debouncedSearchTerm]);

  const fetchData = async () => {
    setLoading(true);
    try {
      const response = await ApiCall.documents.getDocumentTypesList(query);

      setData(response?.data?.tableData ?? []);
      setTotalRecords(response?.data?.totalRecords ?? 0);
    } catch (err) {
      console.error(err);
    }
    setLoading(false);
  };

  useEffect(() => {
    fetchData();
  }, [query]);

  const handleSelectAction = async (actionType, rowsOrRow) => {
    const selected = Array.isArray(rowsOrRow) ? rowsOrRow : [rowsOrRow];
    const ids = selected.map((r) => r.id);

    if (actionType === "delete") {
      const confirmDelete = confirm(`Confirm delete:\n${ids.join(", ")}`);
      if (confirmDelete) {
        try {
          await ApiCall.documents.deleteDocumentType(ids);
          fetchData();
          setToggleCleared(!toggleCleared);
          toast.success("Deleted successfully");
        } catch (err) {
          console.error("Delete failed:", err);
          toast.error("Delete failed");
        }
      }
    }
  };
  const customStyles = {
    rows: {
      style: {
        fontSize: "11px !important", // Change row font size
        minHeight: "48px !important", // Optional: row height
      },
    },
    headCells: {
      style: {
        fontSize: "12px !important", // Header font size
        fontWeight: "bold !important",
        backgroundColor: "#f9f9f9 !important",
      },
    },
    cells: {
      style: {
        fontSize: "0.73rem !important", // Body cell font size
      },
    },
  };

  const columns = useMemo(
    () => [
      ...(showIdColumn
        ? [
            {
              id: "_id",
              name: "ID",
              selector: (row) => row._id,
              sortable: true,
            },
          ]
        : []),
      {
        id: "display_name",
        name: "Dokumentum típus neve",
        selector: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.display_name}>
            {row.display_name}
          </div>
        ),
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.display_name}>
            {row.display_name}
          </div>
        ),
        sortable: true,
      },
      {
        id: "description",
        name: "Dokumentum típus leírás",
        selector: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.description}>
            {row.description}
          </div>
        ),
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.description}>
            {row.description}
          </div>
        ),
        sortable: true,
      },
      {
        id: "created_at",
        name: "Létrehozva",
        selector: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.created_at}>
            {row.created_at}
          </div>
        ),
        sortable: true,
      },
      {
        id: "updated_at",
        name: "Módosítva",
        selector: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.updated_at}>
            {row.updated_at}
          </div>
        ),
        sortable: true,
      },
      {
        name: "Műveletek",
        cell: (row) => (
          <div className="flex space-x-2">
            <button
              onClick={() => handleSelectAction("delete", row)}
              className="rounded bg-red-500/75 px-3 py-1 text-sm text-white hover:bg-red-600">
              <Trash2 className="w-4 h-4" />
            </button>
          </div>
        ),
      },
    ],
    []
  );

  const subHeaderComponent = (
    <div className="flex flex-wrap w-full justify-between items-center gap-4">
      <Autocomplete
        freeSolo
        id="subHeaderSearch"
        className="w-72"
        disableClearable
        options={[]}
        inputValue={inputValue}
        onInputChange={(event, newInputValue) => {
          setInputValue(newInputValue);
        }}
        renderInput={(params) => (
          <TextField
            {...params}
            label="Keresés"
            type="search"
            variant="outlined"
          />
        )}
      />
    </div>
  );

  const renderExpandedGroup = useCallback(
    (data, onRowUpdated) => {
      return (
        <div ref={expandedRowRef}>
          <ExpandedDocumentTypeRow id={data?.id} onRowUpdated={onRowUpdated} />
        </div>
      );
    },
    [] // Proper dependencies
  );

  const handleRowUpdated = () => {
    fetchData(); // Refetches updated data, including updated image
  };

  return (
    <DataTableWrapper
      tableTitle="Dokumentum típusok"
      data={data}
      columns={columns}
      customStyles={customStyles}
      progressPending={loading}
      paginationDefaultPage={page}
      defaultSortFieldId="name"
      paginationServer
      onChangePage={(pageNumber) => setPage(pageNumber)}
      progressComponent={<Loader />}
      paginationTotalRows={totalRecords}
      paginationPerPage={pageSize}
      onSort={(col, dir) => {
        setSortField(col.id);
        setSortDirection(dir);
        setPage(1);
      }}
      onChangeRowsPerPage={(size) => {
        setPageSize(size);
        setPage(1);
      }}
      onFilterClear={fetchData}
      subHeader
      subHeaderComponent={subHeaderComponent}
      subHeaderAlign="right"
      subHeaderWrap
      tableOptionsPath="document-types/create"
      selectableRows={true}
      onRowSelected={setSelectedRows}
      toggleCleared={toggleCleared}
      expandableRows
      expandableRowExpanded={(rowData) => rowData?.id === expandedRow?.id}
      onRowExpandToggled={(toggleState, rowIndex) => {
        if (toggleState) {
          // A row is being expanded, so close previously expanded row
          if (expandedRow !== null && expandedRow._id !== rowIndex._id) {
            setExpandedRow(null);
          }
          setExpandedRow(rowIndex);
        } else {
          // A row is being collapsed
          setExpandedRow(null);
        }
      }}
      renderExpandableRows={(rowData) =>
        renderExpandedGroup(rowData, handleRowUpdated)
      }
    />
  );
};

export default DocumentTypesList;
