// app/admin/email-registry/create/page.tsx
"use client";

import { useRouter } from "next/navigation";
import { useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import EmailRegistryForm, {
  EmailUpsertPayload,
} from "@/components/email-registry/EmailRegistryForm";

export default function EmailRegistryCreatePage() {
  const router = useRouter();

  // 🔹 Local UI state for cPanel creation
  const [cpanelCreate, setCpanelCreate] = useState(false);
  const [cpanelPassword, setCpanelPassword] = useState("");
  const [cpanelQuotaMb, setCpanelQuotaMb] = useState<number>(1024); // default 1GB
  const [submitting, setSubmitting] = useState(false);

  const handleCreate = async (payload: EmailUpsertPayload) => {
    if (submitting) return;
    setSubmitting(true);

    try {
      // 1) Create registry row (Postgres / company metadata)
      const res = await ApiCall.emailRegistry.create(payload);

      if (res?.status !== 201) {
        toast.error(res?.data?.error ?? "Sikertelen létrehozás");
        return;
      }

      const email = (payload.email ?? "").trim().toLowerCase();
      const [localPart, domain] = email.split("@");

      if (!localPart || !domain) {
        toast.success("Létrehozva (csak nyilvántartásban – érvénytelen cím)");
        router.push("/admin/email-registry/list");
        return;
      }

      // 2) If user does NOT want cPanel mailbox → done
      if (!cpanelCreate) {
        toast.success("Létrehozva (csak nyilvántartásban)");
        router.push("/admin/email-registry/list");
        return;
      }

      // 3) Validate cPanel settings
      if (!cpanelPassword || cpanelPassword.trim().length < 8) {
        toast.error(
          "Adj meg legalább 8 karakteres jelszót a cPanel postafiókhoz."
        );
        return;
      }

      const quota =
        Number.isFinite(cpanelQuotaMb) && cpanelQuotaMb > 0
          ? cpanelQuotaMb
          : 1024;

      // 4) Create mailbox in cPanel
      await ApiCall.emailRegistry.createAccount(domain, localPart, {
        password: cpanelPassword.trim(),
        quotaMb: quota,
      });

      // 5) Map registry flags → cPanel suspension / forwarders
      const forwardTo =
        payload.is_forwarded && Array.isArray(payload.forward_to)
          ? payload.forward_to
          : [];

      const outgoingMode =
        payload.send_status === "hold"
          ? "hold"
          : payload.send_status === "suspended"
          ? "suspend"
          : "allow";

      await ApiCall.emailRegistry.updateAccount(domain, localPart, {
        suspendLogin: payload.login_status === "suspended",
        suspendIncoming: payload.receive_status === "suspended",
        outgoingMode,
        forwardTo,
        enableForward: !!payload.is_forwarded,
      });

      toast.success(
        "Nyilvántartás + cPanel postafiók + átirányítások létrehozva"
      );

      // 6) Go back to list
      router.push("/admin/email-registry/list");
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Sikertelen létrehozás");
    } finally {
      setSubmitting(false);
    }
  };

  return (
    <main className="p-4 space-y-4 max-w-3xl">
      <h1 className="text-xl font-semibold">Új e-mail felvétele</h1>

      <div className="space-y-6">
        {/* Registry (DB) form */}
        <EmailRegistryForm onSubmit={handleCreate} />

        {/* cPanel options panel */}
        <section className="rounded border bg-white/70 shadow-sm p-4 space-y-3">
          <div className="flex items-center justify-between">
            <div>
              <h2 className="text-sm font-semibold text-gray-800">
                cPanel postafiók létrehozása
              </h2>
              <p className="text-xs text-gray-500">
                Ha bekapcsolod, a nyilvántartás felvétele után a cPanelen is
                létrejön a postafiók, a megfelelő korlátozásokkal /
                átirányítással.
              </p>
            </div>

            <label className="flex items-center gap-2 text-xs">
              <input
                type="checkbox"
                className="h-4 w-4"
                checked={cpanelCreate}
                onChange={(e) => setCpanelCreate(e.target.checked)}
              />
              <span>cPanel postafiók létrehozása</span>
            </label>
          </div>

          <div
            className={`grid grid-cols-1 md:grid-cols-2 gap-3 transition-opacity ${
              cpanelCreate ? "opacity-100" : "opacity-50 pointer-events-none"
            }`}>
            <div className="flex flex-col">
              <label className="text-xs text-gray-500 mb-1">
                cPanel postafiók jelszó
              </label>
              <input
                type="password"
                className="border rounded px-3 py-2 text-sm"
                placeholder="Legalább 8 karakter…"
                value={cpanelPassword}
                onChange={(e) => setCpanelPassword(e.target.value)}
                autoComplete="new-password"
              />
              <p className="text-[11px] text-gray-400 mt-1">
                A jelszó csak a cPanel felé kerül elküldésre, a nyilvántartásban
                külön titkosítva tárolódik (ha ott is mented).
              </p>
            </div>

            <div className="flex flex-col">
              <label className="text-xs text-gray-500 mb-1">
                Kezdő kvóta (MB)
              </label>
              <input
                type="number"
                min={1}
                className="border rounded px-3 py-2 text-sm"
                value={cpanelQuotaMb}
                onChange={(e) =>
                  setCpanelQuotaMb(
                    e.target.value === "" ? 0 : Number(e.target.value)
                  )
                }
              />
              <p className="text-[11px] text-gray-400 mt-1">
                0 helyett inkább adj meg egy reális kvótát (pl. 1024 MB).
              </p>
            </div>
          </div>

          <p className="text-[11px] text-gray-500">
            A cPanel postafiók létrehozása opcionális. Később bármikor
            létrehozhatod vagy módosíthatod a részleteket az adott e-mail
            szerkesztő oldalán.
          </p>
        </section>
      </div>

      {submitting && (
        <p className="text-xs text-gray-500 mt-2">Mentés folyamatban…</p>
      )}
    </main>
  );
}
