"use client";

import React, { useEffect, useMemo, useRef, useState } from "react";
import ApiCall from "@/api/api-calls";
import type {
  MedicalCenterOption,
  CompanyAddressOption,
  PhoneLineOption,
} from "@/api/routes/email-signature";
import Image from "next/image";

type TemplateId = "classic" | "compact" | "logoTop" | "cardSplit";
type IconPackId = "flatBlue" | "monoDark";
type SocialType = "linkedin" | "facebook" | "instagram" | "youtube" | "x";

type SocialLink = {
  id: string;
  type: SocialType;
  url: string;
};

type IconPack = {
  id: IconPackId;
  label: string;
  icons: {
    phone: string;
    email: string;
    web: string;
    social: Record<SocialType, string>;
  };
};

type ImageMode = "embedded" | "linked";

/** ✅ Logos from /public/logos */
const LOGO_OPTIONS: {
  id: string;
  label: string;
  path: string;
  alt: string;
  widthClassic: number;
  widthTop: number;
  widthSplit: number;
}[] = [
  {
    id: "pmh",
    label: "Prima Medica Egészséghálózat",
    path: "/logos/Logó_álló_PrimaMedicaEgészségHálózat.png",
    alt: "Prima Medica Egészséghálózat logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "allergia",
    label: "Allergiaközpont",
    path: "/logos/Logó_álló_Allergiaközpont.png",
    alt: "Prima Medica Allergiaközpont logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "anyajegy",
    label: "Anyajegyszűrőközpont",
    path: "/logos/Logó_álló_AnyajegyszűrőKözpont.png",
    alt: "Prima Medica Anyajegyszűrőközpont logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "cukorbeteg",
    label: "Cukorbetegközpont",
    path: "/logos/Logó_álló_Cukorbetegközpont.png",
    alt: "Prima Medica Cukorbetegközpont logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "dermatica",
    label: "Dermatica",
    path: "/logos/Logó_álló_Dermatica.png",
    alt: "Prima Medica Dermatica logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "endokrin",
    label: "Endokrinközpont",
    path: "/logos/Logó_álló_Endokrinközpont.png",
    alt: "Prima Medica Endokrinközpont logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "fajdalom",
    label: "Fájdalomközpont",
    path: "/logos/Logó_álló_Fájdalomközpont.png",
    alt: "Prima Medica Fájdalomközpont logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "fulorrgege",
    label: "Fülorrgégeközpont",
    path: "/logos/Logó_álló_Fülorrgégeközpont.png",
    alt: "Prima Medica Fülorrgégeközpont logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "gasztro",
    label: "Gasztroenterológiai Központ",
    path: "/logos/Logó_álló_GasztroenterológiaiKözpont.png",
    alt: "Prima Medica Gasztroenterológiai Központ logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "gyermekgyogyaszat",
    label: "Gyermekgyógyászati Központ",
    path: "/logos/Logó_álló_GyermekgyógyászatiKözpont.png",
    alt: "Prima Medica Gyermekgyógyászati Központ logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "hepatologia",
    label: "Hepatológiai Központ",
    path: "/logos/Logó_álló_HepatológiaiKözpont.png",
    alt: "Prima Medica Hepatológiai Központ logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "immun",
    label: "Immunközpont",
    path: "/logos/Logó_álló_Immunközpont.png",
    alt: "Prima Medica Immunközpont logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "infektologia",
    label: "Infektológiai Központ",
    path: "/logos/Logó_álló_InfektológiaiKözpont.png",
    alt: "Prima Medica Infektológiai Központ logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "joalvas",
    label: "JóAlvás Központ",
    path: "/logos/Logó_álló_JóAlvásKözpont.png",
    alt: "Prima Medica JóAlvás Központ logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "kardio",
    label: "Kardioközpont",
    path: "/logos/Logó_álló_Kardioközpont.png",
    alt: "Prima Medica Kardioközpont logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "neuro",
    label: "Neurológiai Központ",
    path: "/logos/Logó_álló_NeurológiaiKözpont.png",
    alt: "Prima Medica Neurológiai Központ logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "nogyogy",
    label: "Nőgyógyászati Központ",
    path: "/logos/Logó_álló_NőgyógyászatiKözpont.png",
    alt: "Prima Medica Nőgyógyászati Központ logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "poliklinika",
    label: "PrimaMedica Poliklinka",
    path: "/logos/Logó_álló_PrimaMedicaPoliklinka.png",
    alt: "Prima Medica PrimaMedica Poliklinka logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "pszichiatria",
    label: "Pszichiátriai Központ",
    path: "/logos/Logó_álló_PszichiátriaiKözpont.png",
    alt: "Prima Medica Pszichiátriai Központ logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "speed",
    label: "SpeedMedical",
    path: "/logos/Logó_álló_SpeedMedical.png",
    alt: "Prima Medica SpeedMedical logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "sport",
    label: "Sportorvosi Központ",
    path: "/logos/Logó_álló_SportorvosiKözpont.png",
    alt: "Prima Medica Sportorvosi Központ logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "szemeszet",
    label: "Szemészeti Központ",
    path: "/logos/Logó_álló_SzemészetiKözpont.png",
    alt: "Prima Medica Szemészeti Központ logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "trombo",
    label: "Trombózis és Hematológiai Központ",
    path: "/logos/Logó_álló_TrombózisésHematológiaiKözpont.png",
    alt: "Prima Medica Trombózis és Hematológiai Központ logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "tudo",
    label: "Tüdőközpont",
    path: "/logos/Logó_álló_Tüdőközpont.png",
    alt: "Prima Medica Tüdőközpont logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "ultrahang",
    label: "Ultrahangközpont",
    path: "/logos/Logó_álló_Ultrahangközpont.png",
    alt: "Prima Medica Ultrahangközpont logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
  {
    id: "urologia",
    label: "Urológiai Központ",
    path: "/logos/Logó_álló_UrológiaiKözpont.png",
    alt: "Prima Medica Urológiai Központ logó",
    widthClassic: 120,
    widthTop: 150,
    widthSplit: 75,
  },
];

const TEMPLATES: { id: TemplateId; label: string; description: string }[] = [
  {
    id: "classic",
    label: "Klasszikus",
    description: "Logo balra, adatok jobbra",
  },
  { id: "compact", label: "Tömör", description: "Tömörebb, kevesebb sor" },
  {
    id: "logoTop",
    label: "Logó Felül",
    description: "Logo felül, adatok alatta",
  },
  {
    id: "cardSplit",
    label: "Osztott",
    description: "Két hasábos kártya (bal: név+kontakt, jobb: cég+logo+social)",
  },
];

const ICON_PACKS: IconPack[] = [
  {
    id: "flatBlue",
    label: "Flat Blue",
    icons: {
      phone: "/icons/flatBlue/phone.png",
      email: "/icons/flatBlue/email.png",
      web: "/icons/flatBlue/web.png",
      social: {
        linkedin: "/icons/flatBlue/linkedin.png",
        facebook: "/icons/flatBlue/facebook.png",
        instagram: "/icons/flatBlue/instagram.png",
        youtube: "/icons/flatBlue/youtube.png",
        x: "/icons/flatBlue/x.png",
      },
    },
  },
  {
    id: "monoDark",
    label: "Mono Dark",
    icons: {
      phone: "/icons/monoDark/phone.png",
      email: "/icons/monoDark/email.png",
      web: "/icons/monoDark/web.png",
      social: {
        linkedin: "/icons/monoDark/linkedin.png",
        facebook: "/icons/monoDark/facebook.png",
        instagram: "/icons/monoDark/instagram.png",
        youtube: "/icons/monoDark/youtube.png",
        x: "/icons/monoDark/x.png",
      },
    },
  },
];

const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;

const normalizeUrl = (raw: string) => {
  const v = raw.trim();
  if (!v) return "";
  if (!/^https?:\/\//i.test(v)) return `https://${v}`;
  return v;
};

const normalizeWebsite = (raw: string) => {
  const v = raw.trim();
  if (!v) return "";
  return v.replace(/^https?:\/\//i, "");
};

/** fetch same-origin assets and convert to data: url for embedding */
async function fetchAsDataUrl(pathOrUrl: string) {
  const url = /^https?:\/\//i.test(pathOrUrl)
    ? pathOrUrl
    : new URL(pathOrUrl, window.location.origin).toString();

  const res = await fetch(url, { cache: "force-cache" });
  if (!res.ok) throw new Error(`Failed to fetch asset: ${url} (${res.status})`);
  const blob = await res.blob();

  return await new Promise<string>((resolve, reject) => {
    const reader = new FileReader();
    reader.onload = () => resolve(String(reader.result)); // data:...
    reader.onerror = reject;
    reader.readAsDataURL(blob);
  });
}

/** tiny transparent pixel (keeps layout stable if something is missing) */
const TRANSPARENT_PIXEL = "data:image/gif;base64,R0lGODlhAQABAAAAACw=";

async function safeDataUrl(pathOrUrl: string) {
  try {
    return await fetchAsDataUrl(pathOrUrl);
  } catch (e) {
    console.error(e);
    return TRANSPARENT_PIXEL;
  }
}

type ResolvedAssets = {
  logo: {
    src: string;
    alt: string;
    fallbackText: string;
    widthClassic: number;
    widthTop: number;
    widthSplit: number;
  };
  icons: {
    phone: string;
    email: string;
    web: string;
    social: Record<SocialType, string>;
  };
};

function buildSocialRow(assets: ResolvedAssets, socials: SocialLink[]) {
  const cells = socials
    .map((s) => ({ ...s, url: normalizeUrl(s.url) }))
    .filter((s) => !!s.url)
    .map((s) => {
      const src = assets.icons.social[s.type];
      const alt = `Social: ${s.type}`;
      return `
<td style="padding-right:6px;vertical-align:middle;">
  <a href="${s.url}" target="_blank" style="text-decoration:none;">
    <img src="${src}" width="18" height="18" alt="${alt}" style="display:block;border:0;outline:none;text-decoration:none;" />
  </a>
</td>`;
    })
    .join("");

  if (!cells) return "";

  return `
<table cellpadding="0" cellspacing="0" style="margin-top:8px;">
  <tr>${cells}</tr>
</table>`;
}

function iconLine(iconSrc: string, label: string, valueHtml: string) {
  return `
<tr>
  <td style="padding-right:6px;vertical-align:middle;">
    <img src="${iconSrc}" width="14" height="14" alt="${label}" style="display:block;border:0;" />
  </td>
  <td style="vertical-align:middle;">
    <span style="color:#555;">${label}:</span> ${valueHtml}
  </td>
</tr>`;
}

type BuildArgs = {
  assets: ResolvedAssets;
  name: string;
  position: string;
  companyName: string;
  addressLabel: string;
  phoneNumber: string;
  website: string;
  email: string;
  quote: string;
  socialLinks: SocialLink[];
  /** ✅ NEW: background color for the signature (empty => transparent) */
  bgColor: string;
};

const bgTdStyle = (bgColor: string) =>
  bgColor ? `background:${bgColor};` : "";

function buildClassic(args: BuildArgs) {
  const web = normalizeWebsite(args.website);
  const webHref = web ? `https://${web}` : "";
  const socials = buildSocialRow(args.assets, args.socialLinks);

  return `
<table cellpadding="0" cellspacing="0" style="font-family:Arial,sans-serif;font-size:13px;color:#000;line-height:1.25;">
  <tr>
    <td style="${bgTdStyle(args.bgColor)}padding:18px;">
      <table cellpadding="0" cellspacing="0" style="${bgTdStyle(args.bgColor)}">
        <tr>
          <td style="padding-right:15px;vertical-align:top;">
            <img src="${args.assets.logo.src}" alt="${
    args.assets.logo.alt
  }" width="${
    args.assets.logo.widthClassic
  }" style="display:block;border:0;outline:none;text-decoration:none;" />
            <div style="font-size:11px;color:#666;margin-top:6px;">${
              args.assets.logo.fallbackText
            }</div>
          </td>

          <td style="vertical-align:top;">
            <div style="font-size:14px;font-weight:bold;">${
              args.name || ""
            }</div>
            ${
              args.position
                ? `<div style="color:#555;margin-top:2px;">${args.position}</div>`
                : ""
            }

            <div style="margin-top:10px;">
              <div style="font-weight:bold;">${args.companyName || ""}</div>
              ${args.addressLabel ? `<div>${args.addressLabel}</div>` : ""}
            </div>

            <div style="margin-top:10px;">
              <table cellpadding="0" cellspacing="0" style="font-family:Arial,sans-serif;font-size:13px;">
                ${
                  args.phoneNumber
                    ? iconLine(
                        args.assets.icons.phone,
                        "Telefon",
                        `${args.phoneNumber}`
                      )
                    : ""
                }
                ${
                  args.email
                    ? iconLine(
                        args.assets.icons.email,
                        "Email",
                        `<a href="mailto:${args.email}" style="color:#108fcf;text-decoration:none;">${args.email}</a>`
                      )
                    : ""
                }
                ${
                  web
                    ? iconLine(
                        args.assets.icons.web,
                        "Web",
                        `<a href="${webHref}" target="_blank" style="color:#108fcf;text-decoration:none;">${web}</a>`
                      )
                    : ""
                }
              </table>
            </div>

            ${socials}

            ${
              args.quote
                ? `<div style="margin-top:10px;font-style:italic;color:#666;">${args.quote}</div>`
                : ""
            }
          </td>
        </tr>
      </table>
    </td>
  </tr>
</table>`.trim();
}

function buildCompact(args: BuildArgs) {
  const socials = buildSocialRow(args.assets, args.socialLinks);

  return `
<table cellpadding="0" cellspacing="0" style="font-family:Arial,sans-serif;font-size:13px;color:#000;line-height:1.25;${bgTdStyle(
    args.bgColor
  )}">
  <tr>
    <td style="${bgTdStyle(args.bgColor)}padding:18px;">
      <div style="font-weight:bold;">
        ${args.name || ""}${
    args.position
      ? ` <span style="font-weight:normal;color:#555;">— ${args.position}</span>`
      : ""
  }
      </div>
      <div style="margin-top:4px;">
        ${args.companyName || ""}${
    args.addressLabel
      ? ` • <span style="color:#555;">${args.addressLabel}</span>`
      : ""
  }
      </div>

      <div style="margin-top:8px;">
        <table cellpadding="0" cellspacing="0" style="font-family:Arial,sans-serif;font-size:13px;">
          ${
            args.phoneNumber
              ? iconLine(
                  args.assets.icons.phone,
                  "Telefon",
                  `${args.phoneNumber}`
                )
              : ""
          }
          ${
            args.email
              ? iconLine(
                  args.assets.icons.email,
                  "Email",
                  `<a href="mailto:${args.email}" style="color:#108fcf;text-decoration:none;">${args.email}</a>`
                )
              : ""
          }
        </table>
      </div>

      ${socials}

      ${
        args.quote
          ? `<div style="margin-top:8px;font-style:italic;color:#666;">${args.quote}</div>`
          : ""
      }
    </td>
  </tr>
</table>`.trim();
}

function buildLogoTop(args: BuildArgs) {
  const socials = buildSocialRow(args.assets, args.socialLinks);

  return `
<table cellpadding="0" cellspacing="0" style="font-family:Arial,sans-serif;font-size:13px;color:#000;line-height:1.25;">
  <tr>
    <td style="${bgTdStyle(args.bgColor)}padding:18px;">
      <table cellpadding="0" cellspacing="0" style="${bgTdStyle(args.bgColor)}">
        <tr>
          <td style="padding-bottom:10px;">
            <img src="${args.assets.logo.src}" alt="${
    args.assets.logo.alt
  }" width="${args.assets.logo.widthTop}" style="display:block;border:0;" />
            <div style="font-size:11px;color:#666;margin-top:6px;">${
              args.assets.logo.fallbackText
            }</div>
          </td>
        </tr>

        <tr>
          <td>
            <div style="font-size:14px;font-weight:bold;">${
              args.name || ""
            }</div>
            ${
              args.position
                ? `<div style="color:#555;margin-top:2px;">${args.position}</div>`
                : ""
            }

            <div style="margin-top:10px;">
              <div style="font-weight:bold;">${args.companyName || ""}</div>
              ${args.addressLabel ? `<div>${args.addressLabel}</div>` : ""}
            </div>

            ${socials}

            ${
              args.quote
                ? `<div style="margin-top:10px;font-style:italic;color:#666;">${args.quote}</div>`
                : ""
            }
          </td>
        </tr>
      </table>
    </td>
  </tr>
</table>`.trim();
}

/**
 * ✅ FIXED cardSplit alignment:
 * - text in right column starts at top
 * - logo sits in a separate cell on the same top row, so logo PNG padding won't push the text down
 */
function buildCardSplit(args: BuildArgs) {
  const web = normalizeWebsite(args.website);
  const webHref = web ? `https://${web}` : "";
  const socials = buildSocialRow(args.assets, args.socialLinks);

  return `
<table cellpadding="0" cellspacing="0" style="font-family:Arial,sans-serif;font-size:13px;color:#000;line-height:1.25;">
  <tr>
    <td style="${bgTdStyle(args.bgColor)}padding:18px;">
      <table cellpadding="0" cellspacing="0" style="${bgTdStyle(args.bgColor)}">
        <tr>
          <!-- LEFT -->
          <td style="vertical-align:top;padding-right:16px; padding-top: 16px;">
            <div style="font-size:16px;font-weight:bold;">${
              args.name || ""
            }</div>
            ${
              args.position
                ? `<div style="color:#555;margin-top:2px;">${args.position}</div>`
                : ""
            }
            <div style="margin-top:6px;color:#555;">${
              args.companyName || ""
            }</div>

            <div style="margin-top:10px;">
              <table cellpadding="0" cellspacing="0" style="font-family:Arial,sans-serif;font-size:13px;">
                ${
                  args.phoneNumber
                    ? iconLine(
                        args.assets.icons.phone,
                        "Telefon",
                        `${args.phoneNumber}`
                      )
                    : ""
                }
                ${
                  args.email
                    ? iconLine(
                        args.assets.icons.email,
                        "Email",
                        `<a href="mailto:${args.email}" style="color:#108fcf;text-decoration:none;">${args.email}</a>`
                      )
                    : ""
                }
              </table>
            </div>

            ${
              args.quote
                ? `<div style="margin-top:10px;font-style:italic;color:#666;">${args.quote}</div>`
                : ""
            }
          </td>

          <!-- DIVIDER -->
          <td style="width:1px;background:#7aa37b;opacity:0.6;"></td>

          <!-- RIGHT -->
          <td style="vertical-align:top;padding-left:16px;min-width:240px;">
            <!-- top row: text-left, logo-right -->
            <table cellpadding="0" cellspacing="0" style="width:100%;">
              <tr>
                <td style="vertical-align:top;text-align:right;padding-left:12px;">
                  <img
                    src="${args.assets.logo.src}"
                    alt="${args.assets.logo.alt}"
                    width="${args.assets.logo.widthSplit}"
                    style="display:block;border:0;outline:none;text-decoration:none;"
                  />
                  <div style="font-size:11px;color:#666;padding-top:2px;text-align:left;">${
                    args.assets.logo.fallbackText
                  }</div>
                    ${
                      args.addressLabel
                        ? `<div style="color:#333; text-align:left;">${args.addressLabel}</div>`
                        : ""
                    }
                  ${
                    web
                      ? `<div style="padding-top:6px; text-align:left;">
                          <a href="${webHref}" target="_blank" style="color:#108fcf;text-decoration:none;">${web}</a>
                        </div>`
                      : ""
                  }
                </td>
              

              </tr>
            </table>

            <div style="margin-top:10px; padding-left: 12px;">
              ${socials}
            </div>
          </td>
        </tr>
      </table>
    </td>
  </tr>
</table>`.trim();
}

function buildSignature(templateId: TemplateId, args: BuildArgs) {
  switch (templateId) {
    case "compact":
      return buildCompact(args);
    case "logoTop":
      return buildLogoTop(args);
    case "cardSplit":
      return buildCardSplit(args);
    case "classic":
    default:
      return buildClassic(args);
  }
}

export default function EmailSignatureGeneratorPage() {
  const [name, setName] = useState("");
  const [position, setPosition] = useState("");
  const [companyId, setCompanyId] = useState("");
  const [addressId, setAddressId] = useState("");
  const [phoneId, setPhoneId] = useState("");
  const [website, setWebsite] = useState("");
  const [email, setEmail] = useState("");
  const [quote, setQuote] = useState("Ahol jó gyógyulni, ahol jó gyógyítani!");

  const [templateId, setTemplateId] = useState<TemplateId>("classic");
  const [iconPackId, setIconPackId] = useState<IconPackId>("flatBlue");
  const [imageMode, setImageMode] = useState<ImageMode>("embedded");

  /** ✅ logo selector */
  const [logoId, setLogoId] = useState<string>(LOGO_OPTIONS[0]?.id ?? "pmh");

  /** ✅ NEW: background color (empty => transparent) */
  const [bgEnabled, setBgEnabled] = useState(false);
  const [bgColor, setBgColor] = useState("#eef2f3"); // only used when bgEnabled=true

  const [socialEnabled, setSocialEnabled] = useState(false);
  const [socialLinks, setSocialLinks] = useState<SocialLink[]>([]);

  const [centers, setCenters] = useState<MedicalCenterOption[]>([]);
  const [addresses, setAddresses] = useState<CompanyAddressOption[]>([]);
  const [phoneLines, setPhoneLines] = useState<PhoneLineOption[]>([]);

  const [emailError, setEmailError] = useState("");
  const [outputHtml, setOutputHtml] = useState("");
  const [previewHtml, setPreviewHtml] = useState("");
  const previewRef = useRef<HTMLDivElement>(null);

  const [status, setStatus] = useState<{
    type: "ok" | "err" | "info";
    msg: string;
  } | null>(null);
  const [busy, setBusy] = useState(false);

  const embeddedCacheRef = useRef<Map<string, ResolvedAssets>>(new Map());

  useEffect(() => {
    const load = async () => {
      const { data: opt } = await ApiCall.emailSignature.getOptions();
      setCenters(opt.centers ?? []);
      setAddresses(opt.addresses ?? []);
      setPhoneLines(opt.phoneLines ?? []);

      const { data: p } = await ApiCall.emailSignature.getPreset();
      if (p?.preset) {
        const s: any = p.preset;
        setTemplateId(s.templateId);
        setIconPackId(s.iconPackId);
        setName(s.name ?? "");
        setPosition(s.position ?? "");
        setCompanyId(s.companyId ?? "");
        setAddressId(s.addressId ?? "");
        setPhoneId(s.phoneId ?? "");
        setWebsite(s.website ?? "");
        setEmail(s.email ?? "");
        setQuote(s.quote ?? "");
        setSocialEnabled(!!s.socialEnabled);
        setSocialLinks(Array.isArray(s.socialLinks) ? s.socialLinks : []);
        if (s.logoId) setLogoId(String(s.logoId));

        // ✅ load background if present
        if (typeof s.bgColor === "string" && s.bgColor) {
          setBgEnabled(true);
          setBgColor(s.bgColor);
        } else {
          setBgEnabled(false);
        }
      }
    };

    load().catch((e) => {
      console.error(e);
      setStatus({ type: "err", msg: "Nem sikerült betölteni az adatokat." });
    });
  }, []);

  const iconPack = useMemo(
    () => ICON_PACKS.find((p) => p.id === iconPackId) ?? ICON_PACKS[0],
    [iconPackId]
  );

  const selectedTemplate = useMemo(
    () => TEMPLATES.find((t) => t.id === templateId) ?? TEMPLATES[0],
    [templateId]
  );

  const selectedLogo = useMemo(
    () => LOGO_OPTIONS.find((l) => l.id === logoId) ?? LOGO_OPTIONS[0],
    [logoId]
  );

  const canGenerate = useMemo(() => emailRegex.test(email.trim()), [email]);

  const addSocial = () => {
    setSocialLinks((prev) => [
      ...prev,
      { id: crypto.randomUUID(), type: "linkedin", url: "" },
    ]);
  };
  const updateSocial = (id: string, patch: Partial<SocialLink>) => {
    setSocialLinks((prev) =>
      prev.map((s) => (s.id === id ? { ...s, ...patch } : s))
    );
  };
  const removeSocial = (id: string) => {
    setSocialLinks((prev) => prev.filter((s) => s.id !== id));
  };

  const handleSavePreset = async () => {
    try {
      setBusy(true);
      setStatus(null);

      const config = {
        templateId,
        iconPackId,
        logoId,
        bgColor: bgEnabled ? bgColor : "", // ✅ NEW
        name,
        position,
        companyId,
        addressId,
        phoneId,
        website,
        email,
        quote,
        socialEnabled,
        socialLinks,
      };

      await ApiCall.emailSignature.savePreset({ config } as any);
      setStatus({ type: "ok", msg: "Beállítások elmentve." });
    } catch (e) {
      console.error(e);
      setStatus({ type: "err", msg: "Nem sikerült menteni a beállításokat." });
    } finally {
      setBusy(false);
    }
  };

  async function resolveAssets(): Promise<ResolvedAssets> {
    const cacheKey = `${iconPackId}:${imageMode}:${
      selectedLogo?.path ?? "no-logo"
    }`;

    if (imageMode === "linked") {
      return {
        logo: {
          src: selectedLogo?.path ?? "/logos/pmh_logo.png",
          alt: selectedLogo?.alt ?? "Cég logó",
          fallbackText: selectedLogo?.label ?? "Logo",
          widthClassic: selectedLogo?.widthClassic ?? 120,
          widthTop: selectedLogo?.widthTop ?? 150,
          widthSplit: selectedLogo?.widthSplit ?? 110,
        },
        icons: {
          phone: iconPack.icons.phone,
          email: iconPack.icons.email,
          web: iconPack.icons.web,
          social: { ...iconPack.icons.social },
        },
      };
    }

    const cached = embeddedCacheRef.current.get(cacheKey);
    if (cached) return cached;

    const logoSrc = await safeDataUrl(
      selectedLogo?.path ?? "/logos/pmh_logo.png"
    );

    const [phone, em, web, li, fb, ig, yt, x] = await Promise.all([
      safeDataUrl(iconPack.icons.phone),
      safeDataUrl(iconPack.icons.email),
      safeDataUrl(iconPack.icons.web),
      safeDataUrl(iconPack.icons.social.linkedin),
      safeDataUrl(iconPack.icons.social.facebook),
      safeDataUrl(iconPack.icons.social.instagram),
      safeDataUrl(iconPack.icons.social.youtube),
      safeDataUrl(iconPack.icons.social.x),
    ]);

    const resolved: ResolvedAssets = {
      logo: {
        src: logoSrc,
        alt: selectedLogo?.alt ?? "Cég logó",
        fallbackText: selectedLogo?.label ?? "Logo",
        widthClassic: selectedLogo?.widthClassic ?? 120,
        widthTop: selectedLogo?.widthTop ?? 150,
        widthSplit: selectedLogo?.widthSplit ?? 110,
      },
      icons: {
        phone,
        email: em,
        web,
        social: { linkedin: li, facebook: fb, instagram: ig, youtube: yt, x },
      },
    };

    embeddedCacheRef.current.set(cacheKey, resolved);
    return resolved;
  }

  const handleGenerate = async () => {
    const trimmedEmail = email.trim();
    if (!emailRegex.test(trimmedEmail)) {
      setEmailError("Kérlek, adj meg egy érvényes email címet!");
      setStatus({ type: "err", msg: "Hibás email cím." });
      return;
    }

    setEmailError("");
    setStatus(null);

    try {
      setBusy(true);

      const companyName = centers.find((c) => c.id === companyId)?.name ?? "";
      const addressLabel =
        addresses.find((a) => a.id === addressId)?.label ?? "";
      const phoneNumber =
        phoneLines.find((p) => p.id === phoneId)?.phone_number ?? "";

      const assets = await resolveAssets();

      const html = buildSignature(templateId, {
        assets,
        name: name.trim(),
        position: position.trim(),
        companyName,
        addressLabel,
        phoneNumber,
        website: website.trim(),
        email: trimmedEmail,
        quote: quote.trim(),
        socialLinks: socialEnabled ? socialLinks : [],
        bgColor: bgEnabled ? bgColor : "", // ✅ NEW
      });

      setOutputHtml(html);
      setPreviewHtml(html);
      setStatus({ type: "ok", msg: "Aláírás generálva." });
    } catch (e) {
      console.error(e);
      setStatus({
        type: "err",
        msg:
          imageMode === "embedded"
            ? "Nem sikerült beágyazni a képeket. Ellenőrizd, hogy a logó/ikon fájlok elérhetők-e."
            : "Nem sikerült generálni az aláírást.",
      });
    } finally {
      setBusy(false);
    }
  };
  const handleCopy = () => {
    const el = previewRef.current;
    if (!el) return;

    const selection = window.getSelection();
    const range = document.createRange();

    range.selectNodeContents(el);
    selection?.removeAllRanges();
    selection?.addRange(range);

    try {
      document.execCommand("copy");
      setStatus({ type: "ok", msg: "Aláírás kimásolva." });
    } catch (e) {
      console.error(e);
      setStatus({ type: "err", msg: "Nem sikerült a másolás." });
    }

    selection?.removeAllRanges();
  };

  // const handleCopy = async () => {
  //   if (!outputHtml) return;

  //   try {
  //     if (navigator.clipboard && "write" in navigator.clipboard) {
  //       const blob = new Blob([outputHtml], { type: "text/html" });
  //       // @ts-ignore
  //       await navigator.clipboard.write([
  //         new ClipboardItem({ "text/html": blob }),
  //       ]);
  //       setStatus({ type: "ok", msg: "Kimásolva (HTML)." });
  //       return;
  //     }
  //   } catch {}

  //   try {
  //     await navigator.clipboard.writeText(outputHtml);
  //     setStatus({ type: "info", msg: "Kimásolva (szövegként)." });
  //   } catch (e) {
  //     console.error(e);
  //     setStatus({ type: "err", msg: "Nem sikerült a másolás." });
  //   }
  // };

  return (
    <div className="page">
      <span className="topbar">
        <span className="logoHeader">
          <Image
            src={"/images/pmh_logo.svg"}
            alt="PMH logó"
            width={1100}
            height={250}
            className="w-full h-full"
          />
        </span>

        <div className="titleBlock">
          <h1>Email Aláírás Generátor</h1>
          <p className="subtitle">
            Tipp: <b>Beágyazott képek</b> ajánlott, ha nincs publikus elérés.
          </p>
        </div>
      </span>

      <main className="grid">
        <div className="card">
          <h2>Beállítások</h2>

          <div className="row2 pb-4 border-b border-b-gray-300">
            <div>
              <label htmlFor="template">Sablon</label>
              <select
                id="template"
                value={templateId}
                onChange={(e) => setTemplateId(e.target.value as TemplateId)}>
                {TEMPLATES.map((t) => (
                  <option key={t.id} value={t.id}>
                    {t.label}
                  </option>
                ))}
              </select>
              <div className="hint">{selectedTemplate.description}</div>
            </div>

            <div>
              <label htmlFor="iconpack">Ikon csomag</label>
              <select
                id="iconpack"
                value={iconPackId}
                onChange={(e) => setIconPackId(e.target.value as IconPackId)}>
                {ICON_PACKS.map((p) => (
                  <option key={p.id} value={p.id}>
                    {p.label}
                  </option>
                ))}
              </select>
              <div className="iconPreview">
                <img
                  src={iconPack.icons.phone}
                  width={16}
                  height={16}
                  alt="Telefon ikon"
                />
                <img
                  src={iconPack.icons.email}
                  width={16}
                  height={16}
                  alt="Email ikon"
                />
                <img
                  src={iconPack.icons.web}
                  width={16}
                  height={16}
                  alt="Web ikon"
                />
                <img
                  src={iconPack.icons.social.linkedin}
                  width={18}
                  height={18}
                  alt="LinkedIn ikon"
                />
              </div>
            </div>
          </div>

          <div className="row2 pb-4 border-b border-b-gray-300">
            <div>
              <label htmlFor="logoSelect">Logó</label>
              <select
                id="logoSelect"
                value={logoId}
                onChange={(e) => setLogoId(e.target.value)}>
                {LOGO_OPTIONS.map((l) => (
                  <option key={l.id} value={l.id}>
                    {l.label}
                  </option>
                ))}
              </select>
              <div className="hint">
                A kiválasztott logó beágyazva kerül az aláírásba.
              </div>
            </div>

            <div>
              <label>Logó előnézet</label>
              <div className="logoPreviewBox">
                {selectedLogo ? (
                  <img
                    src={selectedLogo.path}
                    alt={selectedLogo.alt}
                    style={{ maxHeight: 44, maxWidth: "100%" }}
                  />
                ) : (
                  <span style={{ color: "#667" }}>Nincs logó</span>
                )}
              </div>
            </div>
          </div>

          {/* ✅ NEW: background picker */}
          <div className="flex flex-wrap">
            <div className="w-full pb-4 border-b border-b-gray-300">
              <label style={{ display: "flex", gap: 8, alignItems: "center" }}>
                Háttérszín (alapból átlátszó)
                <input
                  type="checkbox"
                  className="w-[1.25rem] h-[1.25rem]"
                  checked={bgEnabled}
                  onChange={(e) => setBgEnabled(e.target.checked)}
                />
              </label>

              <div className="bgRow">
                <input
                  type="color"
                  value={bgColor}
                  onChange={(e) => setBgColor(e.target.value)}
                  disabled={!bgEnabled}
                  aria-label="Háttérszín választó"
                />
                <input
                  value={bgEnabled ? bgColor : ""}
                  readOnly
                  placeholder="transparent"
                  className="bgCode"
                />
                <button
                  type="button"
                  className="ghost"
                  onClick={() => {
                    setBgEnabled(false);
                  }}>
                  Átlátszó
                </button>
              </div>

              <div className="hint">
                Ha átlátszó, a levelező kliens saját hátterét használja.
                (Ajánlott)
              </div>
            </div>

            <div className="w-full pb-4 border-b border-b-gray-300">
              <label htmlFor="imagemode">Kép mód</label>
              <select
                id="imagemode"
                value={imageMode}
                onChange={(e) => setImageMode(e.target.value as ImageMode)}>
                <option value="embedded">Beágyazott (ajánlott)</option>
                <option value="linked">Linkelt (csak ha publikus)</option>
              </select>
              <div className="hint">
                {imageMode === "embedded"
                  ? "A logó és ikonok base64-ként kerülnek az HTML-be."
                  : "A képek URL-ként kerülnek az aláírásba (gyakran blokkolja a kliens)."}
              </div>
            </div>
          </div>

          <div className="row2">
            <div>
              <label>Állapot</label>
              <div className="statusBox">
                {status ? (
                  <span className={`status ${status.type}`}>{status.msg}</span>
                ) : (
                  <span className="status neutral">—</span>
                )}
              </div>
            </div>
          </div>
        </div>

        {/* DATA */}
        <div className="card">
          <h2>Adatok</h2>

          <div className="row2">
            <div>
              <label htmlFor="name">Név</label>
              <input
                id="name"
                value={name}
                onChange={(e) => setName(e.target.value)}
              />
            </div>
            <div>
              <label htmlFor="position">Beosztás</label>
              <input
                id="position"
                value={position}
                onChange={(e) => setPosition(e.target.value)}
              />
            </div>
          </div>

          <label htmlFor="company">Cég/Központ neve</label>
          <select
            id="company"
            value={companyId}
            onChange={(e) => setCompanyId(e.target.value)}>
            <option value="">Válassz...</option>
            {centers.map((c) => (
              <option key={c.id} value={c.id}>
                {c.name}
              </option>
            ))}
          </select>

          <label htmlFor="address">Cím</label>
          <select
            id="address"
            value={addressId}
            onChange={(e) => setAddressId(e.target.value)}>
            <option value="">Válassz...</option>
            {addresses.map((a) => (
              <option key={a.id} value={a.id}>
                {a.label}
              </option>
            ))}
          </select>

          <label htmlFor="phone">Telefonszám</label>
          <select
            id="phone"
            value={phoneId}
            onChange={(e) => setPhoneId(e.target.value)}>
            <option value="">Válassz...</option>
            {phoneLines.map((p) => (
              <option key={p.id} value={p.id}>
                {p.phone_number}
              </option>
            ))}
          </select>

          <div className="row2">
            <div>
              <label htmlFor="website">Weboldal</label>
              <input
                id="website"
                placeholder="https nélkül, pl.: www.healzz2.hu"
                value={website}
                onChange={(e) => setWebsite(e.target.value)}
              />
            </div>
            <div>
              <label htmlFor="email">Email</label>
              <input
                id="email"
                type="email"
                value={email}
                onChange={(e) => setEmail(e.target.value)}
              />
              {emailError && (
                <small style={{ color: "red" }}>{emailError}</small>
              )}
            </div>
          </div>

          <label htmlFor="quote">Idézet</label>
          <textarea
            id="quote"
            rows={2}
            value={quote}
            onChange={(e) => setQuote(e.target.value)}
          />

          <div className="socialBlock">
            <label style={{ display: "flex", gap: 8, alignItems: "center" }}>
              <input
                type="checkbox"
                checked={socialEnabled}
                onChange={(e) => setSocialEnabled(e.target.checked)}
              />
              Social linkek
            </label>

            {socialEnabled && (
              <div style={{ marginTop: 10 }}>
                {socialLinks.map((s) => (
                  <div key={s.id} className="socialRow">
                    <select
                      value={s.type}
                      onChange={(e) =>
                        updateSocial(s.id, {
                          type: e.target.value as SocialType,
                        })
                      }>
                      <option value="linkedin">LinkedIn</option>
                      <option value="facebook">Facebook</option>
                      <option value="instagram">Instagram</option>
                      <option value="youtube">YouTube</option>
                      <option value="x">X</option>
                    </select>

                    <input
                      value={s.url}
                      onChange={(e) =>
                        updateSocial(s.id, { url: e.target.value })
                      }
                      placeholder="facebook.com/..."
                    />

                    <button
                      type="button"
                      className="ghost"
                      onClick={() => removeSocial(s.id)}>
                      Törlés
                    </button>
                  </div>
                ))}

                <button type="button" className="ghost" onClick={addSocial}>
                  + Social hozzáadása
                </button>
              </div>
            )}
          </div>
        </div>

        {/* ACTIONS */}
        <div className="actions card">
          <div className="actionRow">
            <button
              onClick={handleGenerate}
              disabled={!canGenerate || busy}
              title={!canGenerate ? "Adj meg érvényes email címet" : ""}>
              {busy ? "Dolgozom..." : "Aláírás generálása"}
            </button>

            <button
              className="secondary"
              type="button"
              onClick={handleSavePreset}
              disabled={busy}>
              Beállítások mentése
            </button>

            <button
              className="secondary"
              type="button"
              onClick={handleCopy}
              disabled={!outputHtml}>
              Másolás
            </button>
          </div>

          <div className="hint">
            A másolt aláírást a Thunderbird, e-mail profilonként elérhető
            aláírás-szerkesztőbe illeszd be (HTML mód bekapcsolás nélkül nem
            működik).
          </div>
        </div>

        {/* OUTPUT */}
        {outputHtml && (
          <div className="card full">
            <h2>Előnézet</h2>
            <div
              className="preview"
              dangerouslySetInnerHTML={{ __html: previewHtml }}
            />

            <h2>Generált HTML</h2>
            <pre className="output">
              <div ref={previewRef}>{outputHtml}</div>
            </pre>
          </div>
        )}
      </main>

      <style jsx>{`
        * {
          box-sizing: border-box;
        }
        .page {
          font-family: sans-serif;
          padding: 20px;
        }

        .topbar {
          display: flex;
          gap: 16px;
          align-items: center;
          margin-bottom: 16px;
        }
        .logoHeader {
          display: flex;
          background-color: rgb(16, 143, 207);
          height: 80px;
          width: 220px;
          align-items: center;
          justify-content: center;
          padding: 8px;
          border-radius: 10px;
        }

        .titleBlock h1 {
          margin: 0;
          color: rgba(16, 143, 207, 1);
        }
        .subtitle {
          margin: 6px 0 0;
          color: #334;
          opacity: 0.8;
          font-size: 0.9rem;
        }

        .grid {
          display: grid;
          gap: 16px;
          grid-template-columns: 1fr;
        }
        .card {
          background: #fff;
          border: 1px solid #e7e7e7;
          border-radius: 12px;
          padding: 14px;
          box-shadow: 0 1px 6px rgba(0, 0, 0, 0.05);
        }

        h2 {
          margin: 0 0 12px;
          color: rgba(16, 143, 207, 1);
          font-size: 1rem;
        }

        label {
          display: block;
          margin-top: 10px;
          font-size: 0.75rem;
          color: rgb(8, 10, 44);
          opacity: 0.7;
        }

        input,
        select,
        textarea {
          width: 100%;
          padding: 8px;
          margin-top: 6px;
          border-radius: 8px;
          border: 1px solid #ddd;
          outline-color: rgba(0, 168, 107, 0.2);
        }

        .row2 {
          display: grid;
          grid-template-columns: 1fr;
          gap: 12px;
        }

        .hint {
          margin-top: 6px;
          font-size: 0.8rem;
          color: #556;
          opacity: 0.85;
        }

        .iconPreview {
          display: flex;
          gap: 10px;
          margin-top: 10px;
          align-items: center;
          opacity: 0.9;
        }

        .logoPreviewBox {
          margin-top: 6px;
          border: 1px solid #eee;
          border-radius: 10px;
          padding: 10px;
          display: flex;
          align-items: center;
          justify-content: center;
          background: #fafafa;
          min-height: 60px;
        }

        .bgRow {
          display: grid;
          grid-template-columns: 60px 1fr 110px;
          gap: 10px;
          align-items: center;
          margin-top: 8px;
        }
        .bgCode {
          font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas,
            "Liberation Mono", "Courier New", monospace;
        }

        .statusBox {
          margin-top: 6px;
        }
        .status {
          display: inline-block;
          padding: 6px 10px;
          border-radius: 999px;
          font-size: 0.85rem;
        }
        .status.ok {
          background: rgba(0, 168, 107, 0.12);
          color: rgb(0, 120, 80);
        }
        .status.err {
          background: rgba(220, 0, 0, 0.1);
          color: rgb(160, 0, 0);
        }
        .status.info {
          background: rgba(16, 143, 207, 0.1);
          color: rgba(16, 143, 207, 1);
        }
        .status.neutral {
          background: #f3f3f3;
          color: #666;
        }

        .actions .actionRow {
          display: grid;
          grid-template-columns: 1fr;
          gap: 10px;
        }

        button {
          width: 100%;
          padding: 10px 14px;
          cursor: pointer;
          border: none;
          border-radius: 10px;
          background-color: rgba(16, 143, 207, 1);
          color: #fff;
          font-weight: 600;
        }
        button:disabled {
          opacity: 0.55;
          cursor: not-allowed;
        }
        button.secondary {
          background: rgba(16, 143, 207, 0.15);
          color: rgba(16, 143, 207, 1);
        }
        button.ghost {
          background: transparent;
          color: rgba(16, 143, 207, 1);
          border: 1px solid rgba(16, 143, 207, 0.35);
          padding: 8px 10px;
        }

        .socialBlock {
          margin-top: 12px;
        }
        .socialRow {
          display: grid;
          grid-template-columns: 140px 1fr 90px;
          gap: 8px;
          margin-bottom: 8px;
          align-items: center;
        }

        .preview {
          padding: 12px;
          border: 1px solid #ddd;
          background: #fff;
          border-radius: 10px;
          overflow: auto;
        }

        .output {
          white-space: pre-wrap;
          font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas,
            "Liberation Mono", "Courier New", monospace;
          background: #f9f9f9;
          padding: 12px;
          border: 1px solid #e2e2e2;
          border-radius: 10px;
          overflow: auto;
        }

        .full {
          grid-column: 1 / -1;
        }

        @media (min-width: 900px) {
          .grid {
            grid-template-columns: repeat(2, minmax(0, 1fr));
          }
          .row2 {
            grid-template-columns: repeat(2, minmax(0, 1fr));
          }
          .actions .actionRow {
            grid-template-columns: repeat(3, minmax(0, 1fr));
          }
        }
      `}</style>
    </div>
  );
}
