// src/app/admin/faqs/list/page.tsx
"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";
import Link from "next/link";
import { slugifyTitle, timestampForFilename } from "@/utils/exportFileSlugify";

import ApiCall from "@/api/api-calls";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";

type Row = {
  id: string; // uuid
  title: string;
  created_at?: string | null;
  updated_at?: string | null;
};

export default function FaqListPage() {
  const router = useRouter();

  const [rows, setRows] = useState<Row[]>([]);
  const [loading, setLoading] = useState(true);
  const [search, setSearch] = useState("");

  const fetchData = useCallback(async () => {
    setLoading(true);
    try {
      const res = await ApiCall.templateDocuments.getList({ kind_code: "faq" });
      if (res.status !== 200) throw new Error("Failed to load FAQs");
      setRows(res.data ?? []);
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "FAQ lista betöltése sikertelen");
    } finally {
      setLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  const filteredRows = useMemo(() => {
    const term = search.trim().toLowerCase();
    if (!term) return rows;
    return rows.filter((r) => r.title.toLowerCase().includes(term));
  }, [rows, search]);

  const handleDownloadDocx = useCallback(async (faq: Row) => {
    try {
      const res = await ApiCall.templateDocuments.download(faq.id);
      const blob = new Blob([res.data], {
        type: "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
      });

      const url = window.URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;

      const slug = slugifyTitle(faq.title);
      const ts = timestampForFilename();
      a.download = `${slug}-${ts}.docx`;

      document.body.appendChild(a);
      a.click();
      a.remove();
      window.URL.revokeObjectURL(url);
    } catch (e) {
      console.error(e);
      toast.error("DOCX letöltés sikertelen");
    }
  }, []);

  const columns = useMemo(
    () => [
      {
        id: "title",
        name: "Cím",
        selector: (r: Row) => r.title,
        cell: (r: Row) => (
          <button
            type="button"
            onClick={() =>
              router.push(`/admin/template-documents/${r.id}/view`)
            }
            className="text-blue-600 hover:underline text-left">
            {r.title}
          </button>
        ),
        sortable: true,
        grow: 2,
      },
      {
        id: "updated_at",
        name: "Módosítva",
        selector: (r: Row) => r.updated_at ?? "",
        cell: (r: Row) => (
          <span className="text-xs text-gray-600">
            {r.updated_at ? new Date(r.updated_at).toLocaleDateString() : "-"}
          </span>
        ),
        sortable: true,
        width: "140px",
      },
      {
        id: "actions",
        name: "Műveletek",
        selector: () => "",
        cell: (r: Row) => (
          <div className="flex justify-end gap-2">
            <button
              type="button"
              onClick={() =>
                router.push(`/admin/template-documents/${r.id}/edit`)
              }
              className="px-2 py-1 text-xs rounded border border-gray-300 hover:bg-gray-50">
              Szerkesztés
            </button>
            <button
              type="button"
              onClick={() => handleDownloadDocx(r)}
              className="px-2 py-1 text-xs rounded border border-gray-300 hover:bg-gray-50">
              DOCX letöltés
            </button>
          </div>
        ),
        ignoreRowClick: true,
        // allowOverflow: true,
        // button: true,
        width: "220px",
      },
    ],
    [router, handleDownloadDocx]
  );

  return (
    <main className="p-4 space-y-4">
      <div className="flex flex-wrap items-center gap-3">
        <div>
          <h1 className="text-lg font-semibold">
            IT FAQ &amp; Security Guidelines
          </h1>
          <p className="text-sm text-gray-600">
            IT hibák megoldásai, biztonsági irányelvek, útmutatók.
          </p>
        </div>

        <div className="flex items-center gap-2 ml-auto">
          <input
            className="border rounded px-3 py-2 text-sm"
            placeholder="Keresés cím alapján…"
            value={search}
            onChange={(e) => setSearch(e.target.value)}
          />

          <Link href="/admin/template-documents/new/edit">
            <button className="px-3 py-2 text-sm rounded bg-emerald-600/80 hover:bg-emerald-600 text-white">
              + Új FAQ
            </button>
          </Link>
        </div>
      </div>

      <DataTableWrapper
        keyField="id"
        tableTitle="FAQ lista"
        data={filteredRows}
        columns={columns as any}
        progressPending={loading}
        pagination
        defaultSortFieldId="title"
      />
    </main>
  );
}
