"use client";

import { useCallback, useEffect, useMemo, useState, useRef } from "react";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import Loader from "@/components/shared/loader/global-loading";
import { TextField } from "@mui/material";
import Autocomplete from "@mui/material/Autocomplete";
import toast from "react-hot-toast";
import { Pencil, Trash2 } from "lucide-react";
import Link from "next/link";
import useTableDebounce from "@/hooks/useTableDebounce";
import ExpandedRow from "@/components/filing/ExpandedRow";
import {
  convertDateString,
  convertGetDateString,
} from "@/utils/converDateString";
import { currencyDisplayHandler } from "@/utils/currencyDisplay";
import Select from "react-select";
import { DatePicker } from "@mui/x-date-pickers";
import { FileDown, Plus } from "lucide-react";

const FilingList = () => {
  const [data, setData] = useState([]);
  const [page, setPage] = useState(1);
  const [loading, setLoading] = useState(true);
  const [pageSize, setPageSize] = useState(25);
  const [sortField, setSortField] = useState("document_arrived_at");
  const [sortDirection, setSortDirection] = useState("desc");
  const [totalRecords, setTotalRecords] = useState(0);
  const [inputValue, setInputValue] = useState("");
  const [selectedRows, setSelectedRows] = useState([]);
  const [toggleCleared, setToggleCleared] = useState(false);
  const debouncedSearchTerm = useTableDebounce(inputValue, 500);
  const [showIdColumn, setShowColumnId] = useState(false);
  const [expandedRow, setExpandedRow] = useState(null);
  const [ownCompanyOptions, setOwnCompanyOptions] = useState([]);
  const [selectedOwnCompanies, setSelectedOwnCompanies] = useState(null);
  const [partnerCompanyOptions, setPartnerCompanyOptions] = useState([]);
  const [selectedPartnerCompanies, setSelectedPartnerCompanies] =
    useState(null);
  const [dateRange, setDateRange] = useState({
    startDate: null,
    endDate: null,
  });

  const expandedRowRef = useRef(null);

  // const query = useMemo(() => {
  //   const params = new URLSearchParams();

  //   params.set("page", page);
  //   params.set("pageSize", pageSize);
  //   params.set("searchTerm", debouncedSearchTerm.trim());
  //   params.set("sortField", sortField);
  //   params.set("sortDirection", sortDirection);

  //   if (selectedOwnCompanies?.length > 0) {
  //     params.set(
  //       "ownCompanyIds",
  //       selectedOwnCompanies.map((c) => c.value).join(",")
  //     );
  //   }

  //   if (selectedPartnerCompanies?.length > 0) {
  //     params.set(
  //       "partnerCompanyIds",
  //       selectedPartnerCompanies.map((c) => c.value).join(",")
  //     );
  //   }

  //   // Example: Add more filters later
  //   if (dateRange.startDate instanceof Date && !isNaN(dateRange.startDate)) {
  //     params.set("startDate", dateRange.startDate.toISOString());
  //   }
  //   if (dateRange.endDate instanceof Date && !isNaN(dateRange.endDate)) {
  //     params.set("endDate", dateRange.endDate.toISOString());
  //   }

  //   return `?${params.toString()}`;
  // }, [
  //   page,
  //   pageSize,
  //   debouncedSearchTerm,
  //   sortField,
  //   sortDirection,
  //   selectedOwnCompanies,
  //   selectedPartnerCompanies,
  //   dateRange,
  // ]);

  const exportQuery = useMemo(() => {
    const params = new URLSearchParams();

    params.set("searchTerm", debouncedSearchTerm.trim());
    params.set("sortField", sortField);
    params.set("sortDirection", sortDirection);

    if (selectedOwnCompanies?.length > 0) {
      params.set(
        "ownCompanyIds",
        selectedOwnCompanies.map((c) => c.value).join(",")
      );
    }

    if (selectedPartnerCompanies?.length > 0) {
      params.set(
        "partnerCompanyIds",
        selectedPartnerCompanies.map((c) => c.value).join(",")
      );
    }

    if (dateRange.startDate && dateRange.startDate.isValid?.()) {
      params.set("startDate", dateRange.startDate.toDate().toISOString());
    }
    if (dateRange.endDate && dateRange.endDate.isValid?.()) {
      params.set("endDate", dateRange.endDate.toDate().toISOString());
    }

    return `?${params.toString()}`;
  }, [
    debouncedSearchTerm,
    sortField,
    sortDirection,
    selectedOwnCompanies,
    selectedPartnerCompanies,
    dateRange,
  ]);

  const fetchData = async () => {
    setLoading(true);
    try {
      const response = await ApiCall.filing.getList(exportQuery);
      const res = await ApiCall.filing.getAllSelectOptions();
      setOwnCompanyOptions(res.data.ownCompanies || []);
      setPartnerCompanyOptions(res.data.partnerCompanies || []);
      console.log("response", response.data.tableData);
      setData(response?.data?.tableData ?? []);
      setTotalRecords(response?.data?.totalRecords ?? 0);
    } catch (err) {
      console.error(err);
    }
    setLoading(false);
  };

  useEffect(() => {
    fetchData();
  }, [exportQuery]);

  const handleClearFilters = () => {
    setInputValue("");
    setSelectedOwnCompanies(null);
    setSelectedPartnerCompanies(null);
    setDateRange({ startDate: null, endDate: null });
    setPage(1);
  };

  const handleSelectAction = async (actionType, rowsOrRow) => {
    const selected = Array.isArray(rowsOrRow) ? rowsOrRow : [rowsOrRow];
    const ids = selected.map((r) => r.id);

    if (actionType === "delete") {
      const confirmDelete = confirm(`Confirm delete:\n${ids.join(", ")}`);
      if (confirmDelete) {
        try {
          await ApiCall.filing.delete({ ids: ids });
          fetchData();
          setToggleCleared(!toggleCleared);
          toast.success("Deleted successfully");
        } catch (err) {
          console.error("Delete failed:", err);
          toast.error("Delete failed");
        }
      }
    }
  };
  const customStyles = {
    rows: {
      style: {
        fontSize: "11px !important", // Change row font size
        minHeight: "48px !important", // Optional: row height
      },
    },
    headCells: {
      style: {
        fontSize: "12px !important", // Header font size
        fontWeight: "bold !important",
        backgroundColor: "#f9f9f9 !important",
      },
    },
    cells: {
      style: {
        fontSize: "0.73rem !important", // Body cell font size
      },
    },
  };

  const columns = useMemo(
    () => [
      ...(showIdColumn
        ? [
            {
              id: "_id",
              name: "ID",
              selector: (row) => row._id,
              sortable: true,
            },
          ]
        : []),

      {
        id: "document_registration_number",
        name: "Iktatószám",
        selector: (row) => row.document_registration_number,
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.document_registration_number}>
            {row.document_registration_number}
          </div>
        ),
        sortable: true,
      },
      {
        id: "invoice_serial_number",
        name: "Számla sorszáma",
        selector: (row) => row.invoice_serial_number,
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.invoice_serial_number}>
            {row.invoice_serial_number}
          </div>
        ),
        sortable: true,
      },
      {
        id: "partner_company_name",
        name: "Szállító cég",
        selector: (row) => row.partner_company_name,
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.partner_company_name}>
            {row.partner_company_name}
          </div>
        ),
        sortable: true,
      },
      {
        id: "own_company_name",
        name: "Saját cég",
        selector: (row) => row.own_company_name,
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.own_company_name}>
            {row.own_company_name}
          </div>
        ),
        sortable: true,
      },
      {
        id: "document_arrived_at",
        name: "Érkeztetve",
        selector: (row) =>
          row.document_arrived_at
            ? convertGetDateString(row.document_arrived_at)
            : "",
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={
              row.document_arrived_at
                ? convertGetDateString(row.document_arrived_at)
                : ""
            }>
            {row.document_arrived_at
              ? convertGetDateString(row.document_arrived_at)
              : ""}
          </div>
        ),
        sortable: true,
      },
      {
        id: "invoice_value",
        name: "Számla értéke",
        selector: (row) => row.invoice_value,
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.invoice_value}>
            {row?.currency_id && row?.currency_code
              ? currencyDisplayHandler(
                  row.invoice_value,
                  "hu-HU",
                  row.currency_code
                )
              : currencyDisplayHandler(row.invoice_value)}
          </div>
        ),
        sortable: true,
      },
      {
        id: "payment_deadline",
        name: "Fizetési határidő",
        selector: (row) =>
          row.payment_deadline
            ? convertGetDateString(row.payment_deadline)
            : "",
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={
              row.payment_deadline
                ? convertGetDateString(row.payment_deadline)
                : ""
            }>
            {row.payment_deadline
              ? convertGetDateString(row.payment_deadline)
              : ""}
          </div>
        ),
        sortable: true,
      },

      {
        name: "Műveletek",
        cell: (row) => (
          <div className="flex space-x-2">
            <button
              onClick={() => handleSelectAction("delete", row)}
              className="rounded bg-red-500/75 px-3 py-1 text-sm text-white hover:bg-red-600">
              <Trash2 className="w-4 h-4" />
            </button>
          </div>
        ),
      },
    ],
    []
  );
  const handleDownload = useCallback(() => {
    const download = async () => {
      try {
        const response = await ApiCall.filing.download(exportQuery);
        const url = window.URL.createObjectURL(new Blob([response.data]));
        const link = document.createElement("a");
        link.href = url;
        link.setAttribute("download", "iktatas_export.xlsx");
        document.body.appendChild(link);
        link.click();
        link.remove();
      } catch (error) {
        console.error("Download failed", error);
        toast.error("Export failed");
      }
    };
    download();
  }, [exportQuery]);

  // const handleDownload = useCallback(() => {
  //   const download = async () => {
  //     try {
  //       const response = await ApiCall.filing.download(query);
  //       const url = window.URL.createObjectURL(new Blob([response.data]));
  //       const link = document.createElement("a");
  //       link.href = url;
  //       link.setAttribute("download", "iktatas_export.xlsx");
  //       document.body.appendChild(link);
  //       link.click();
  //       link.remove();
  //     } catch (error) {
  //       console.error("Download failed", error);
  //       toast.error("Export failed");
  //     }
  //   };
  //   download();
  // }, [query]);

  const tableSelectOptions = [
    {
      text: "Szerkesztés",
      type: "edit",
      className: "bg-blue-500 text-white",
    },
    {
      text: "Törlés",
      type: "delete",
      className: "bg-red-500 text-white",
    },
  ];
  const subHeaderComponent = (
    <div className="flex flex-wrap w-full justify-between items-center gap-4">
      {selectedRows.length > 0 && (
        <div className="flex items-center gap-2 bg-gray-100 px-4 py-2 border rounded">
          <span className="text-sm text-gray-600">
            {selectedRows.length} kiválasztva
          </span>
          {selectedRows.length === 1 && (
            <button
              onClick={() => handleSelectAction("edit", selectedRows)}
              className="rounded px-3 py-1 text-sm bg-blue-500 text-white">
              Szerkesztés
            </button>
          )}
          <button
            onClick={() => handleSelectAction("delete", selectedRows)}
            className="rounded px-3 py-1 text-sm bg-red-500 text-white">
            Törlés
          </button>
        </div>
      )}
      <div className="w-full flex flex-wrap gap-4 items-stretch">
        <div className="flex-1 flex flex-col justify-stretch h-full">
          <Autocomplete
            freeSolo
            id="subHeaderSearch"
            disableClearable
            options={[]}
            inputValue={inputValue}
            onInputChange={(event, newInputValue) =>
              setInputValue(newInputValue)
            }
            renderInput={(params) => (
              <TextField
                {...params}
                label="Keresés"
                type="search"
                variant="outlined"
                className="h-full"
                sx={{ height: "100%" }}
                InputProps={{
                  ...params.InputProps,
                  sx: { height: "100%" },
                }}
                InputLabelProps={{
                  sx: { height: "100%" },
                }}
              />
            )}
          />
        </div>

        <div className="flex-1 flex flex-col justify-stretch h-full">
          <Select
            className="w-full h-full min-w-72"
            menuPosition="fixed"
            isMulti
            options={ownCompanyOptions}
            placeholder="Szűrés saját cég alapján"
            value={selectedOwnCompanies}
            onChange={(selected) => {
              setSelectedOwnCompanies(selected);
              setPage(1);
            }}
            isClearable
            styles={{
              control: (base) => ({
                ...base,
                height: "100%",
                minHeight: "100%",
                padding: "10px 0",
              }),
              valueContainer: (base) => ({
                ...base,
                height: "100%",
              }),
            }}
          />
        </div>

        <div className="flex-1 flex flex-col justify-stretch h-full">
          <Select
            className="w-full h-full min-w-72"
            menuPosition="fixed"
            isMulti
            options={partnerCompanyOptions}
            placeholder="Szűrés partner cég alapján"
            value={selectedPartnerCompanies}
            onChange={(selected) => {
              setSelectedPartnerCompanies(selected);
              setPage(1);
            }}
            isClearable
            styles={{
              control: (base) => ({
                ...base,
                height: "100%",
                minHeight: "100%",
                padding: "10px 0",
              }),
              valueContainer: (base) => ({
                ...base,
                height: "100%",
              }),
            }}
          />
        </div>

        <div className="flex-1 flex flex-col justify-stretch h-full">
          <DatePicker
            label="Kezdő dátum"
            value={dateRange.startDate}
            onChange={(newValue) => {
              setDateRange((prev) => ({ ...prev, startDate: newValue }));
              setPage(1);
            }}
            slotProps={{
              textField: {
                fullWidth: true,
                sx: { height: "100%" },
              },
            }}
          />
        </div>

        <div className="flex-1 flex flex-col justify-stretch h-full">
          <DatePicker
            label="Végdátum"
            value={dateRange.endDate}
            onChange={(newValue) => {
              setDateRange((prev) => ({ ...prev, endDate: newValue }));
              setPage(1);
            }}
            slotProps={{
              textField: {
                fullWidth: true,
                sx: { height: "100%" },
              },
            }}
          />
        </div>
      </div>
      <div className="w-full flex flex-wrap justify-end">
        <button
          className="flex items-center gap-2 px-4 py-2 text-gray-600 hover:bg-yellow-600 hover:text-white rounded cursor-pointer"
          onClick={handleClearFilters}>
          🧹 Szűrők törlése
        </button>
        <button
          className="flex items-center gap-2 px-4 py-2 text-gray-600 hover:bg-green-600 hover:text-white rounded cursor-pointer"
          onClick={handleDownload}>
          <FileDown className="w-4 h-4 fill-gray-600 hover:fill-white" />
          XLS export
        </button>
        <Link
          href="/admin/filing/create"
          className="flex items-center gap-2 px-4 py-2 text-gray-600 hover:bg-blue-600 hover:text-white rounded">
          <Plus className="w-4 h-4 fill-gray-600 hover:fill-white" />
          Új hozzáadása
        </Link>
      </div>
    </div>
  );

  const renderExpandedGroup = useCallback(
    (data, onRowUpdated) => {
      return (
        <div ref={expandedRowRef}>
          <ExpandedRow filingId={data?.id} onRowUpdated={onRowUpdated} />
        </div>
      );
    },
    [] // Proper dependencies
  );

  const handleRowUpdated = () => {
    fetchData(); // Refetches updated data, including updated image
  };

  return (
    <DataTableWrapper
      tableTitle="Iktatás"
      data={data}
      columns={columns}
      customStyles={customStyles}
      progressPending={loading}
      paginationDefaultPage={page}
      defaultSortFieldId="document_arrived_at"
      paginationServer
      onChangePage={(pageNumber) => setPage(pageNumber)}
      progressComponent={<Loader />}
      paginationTotalRows={totalRecords}
      paginationPerPage={pageSize}
      sortServer={true}
      onSort={(col, dir) => {
        setSortField(col.id);
        setSortDirection(dir);
        setPage(1);
      }}
      onChangeRowsPerPage={(size) => {
        setPageSize(size);
        setPage(1);
      }}
      onFilterClear={fetchData}
      subHeader
      subHeaderComponent={subHeaderComponent}
      subHeaderAlign="right"
      subHeaderWrap
      //   tableOptionsPath="filing/create"
      selectableRows={true}
      onRowSelected={setSelectedRows}
      toggleCleared={toggleCleared}
      expandableRows
      expandableRowExpanded={(rowData) => rowData?.id === expandedRow?.id}
      onRowExpandToggled={(toggleState, rowIndex) => {
        if (toggleState) {
          // A row is being expanded, so close previously expanded row
          if (expandedRow !== null && expandedRow._id !== rowIndex._id) {
            setExpandedRow(null);
          }
          setExpandedRow(rowIndex);
        } else {
          // A row is being collapsed
          setExpandedRow(null);
        }
      }}
      renderExpandableRows={(rowData) =>
        renderExpandedGroup(rowData, handleRowUpdated)
      }
    />
  );
};

export default FilingList;
