"use client";

import { useEffect, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";

type TaskCategory = {
  id: string;
  code: string;
  name: string;
  description?: string | null;
  sort_order?: number | null;
  is_active: boolean;
};

export default function HrTaskCategoriesPage() {
  const [rows, setRows] = useState<TaskCategory[]>([]);
  const [loading, setLoading] = useState(false);

  const [editingId, setEditingId] = useState<string | null>(null);
  const [code, setCode] = useState("");
  const [name, setName] = useState("");
  const [description, setDescription] = useState("");
  const [sortOrder, setSortOrder] = useState<string>("");
  const [isActive, setIsActive] = useState(true);

  const resetForm = () => {
    setEditingId(null);
    setCode("");
    setName("");
    setDescription("");
    setSortOrder("");
    setIsActive(true);
  };

  const loadAll = async () => {
    setLoading(true);
    try {
      const res = await ApiCall.hr.getTaskCategories(); // separate admin endpoint if needed
      setRows(res.data?.items ?? []);
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message ||
          "HR feladat kategóriák betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    loadAll();
  }, []);

  const startEdit = (row: TaskCategory) => {
    setEditingId(row.id);
    setCode(row.code);
    setName(row.name);
    setDescription(row.description ?? "");
    setSortOrder(
      row.sort_order !== null && row.sort_order !== undefined
        ? String(row.sort_order)
        : ""
    );
    setIsActive(row.is_active);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!code.trim() || !name.trim()) {
      toast.error("Kód és megnevezés kötelező");
      return;
    }

    try {
      await ApiCall.hr.upsertTaskCategory({
        id: editingId ?? undefined,
        code: code.trim(),
        name: name.trim(),
        description: description.trim() || null,
        sort_order: sortOrder ? Number(sortOrder) : null,
        is_active: isActive,
      });
      toast.success(editingId ? "Kategória frissítve" : "Kategória létrehozva");
      resetForm();
      await loadAll();
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message ||
          "HR feladat kategória mentése sikertelen"
      );
    }
  };

  const handleDelete = async (row: TaskCategory) => {
    if (!confirm(`Biztosan törlöd a kategóriát? (${row.name})`)) return;
    try {
      await ApiCall.hr.deleteTaskCategory(row.id);
      toast.success("Kategória törölve");
      if (editingId === row.id) resetForm();
      await loadAll();
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message ||
          "HR feladat kategória törlése sikertelen"
      );
    }
  };

  const columns = [
    {
      id: "code",
      name: "Kód",
      selector: (r: TaskCategory) => r.code,
      sortable: true,
      width: "120px",
    },
    {
      id: "name",
      name: "Megnevezés",
      selector: (r: TaskCategory) => r.name,
      sortable: true,
      grow: 2,
    },
    {
      id: "sort_order",
      name: "Sorrend",
      selector: (r: TaskCategory) => r.sort_order ?? 0,
      cell: (r: TaskCategory) =>
        r.sort_order !== null && r.sort_order !== undefined
          ? r.sort_order
          : "—",
      sortable: true,
      width: "100px",
    },
    {
      id: "is_active",
      name: "Aktív?",
      selector: (r: TaskCategory) => (r.is_active ? "Igen" : "Nem"),
      cell: (r: TaskCategory) => (r.is_active ? "✅" : "—"),
      width: "80px",
    },
    {
      id: "actions",
      name: "Művelet",
      cell: (r: TaskCategory) => (
        <div className="flex gap-2">
          <button
            type="button"
            className="px-2 py-1 text-xs rounded bg-blue-600 text-white"
            onClick={() => startEdit(r)}>
            Szerkesztés
          </button>
          <button
            type="button"
            className="px-2 py-1 text-xs rounded bg-red-600 text-white"
            onClick={() => handleDelete(r)}>
            Törlés
          </button>
        </div>
      ),
      width: "180px",
    },
  ];

  return (
    <main className="p-4 space-y-4">
      <h1 className="text-lg font-semibold">HR – Feladat kategóriák</h1>

      <DataTableWrapper
        tableTitle="Kategóriák"
        data={rows}
        columns={columns as any}
        progressPending={loading}
        pagination
        keyField="id"
      />

      <section className="mt-4 border rounded p-4 bg-gray-50 space-y-3">
        <div className="flex justify-between items-center">
          <h2 className="font-semibold text-sm">
            {editingId ? "Kategória szerkesztése" : "Új kategória létrehozása"}
          </h2>
          {editingId && (
            <button
              type="button"
              className="text-xs text-gray-500 underline"
              onClick={resetForm}>
              Új kategória mód
            </button>
          )}
        </div>

        <form
          onSubmit={handleSubmit}
          className="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">
          <div>
            <label className="block text-xs font-semibold mb-1">Kód</label>
            <input
              className="border rounded px-2 py-1 w-full"
              value={code}
              onChange={(e) => setCode(e.target.value)}
              placeholder="pl. onboarding, offboarding"
            />
          </div>
          <div>
            <label className="block text-xs font-semibold mb-1">
              Megnevezés
            </label>
            <input
              className="border rounded px-2 py-1 w-full"
              value={name}
              onChange={(e) => setName(e.target.value)}
            />
          </div>

          <div className="md:col-span-2">
            <label className="block text-xs font-semibold mb-1">
              Leírás (opcionális)
            </label>
            <textarea
              className="border rounded px-2 py-1 w-full min-h-[60px]"
              value={description}
              onChange={(e) => setDescription(e.target.value)}
            />
          </div>

          <div>
            <label className="block text-xs font-semibold mb-1">Sorrend</label>
            <input
              type="number"
              className="border rounded px-2 py-1 w-full"
              value={sortOrder}
              onChange={(e) => setSortOrder(e.target.value)}
              placeholder="pl. 10"
            />
          </div>

          <div className="flex items-center gap-2 mt-2">
            <input
              id="cat-active"
              type="checkbox"
              className="h-4 w-4"
              checked={isActive}
              onChange={(e) => setIsActive(e.target.checked)}
            />
            <label htmlFor="cat-active" className="text-xs">
              Aktív kategória
            </label>
          </div>

          <div className="md:col-span-2 flex justify-end gap-2 mt-2">
            <button
              type="button"
              onClick={resetForm}
              className="px-3 py-1.5 rounded border text-xs">
              Mégse
            </button>
            <button
              type="submit"
              className="px-3 py-1.5 rounded bg-emerald-600 text-white text-xs">
              {editingId ? "Kategória frissítése" : "Kategória létrehozása"}
            </button>
          </div>
        </form>
      </section>
    </main>
  );
}
