"use client";

import { useEffect, useState, useCallback } from "react";
import Select from "react-select";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";

type HrTaskPackage = {
  id: string;
  code: string;
  name: string;
  description?: string | null;
  kind: "onboarding" | "offboarding" | "other";
  is_active: boolean;
  created_at?: string;
  updated_at?: string;
};

type HrTaskPackageItem = {
  id?: string;
  template_id: string;
  template_code: string;
  template_title: string;
  offset_days: number | null;
  sort_order: number;
};

type TemplateOption = {
  id: string;
  code: string;
  title: string;
  description?: string | null;
  category_code?: string | null;
  is_active: boolean;
};

type SelectOption = { value: string; label: string };

export default function HrTaskPackageManager() {
  const [packages, setPackages] = useState<HrTaskPackage[]>([]);
  const [packagesLoading, setPackagesLoading] = useState(false);

  const [selectedPackageId, setSelectedPackageId] = useState<string | null>(
    null
  );

  const [packageForm, setPackageForm] = useState<{
    id?: string;
    code: string;
    name: string;
    description: string;
    kind: "onboarding" | "offboarding" | "other";
    is_active: boolean;
  }>({
    id: undefined,
    code: "",
    name: "",
    description: "",
    kind: "onboarding",
    is_active: true,
  });

  const [packageLoading, setPackageLoading] = useState(false);
  const [savingPackage, setSavingPackage] = useState(false);

  const [templates, setTemplates] = useState<TemplateOption[]>([]);
  const [templatesLoading, setTemplatesLoading] = useState(false);

  const [items, setItems] = useState<HrTaskPackageItem[]>([]);
  const [itemsLoading, setItemsLoading] = useState(false);
  const [savingItems, setSavingItems] = useState(false);

  // Load all packages
  const reloadPackages = useCallback(async () => {
    setPackagesLoading(true);
    try {
      const res = await ApiCall.hr.listTaskPackages();
      setPackages(res.data?.items ?? []);
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message || "HR csomagok betöltése sikertelen"
      );
    } finally {
      setPackagesLoading(false);
    }
  }, []);

  // Load all templates (for picker)
  const reloadTemplates = useCallback(async () => {
    setTemplatesLoading(true);
    try {
      const res = await ApiCall.hr.listTaskTemplates();
      setTemplates(res.data?.items ?? []);
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message ||
          "HR feladat sablonok betöltése sikertelen"
      );
    } finally {
      setTemplatesLoading(false);
    }
  }, []);

  // Load single package + its items
  const loadPackageDetail = useCallback(async (pkgId: string) => {
    setPackageLoading(true);
    setItemsLoading(true);
    try {
      const res = await ApiCall.hr.getTaskPackage(pkgId);
      const pkg: HrTaskPackage = res.data.package;
      const pkgItems: HrTaskPackageItem[] = res.data.items ?? [];

      setPackageForm({
        id: pkg.id,
        code: pkg.code,
        name: pkg.name,
        description: pkg.description ?? "",
        kind: pkg.kind,
        is_active: pkg.is_active,
      });

      setItems(
        pkgItems.map((i) => ({
          id: i.id,
          template_id: i.template_id,
          template_code: i.template_code,
          template_title: i.template_title,
          offset_days: i.offset_days ?? 0,
          sort_order: i.sort_order ?? 100,
        }))
      );
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message ||
          "HR csomag részleteinek betöltése sikertelen"
      );
    } finally {
      setPackageLoading(false);
      setItemsLoading(false);
    }
  }, []);

  // New package
  const startNewPackage = () => {
    setSelectedPackageId(null);
    setPackageForm({
      id: undefined,
      code: "",
      name: "",
      description: "",
      kind: "onboarding",
      is_active: true,
    });
    setItems([]);
  };

  // Save package header
  const handleSavePackage = async () => {
    if (!packageForm.code.trim() || !packageForm.name.trim()) {
      toast.error("Kód és név kötelező");
      return;
    }

    setSavingPackage(true);
    try {
      const res = await ApiCall.hr.upsertTaskPackage({
        id: packageForm.id,
        code: packageForm.code.trim(),
        name: packageForm.name.trim(),
        description: packageForm.description.trim() || null,
        kind: packageForm.kind,
        is_active: packageForm.is_active,
      });

      const saved = res.data as HrTaskPackage;
      toast.success("Csomag mentve");
      await reloadPackages();

      setSelectedPackageId(saved.id);
      setPackageForm((prev) => ({
        ...prev,
        id: saved.id,
      }));
    } catch (err: any) {
      console.error(err);
      toast.error(err?.response?.data?.message || "Csomag mentése sikertelen");
    } finally {
      setSavingPackage(false);
    }
  };

  // Save items (replace all items for this package)
  const handleSaveItems = async () => {
    if (!packageForm.id) {
      toast.error("Először mentsd a csomag adatait");
      return;
    }
    setSavingItems(true);
    try {
      await ApiCall.hr.saveTaskPackageItems(packageForm.id, {
        items: items.map((i) => ({
          id: i.id,
          template_id: i.template_id,
          sort_order: i.sort_order ?? 100,
          offset_days:
            i.offset_days === null || i.offset_days === undefined
              ? null
              : Number(i.offset_days),
        })),
      });
      toast.success("Csomag tételei mentve");
      // Optionally reload items from backend to ensure consistency
      await loadPackageDetail(packageForm.id);
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message || "Csomag tételeinek mentése sikertelen"
      );
    } finally {
      setSavingItems(false);
    }
  };

  // Add item (from template picker)
  const handleAddTemplateToPackage = (opt: SelectOption | null) => {
    if (!opt) return;
    const templateId = opt.value;
    const template = templates.find((t) => t.id === templateId);
    if (!template) return;

    // Skip if already added
    if (items.some((i) => i.template_id === templateId)) {
      toast.error("Ez a sablon már szerepel a csomagban");
      return;
    }

    setItems((prev) => [
      ...prev,
      {
        template_id: template.id,
        template_code: template.code,
        template_title: template.title,
        offset_days: 0,
        sort_order: (prev[prev.length - 1]?.sort_order ?? 100) + 10,
      },
    ]);
  };

  const handleRemoveItem = (idx: number) => {
    setItems((prev) => prev.filter((_, i) => i !== idx));
  };

  // filtered template options (exclude already added)
  const templateSelectOptions: SelectOption[] = templates
    .filter((t) => !items.some((i) => i.template_id === t.id) && t.is_active)
    .map((t) => ({
      value: t.id,
      label: `${t.code} – ${t.title}`,
    }));

  // Init
  useEffect(() => {
    reloadPackages();
    reloadTemplates();
  }, [reloadPackages, reloadTemplates]);

  useEffect(() => {
    if (selectedPackageId) {
      loadPackageDetail(selectedPackageId);
    }
  }, [selectedPackageId, loadPackageDetail]);

  return (
    <section className="mt-4 border rounded p-4 bg-gray-50 space-y-4">
      <h1 className="font-semibold text-base mb-2">
        HR feladat csomagok (onboarding / offboarding)
      </h1>

      {/* Top: package list + "new" button */}
      <div className="flex justify-between items-center mb-2">
        <h2 className="font-semibold text-sm">Csomagok</h2>
        <button
          type="button"
          onClick={startNewPackage}
          className="px-3 py-1.5 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white text-xs">
          Új csomag
        </button>
      </div>

      <div className="border rounded bg-white max-h-64 overflow-auto">
        {packagesLoading ? (
          <div className="p-2 text-xs text-gray-500">Csomagok betöltése…</div>
        ) : packages.length === 0 ? (
          <div className="p-2 text-xs text-gray-500">
            Még nincs HR feladat csomag rögzítve.
          </div>
        ) : (
          <table className="min-w-full text-xs">
            <thead className="bg-gray-100">
              <tr>
                <th className="px-2 py-1 border">Kód</th>
                <th className="px-2 py-1 border">Név</th>
                <th className="px-2 py-1 border">Típus</th>
                <th className="px-2 py-1 border">Aktív?</th>
                <th className="px-2 py-1 border">Művelet</th>
              </tr>
            </thead>
            <tbody>
              {packages.map((p) => (
                <tr
                  key={p.id}
                  className={selectedPackageId === p.id ? "bg-blue-50" : ""}>
                  <td className="px-2 py-1 border text-[11px]">{p.code}</td>
                  <td className="px-2 py-1 border text-[11px]">{p.name}</td>
                  <td className="px-2 py-1 border text-[11px]">
                    {p.kind === "onboarding"
                      ? "Onboarding"
                      : p.kind === "offboarding"
                      ? "Offboarding"
                      : "Egyéb"}
                  </td>
                  <td className="px-2 py-1 border text-[11px]">
                    {p.is_active ? "Igen" : "Nem"}
                  </td>
                  <td className="px-2 py-1 border text-[11px]">
                    <button
                      type="button"
                      onClick={() => setSelectedPackageId(p.id)}
                      className="px-2 py-1 rounded bg-blue-600/80 hover:bg-blue-600 text-white text-[11px]">
                      Szerkesztés
                    </button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        )}
      </div>

      {/* Package form + items */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-4 text-xs">
        {/* Left: package header form */}
        <div className="border rounded p-3 bg-white space-y-2">
          <div className="flex justify-between items-center">
            <h2 className="font-semibold text-sm">Csomag adatai</h2>
            {packageLoading && (
              <span className="text-[11px] text-gray-500">Betöltés…</span>
            )}
          </div>

          <div className="space-y-2">
            <div>
              <label className="block text-[11px] font-semibold mb-1">
                Kód
              </label>
              <input
                className="border rounded px-2 py-1 w-full"
                value={packageForm.code}
                onChange={(e) =>
                  setPackageForm((prev) => ({
                    ...prev,
                    code: e.target.value,
                  }))
                }
                placeholder="pl. onboarding_default"
              />
            </div>

            <div>
              <label className="block text-[11px] font-semibold mb-1">
                Név
              </label>
              <input
                className="border rounded px-2 py-1 w-full"
                value={packageForm.name}
                onChange={(e) =>
                  setPackageForm((prev) => ({
                    ...prev,
                    name: e.target.value,
                  }))
                }
                placeholder="pl. Alap onboarding csomag"
              />
            </div>

            <div>
              <label className="block text-[11px] font-semibold mb-1">
                Leírás
              </label>
              <textarea
                className="border rounded px-2 py-1 w-full min-h-[60px]"
                value={packageForm.description}
                onChange={(e) =>
                  setPackageForm((prev) => ({
                    ...prev,
                    description: e.target.value,
                  }))
                }
              />
            </div>

            <div>
              <label className="block text-[11px] font-semibold mb-1">
                Típus
              </label>
              <select
                className="border rounded px-2 py-1 w-full"
                value={packageForm.kind}
                onChange={(e) =>
                  setPackageForm((prev) => ({
                    ...prev,
                    kind: e.target.value as
                      | "onboarding"
                      | "offboarding"
                      | "other",
                  }))
                }>
                <option value="onboarding">Onboarding</option>
                <option value="offboarding">Offboarding</option>
                <option value="other">Egyéb</option>
              </select>
            </div>

            <div className="flex items-center gap-2 mt-1">
              <input
                id="pkg-active"
                type="checkbox"
                className="h-4 w-4"
                checked={packageForm.is_active}
                onChange={(e) =>
                  setPackageForm((prev) => ({
                    ...prev,
                    is_active: e.target.checked,
                  }))
                }
              />
              <label htmlFor="pkg-active" className="text-[11px]">
                Aktív csomag
              </label>
            </div>
          </div>

          <div className="flex justify-end mt-3">
            <button
              type="button"
              onClick={handleSavePackage}
              disabled={savingPackage}
              className="px-3 py-1.5 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white text-xs disabled:opacity-50">
              {savingPackage ? "Mentés…" : "Csomag mentése"}
            </button>
          </div>
        </div>

        {/* Right: package items (templates inside package) */}
        <div className="border rounded p-3 bg-white space-y-2">
          <div className="flex justify-between items-center">
            <h2 className="font-semibold text-sm">Csomag tételek</h2>
            {itemsLoading && (
              <span className="text-[11px] text-gray-500">
                Tételek betöltése…
              </span>
            )}
          </div>

          {!packageForm.id ? (
            <div className="text-[11px] text-gray-500">
              Először mentsd a csomag adatait, utána tudsz tételeket hozzáadni.
            </div>
          ) : (
            <>
              {/* Add template to package */}
              <div className="border rounded p-2 bg-gray-50 space-y-1">
                <div className="text-[11px] font-semibold mb-1">
                  Sablon hozzáadása a csomaghoz
                </div>
                {templatesLoading ? (
                  <div className="text-[11px] text-gray-500">
                    Sablonok betöltése…
                  </div>
                ) : (
                  <Select
                    classNamePrefix="hr-pkg-tpl"
                    options={templateSelectOptions}
                    onChange={(v) =>
                      handleAddTemplateToPackage(v as SelectOption)
                    }
                    isClearable
                    placeholder="Válassz sablont…"
                  />
                )}
              </div>

              {/* Item list */}
              {items.length === 0 ? (
                <div className="text-[11px] text-gray-500 mt-2">
                  Még nincs sablon hozzárendelve ehhez a csomaghoz.
                </div>
              ) : (
                <div className="mt-2 max-h-64 overflow-auto border rounded">
                  <table className="min-w-full text-[11px]">
                    <thead className="bg-gray-100">
                      <tr>
                        <th className="px-2 py-1 border">Sablon</th>
                        <th className="px-2 py-1 border">Offset nap</th>
                        <th className="px-2 py-1 border">Sorrend</th>
                        <th className="px-2 py-1 border">Művelet</th>
                      </tr>
                    </thead>
                    <tbody>
                      {items.map((it, idx) => (
                        <tr key={`${it.template_id}-${idx}`}>
                          <td className="px-2 py-1 border align-top">
                            <div className="font-semibold">
                              {it.template_code}
                            </div>
                            <div>{it.template_title}</div>
                          </td>
                          <td className="px-2 py-1 border">
                            <input
                              type="number"
                              className="border rounded px-1 py-0.5 w-20"
                              value={
                                it.offset_days === null ||
                                it.offset_days === undefined
                                  ? ""
                                  : String(it.offset_days)
                              }
                              onChange={(e) => {
                                const val = e.target.value;
                                setItems((prev) =>
                                  prev.map((row, i) =>
                                    i === idx
                                      ? {
                                          ...row,
                                          offset_days:
                                            val === "" ? null : Number(val),
                                        }
                                      : row
                                  )
                                );
                              }}
                              placeholder="pl. 0"
                            />
                          </td>
                          <td className="px-2 py-1 border">
                            <input
                              type="number"
                              className="border rounded px-1 py-0.5 w-20"
                              value={String(it.sort_order ?? 100)}
                              onChange={(e) => {
                                const val = e.target.value || "100";
                                setItems((prev) =>
                                  prev.map((row, i) =>
                                    i === idx
                                      ? {
                                          ...row,
                                          sort_order: Number(val),
                                        }
                                      : row
                                  )
                                );
                              }}
                              placeholder="pl. 100"
                            />
                          </td>
                          <td className="px-2 py-1 border text-center">
                            <button
                              type="button"
                              onClick={() => handleRemoveItem(idx)}
                              className="px-2 py-1 rounded bg-red-600/80 hover:bg-red-600 text-white text-[11px]">
                              Törlés
                            </button>
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              )}

              <div className="flex justify-end mt-2">
                <button
                  type="button"
                  onClick={handleSaveItems}
                  disabled={savingItems || !packageForm.id}
                  className="px-3 py-1.5 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white text-xs disabled:opacity-50">
                  {savingItems ? "Mentés…" : "Tételek mentése"}
                </button>
              </div>
            </>
          )}
        </div>
      </div>
    </section>
  );
}
