"use client";

import { useEffect, useState } from "react";
import Link from "next/link";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";

type TaskTemplate = {
  id: string;
  code: string;
  title: string;
  description?: string | null;
  category_id?: string | null;
  category_name?: string | null;
  category_code?: string | null;
  default_offset_days?: number | null;
  default_role?: string | null;
  document_code?: string | null;
  sort_order?: number | null;
  is_active: boolean;
};

export default function HrTaskTemplatesListPage() {
  const [rows, setRows] = useState<TaskTemplate[]>([]);
  const [loading, setLoading] = useState(false);

  const loadAll = async () => {
    setLoading(true);
    try {
      const tplRes = await ApiCall.hr.getTaskTemplates();
      setRows(tplRes.data?.items ?? []);
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message ||
          "HR feladat sablonok betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    loadAll();
  }, []);

  const handleDelete = async (row: TaskTemplate) => {
    if (!confirm(`Biztosan törlöd a sablont? (${row.title})`)) return;
    try {
      await ApiCall.hr.deleteTaskTemplate(row.id);
      toast.success("Sablon törölve");
      await loadAll();
    } catch (err: any) {
      console.error(err);
      toast.error(err?.response?.data?.message || "Sablon törlése sikertelen");
    }
  };

  const columns = [
    {
      id: "code",
      name: "Kód",
      selector: (r: TaskTemplate) => r.code,
      sortable: true,
      width: "140px",
    },
    {
      id: "title",
      name: "Megnevezés",
      selector: (r: TaskTemplate) => r.title,
      sortable: true,
      grow: 2,
    },
    {
      id: "category",
      name: "Kategória",
      selector: (r: TaskTemplate) => r.category_name ?? "",
      cell: (r: TaskTemplate) =>
        r.category_name ? `${r.category_name} (${r.category_code})` : "—",
      sortable: true,
      grow: 1.5,
    },
    {
      id: "default_offset_days",
      name: "Offset (nap)",
      selector: (r: TaskTemplate) => r.default_offset_days ?? 0,
      cell: (r: TaskTemplate) =>
        r.default_offset_days !== null && r.default_offset_days !== undefined
          ? r.default_offset_days
          : "—",
      sortable: true,
      width: "110px",
    },
    {
      id: "default_role",
      name: "Alapértelmezett felelős szerep",
      selector: (r: TaskTemplate) => r.default_role ?? "",
      cell: (r: TaskTemplate) => r.default_role ?? "—",
      sortable: true,
      width: "180px",
    },
    {
      id: "document_code",
      name: "Dok. kód",
      selector: (r: TaskTemplate) => r.document_code ?? "",
      cell: (r: TaskTemplate) => r.document_code ?? "—",
      width: "140px",
    },
    {
      id: "is_active",
      name: "Aktív?",
      selector: (r: TaskTemplate) => (r.is_active ? "Igen" : "Nem"),
      cell: (r: TaskTemplate) => (r.is_active ? "✅" : "—"),
      width: "80px",
    },
    {
      id: "actions",
      name: "Művelet",
      cell: (r: TaskTemplate) => (
        <div className="flex gap-2">
          <Link
            href={`/admin/hr-task-template/${r.id}`}
            className="px-2 py-1 text-xs rounded bg-blue-600 text-white">
            Szerkesztés
          </Link>
          <button
            type="button"
            className="px-2 py-1 text-xs rounded bg-red-600 text-white"
            onClick={() => handleDelete(r)}>
            Törlés
          </button>
        </div>
      ),
      width: "200px",
    },
  ];

  return (
    <main className="p-4 space-y-4">
      <div className="flex justify-between items-center">
        <h1 className="text-lg font-semibold">HR – Feladat sablonok</h1>
        <Link
          href="/admin/hr-task-template/create"
          className="px-3 py-1.5 rounded bg-emerald-600 text-white text-xs">
          Új sablon
        </Link>
      </div>

      <DataTableWrapper
        tableTitle="Feladat sablonok"
        data={rows}
        columns={columns as any}
        progressPending={loading}
        pagination
        keyField="id"
      />
    </main>
  );
}
