"use client";

import { useEffect, useMemo, useState, useCallback } from "react";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import Loader from "@/components/shared/loader/global-loading";
import { TextField } from "@mui/material";
import Autocomplete from "@mui/material/Autocomplete";
import Link from "next/link";
import toast from "react-hot-toast";
import useTableDebounce from "@/hooks/useTableDebounce";
import { Pencil, Trash2 } from "lucide-react";

const OwnCompanyGroupList = () => {
  const [data, setData] = useState([]);
  const [page, setPage] = useState(1);
  const [loading, setLoading] = useState(true);
  const [pageSize, setPageSize] = useState(25);
  const [sortField, setSortField] = useState("groupName");
  const [sortDirection, setSortDirection] = useState("desc");
  const [totalRecords, setTotalRecords] = useState(0);
  const [inputValue, setInputValue] = useState("");
  const [selectedRows, setSelectedRows] = useState([]);
  const [toggleCleared, setToggleCleared] = useState(false);
  const debouncedSearchTerm = useTableDebounce(inputValue, 500);
  const [showIdColumn, setShowColumnId] = useState(false);
  //   const query = `?page=${page}&pageSize=${pageSize}&sortField=${sortField}&sortDirection=${sortDirection}&searchTerm=${debouncedSearchTerm.trim()}`;

  const query = useMemo(() => {
    return `?page=${page}&pageSize=${pageSize}&sortField=${sortField}&sortDirection=${sortDirection}&searchTerm=${debouncedSearchTerm.trim()}`;
  }, [page, pageSize, sortField, sortDirection, debouncedSearchTerm]);

  const fetchData = async () => {
    setLoading(true);
    try {
      const response = await ApiCall.ownCompanyGroups.getList(query);

      setData(response?.data?.tableData ?? []);
      setTotalRecords(response?.data?.totalRecords ?? 0);
    } catch (err) {
      console.error(err);
    }
    setLoading(false);
  };
  useEffect(() => {
    fetchData();
  }, [query]);

  const handleSelectAction = async (actionType, rowsOrRow) => {
    const selected = Array.isArray(rowsOrRow) ? rowsOrRow : [rowsOrRow];
    const ids = selected.map((r) => r.id);

    if (actionType === "delete") {
      const confirmDelete = confirm(`Törlés megerősítése:\n${ids.join(", ")}`);
      if (confirmDelete) {
        try {
          await ApiCall.ownCompanyGroups.delete(ids);
          fetchData();
          setToggleCleared(!toggleCleared);
          toast.success("Törlés sikeres");
        } catch (err) {
          console.error("Delete failed:", err);
          toast.error("Törlés sikertelen");
        }
      }
    }

    if (actionType === "edit") {
      if (selected.length !== 1) {
        alert("Csak egy elem szerkeszthető egyszerre.");
        return;
      }
      const id = selected[0]._id;
      window.location.href = `/admin/own-company-group/edit/${id}`;
    }
  };

  const tableSelectOptions = [
    {
      icon: <Pencil className="w-4 h-4" />,
      type: "edit",
      className: "bg-blue-500 text-white",
    },
    {
      icon: <Trash2 className="w-4 h-4" />,
      type: "delete",
      className: "bg-red-500 text-white",
    },
  ];

  const columns = useMemo(
    () => [
      ...(showIdColumn
        ? [
            {
              id: "id",
              name: "ID",
              selector: (row) => row.id,
              sortable: true,
            },
          ]
        : []),
      {
        id: "name",
        name: "Cégcsoport",
        selector: (row) => row.name || "-",
        sortable: true,
      },
      {
        id: "company_count",
        name: "Cégek száma",
        selector: (row) => row.company_count || "-",
        sortable: true,
      },

      {
        id: "created_at",
        name: "Létrehozva",
        selector: (row) => row.created_at,
        sortable: true,
      },
      {
        id: "updated_at",
        name: "Módosítva",
        selector: (row) => row.updated_at,
        sortable: true,
      },
      {
        name: "Műveletek",
        cell: (row) => (
          <div className="flex space-x-2">
            <Link
              href={`/admin/own-company-group/edit/${row.id}`}
              className="rounded bg-blue-500/75 px-3 py-1 text-sm text-white hover:bg-blue-600">
              <Pencil className="w-4 h-4" />
            </Link>
            <button
              onClick={() => handleSelectAction("delete", row)}
              className="rounded bg-red-500/75 px-3 py-1 text-sm text-white hover:bg-red-600">
              <Trash2 className="w-4 h-4" />
            </button>
          </div>
        ),
      },
    ],
    []
  );

  const subHeaderComponent = (
    <div className="flex flex-wrap w-full justify-between items-center gap-4">
      <Autocomplete
        freeSolo
        id="subHeaderSearch"
        className="w-72"
        disableClearable
        options={[]}
        inputValue={inputValue}
        onInputChange={(event, newInputValue) => {
          setInputValue(newInputValue);
        }}
        renderInput={(params) => (
          <TextField
            {...params}
            label="Keresés"
            type="search"
            variant="outlined"
          />
        )}
      />
      {selectedRows.length > 0 && (
        <div className="flex items-center gap-2 bg-gray-100 px-4 py-2 border rounded">
          <span className="text-sm text-gray-600">
            {selectedRows.length} kiválasztva
          </span>
          {tableSelectOptions
            .filter((option) => {
              // Only show "edit" if exactly 1 row is selected
              if (option.type === "edit" && selectedRows.length !== 1)
                return false;
              return true;
            })
            .map((option) => (
              <button
                key={option.type}
                title={option.type === "edit" ? "Szerkesztés" : "Törlés"}
                onClick={() => handleSelectAction(option.type, selectedRows)}
                className={`rounded px-3 py-1 text-sm flex items-center gap-1 ${option.className}`}>
                {option.icon}
              </button>
            ))}
        </div>
      )}
    </div>
  );

  return (
    <DataTableWrapper
      tableTitle="Cégcsoportok listája"
      data={data}
      columns={columns}
      progressPending={loading}
      paginationDefaultPage={page}
      defaultSortFieldId="createdAt"
      paginationServer
      onChangePage={(pageNumber) => setPage(pageNumber)}
      progressComponent={<Loader />}
      paginationTotalRows={totalRecords}
      paginationPerPage={pageSize}
      onSort={(col, dir) => {
        setSortField(col.id);
        setSortDirection(dir);
        setPage(1);
      }}
      onChangeRowsPerPage={(size) => {
        setPageSize(size);
        setPage(1);
      }}
      onFilterClear={fetchData}
      subHeader
      subHeaderComponent={subHeaderComponent}
      subHeaderAlign="right"
      subHeaderWrap
      tableOptionsPath="own-company-group/create"
      selectableRows={true}
      onRowSelected={setSelectedRows}
      toggleCleared={toggleCleared}
    />
  );
};

export default OwnCompanyGroupList;
