"use client";

import { useEffect, useState, useCallback } from "react";
import Input from "@/components/shared/input/Input";
import { useForm } from "@/hooks/custom-form-hook";
import { useRouter } from "next/navigation";
import Select from "react-select";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import {
  VALIDATOR_REQUIRE,
  VALIDATOR_MINLENGTH,
  VALIDATOR_EMAIL,
  VALIDATOR_PHONE_HU,
  VALIDATOR_TAXNUMBER_HU,
} from "@/utils/validators";

const CreatePartnerCompany = () => {
  const router = useRouter();
  const [formState, handleInput] = useForm({}, false);
  const [groupOptions, setGroupOptions] = useState([]);
  const [shortCodeOptions, setShortCodeOptions] = useState([]);

  useEffect(() => {
    const fetchOptions = async () => {
      try {
        const [groupsRes, shortCodesRes] = await Promise.all([
          ApiCall.partnerCompanies.getAllGroupNames(),
          ApiCall.partnerCompanies.getAllShortNames(),
        ]);
        if (groupsRes.status === 200) setGroupOptions(groupsRes.data.groups);
        if (shortCodesRes.status === 200)
          setShortCodeOptions(shortCodesRes.data.shortCodes);
      } catch (error) {
        console.error("Option fetch failed:", error);
      }
    };
    fetchOptions();
  }, []);

  const handleSubmit = useCallback(async () => {
    const payload = Object.fromEntries(
      Object.entries(formState.inputs).map(([k, v]) => [k, v.value])
    );

    try {
      const res = await ApiCall.partnerCompanies.postCreate(payload);
      if (res.status === 200) {
        toast.success(res.data.message || "Cég mentve");
        router.push("/admin/partner-company/list");
      } else {
        toast.error(res.data.error || "Hiba történt mentés közben");
      }
    } catch (err) {
      console.error("Submit failed:", err);
      toast.error("Mentés nem sikerült");
    }
  }, [formState]);

  return (
    <div className="p-6 max-w-4xl mx-auto">
      <h2 className="text-xl font-bold mb-4">Partner cég létrehozása</h2>
      <Input
        id="name"
        name="name"
        element="input"
        placeholder="Cégnév"
        validators={[VALIDATOR_REQUIRE(), VALIDATOR_MINLENGTH(2)]}
        errorText="Kötelező mező"
        initialValid={false}
        onInput={handleInput}
      />
      <Input
        id="email"
        name="email"
        element="input"
        placeholder="Email"
        validators={[VALIDATOR_EMAIL]}
        errorText="Nem megfelelő email"
        initialValid={true}
        onInput={handleInput}
      />
      <Input
        id="phone"
        name="phone"
        element="input"
        placeholder="Telefon"
        validators={[VALIDATOR_PHONE_HU]}
        errorText="Nem megfelelő telefonszám"
        initialValid={true}
        onInput={handleInput}
      />
      <Input
        id="address"
        name="address"
        element="input"
        placeholder="Cím"
        onInput={handleInput}
        initialValid={true}
      />
      <Input
        id="tax_number"
        name="tax_number"
        element="input"
        placeholder="Adószám"
        validators={[VALIDATOR_TAXNUMBER_HU]}
        errorText="Nem megfelelő adószám"
        onInput={handleInput}
        initialValid={true}
      />
      <Input
        id="community_tax_number"
        name="community_tax_number"
        element="input"
        placeholder="Közösségi adószám"
        onInput={handleInput}
        initialValid={true}
      />
      <Input
        id="company_registration_number"
        name="company_registration_number"
        element="input"
        placeholder="Cégjegyzékszám"
        onInput={handleInput}
        initialValid={true}
      />
      <div className="mb-4">
        <label className="text-sm text-gray-600 mb-1 block">
          Magánszemély?
        </label>
        <select
          onChange={(e) =>
            handleInput("isPrivatePerson", e.target.value === "true", true)
          }
          className="w-full border rounded px-3 py-2">
          <option value="false">Nem</option>
          <option value="true">Igen</option>
        </select>
      </div>
      <Input
        id="notes"
        name="notes"
        element="textarea"
        placeholder="Megjegyzés"
        initialValid={true}
        onInput={handleInput}
      />
      <div className="mb-4">
        <label className="text-sm text-gray-600 mb-1 block">
          Cégcsoport (opcionális)
        </label>
        <Select
          id="company_group_id"
          name="company_group_id"
          options={groupOptions}
          isClearable
          placeholder="Válassz egy cégcsoportot"
          onChange={(selected) =>
            handleInput("company_group_id", selected?.value || "", true)
          }
        />
      </div>
      <div className="mb-4">
        <label className="text-sm text-gray-600 mb-1 block">
          Névkód (opcionális)
        </label>
        <Select
          id="short_code_id"
          name="short_code_id"
          options={shortCodeOptions}
          isClearable
          placeholder="Válassz egy névkódot"
          onChange={(selected) =>
            handleInput("short_code_id", selected?.value || "", true)
          }
        />
      </div>
      <button
        className={`mt-4 px-4 py-2 ${
          !formState.isValid
            ? "bg-gray-500 cursor-not-allowed"
            : "bg-green-600 hover:bg-green-700"
        } text-white rounded`}
        onClick={handleSubmit}
        disabled={!formState.isValid}>
        Mentés
      </button>
    </div>
  );
};

export default CreatePartnerCompany;
