"use client";

import { useEffect, useMemo, useState, useCallback, useRef } from "react";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import Loader from "@/components/shared/loader/global-loading";
import { TextField } from "@mui/material";
import Autocomplete from "@mui/material/Autocomplete";
import toast from "react-hot-toast";
import useTableDebounce from "@/hooks/useTableDebounce";
import { Trash2 } from "lucide-react";

const PartnerCompanyList = () => {
  const [data, setData] = useState([]);
  const [page, setPage] = useState(1);
  const [loading, setLoading] = useState(true);
  const [pageSize, setPageSize] = useState(25);
  const [sortField, setSortField] = useState("name");
  const [sortDirection, setSortDirection] = useState("asc");
  const [totalRecords, setTotalRecords] = useState(0);
  const [inputValue, setInputValue] = useState("");
  const [selectedRows, setSelectedRows] = useState([]);
  const [toggleCleared, setToggleCleared] = useState(false);
  const debouncedSearchTerm = useTableDebounce(inputValue, 500);

  const query = useMemo(() => {
    return `?page=${page}&pageSize=${pageSize}&sortField=${sortField}&sortDirection=${sortDirection}&searchTerm=${debouncedSearchTerm.trim()}`;
  }, [page, pageSize, sortField, sortDirection, debouncedSearchTerm]);

  const fetchData = async () => {
    setLoading(true);
    try {
      const response = await ApiCall.partnerCompanies.getList(query);
      setData(response?.data?.tableData ?? []);
      setTotalRecords(response?.data?.totalRecords ?? 0);
    } catch (err) {
      console.error(err);
    }
    setLoading(false);
  };

  useEffect(() => {
    fetchData();
  }, [query]);

  const handleSelectAction = async (actionType, rowsOrRow) => {
    const selected = Array.isArray(rowsOrRow) ? rowsOrRow : [rowsOrRow];
    const ids = selected.map((r) => r._id);

    if (actionType === "delete") {
      const confirmDelete = confirm(`Törlés megerősítése:\n${ids.join(", ")}`);
      if (confirmDelete) {
        try {
          await ApiCall.partnerCompanies.delete(ids);
          fetchData();
          setToggleCleared(!toggleCleared);
          toast.success("Törlés sikeres");
        } catch (err) {
          console.error("Delete failed:", err);
          toast.error("Törlés sikertelen");
        }
      }
    }
  };
  const customStyles = {
    rows: {
      style: {
        fontSize: "11px !important", // Change row font size
        minHeight: "48px !important", // Optional: row height
      },
    },
    headCells: {
      style: {
        fontSize: "12px !important", // Header font size
        fontWeight: "bold !important",
        backgroundColor: "#f9f9f9 !important",
      },
    },
    cells: {
      style: {
        fontSize: "0.73rem !important", // Body cell font size
      },
    },
  };
  const columns = useMemo(
    () => [
      {
        id: "name",
        name: "Cég neve",
        selector: (row) => row.name,
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.name}>
            {row.name}
          </div>
        ),
        sortable: true,
      },
      {
        id: "email",
        name: "Email",
        selector: (row) => row.email,
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.email}>
            {row.email}
          </div>
        ),
        sortable: true,
      },
      {
        id: "phone",
        name: "Telefon",
        selector: (row) => row.phone,
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.phone}>
            {row.phone}
          </div>
        ),
        sortable: true,
      },
      {
        id: "address",
        name: "Cím",
        selector: (row) => row.address,
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.address}>
            {row.address}
          </div>
        ),
        sortable: true,
      },
      {
        id: "taxNumber",
        name: "Adószám",
        selector: (row) => row.taxNumber,
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.taxNumber}>
            {row.taxNumber}
          </div>
        ),
        sortable: true,
      },
      {
        id: "communityTaxNumber",
        name: "Közösségi adószám",
        selector: (row) => row.communityTaxNumber || "-",
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.communityTaxNumber}>
            {row.communityTaxNumber}
          </div>
        ),
        sortable: true,
      },
      {
        id: "registrationNumber",
        name: "Cégjegyzékszám",
        selector: (row) => row.registrationNumber || "-",
        cell: (row) => (
          <div
            className="whitespace-pre-wrap break-words max-w-[200px]"
            title={row.registrationNumber}>
            {row.registrationNumber}
          </div>
        ),
        sortable: true,
      },
      {
        id: "isPrivatePerson",
        name: "Magánszemély",
        selector: (row) => (row.isPrivatePerson ? "Igen" : "Nem"),
        sortable: true,
      },
      {
        name: "Műveletek",
        cell: (row) => (
          <button
            onClick={() => handleSelectAction("delete", row)}
            className="rounded bg-red-500/75 px-3 py-1 text-sm text-white hover:bg-red-600">
            <Trash2 className="w-4 h-4" />
          </button>
        ),
      },
    ],
    []
  );

  const subHeaderComponent = (
    <Autocomplete
      freeSolo
      id="partnerCompanySearch"
      className="w-72"
      disableClearable
      options={[]}
      inputValue={inputValue}
      onInputChange={(event, newInputValue) => {
        setInputValue(newInputValue);
      }}
      renderInput={(params) => (
        <TextField
          {...params}
          label="Keresés"
          type="search"
          variant="outlined"
        />
      )}
    />
  );

  return (
    <DataTableWrapper
      tableTitle="Partner cégek listája"
      data={data}
      columns={columns}
      customStyles={customStyles}
      progressPending={loading}
      paginationDefaultPage={page}
      defaultSortFieldId="name"
      paginationServer
      onChangePage={(pageNumber) => setPage(pageNumber)}
      progressComponent={<Loader />}
      paginationTotalRows={totalRecords}
      paginationPerPage={pageSize}
      onSort={(col, dir) => {
        setSortField(col.id);
        setSortDirection(dir);
        setPage(1);
      }}
      onChangeRowsPerPage={(size) => {
        setPageSize(size);
        setPage(1);
      }}
      onFilterClear={fetchData}
      subHeader
      subHeaderComponent={subHeaderComponent}
      subHeaderAlign="right"
      subHeaderWrap
      selectableRows={true}
      onRowSelected={setSelectedRows}
      toggleCleared={toggleCleared}
    />
  );
};

export default PartnerCompanyList;
