// src/app/admin/people/list/page.tsx
"use client";

import { useEffect, useMemo, useState, useCallback } from "react";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";
import Select from "react-select";
import ExpandedPerson from "@/components/people/ExpandedPerson";
import CreatePersonModal from "@/components/people/CreatePersonModal";

export type PersonRow = {
  id: string;
  display_name: string;
  normalized_name?: string | null;
  is_active: boolean;
  status_id?: string | null;
  department_name?: string | null;
  position_title?: string | null;
  manager_name?: string | null;
  hired_at?: string | null;
  exited_at?: string | null;
  open_hr_tasks?: number;
};
type HrOption = { value: string; label: string; code?: string };

type ActiveFilterOption = { value: "all" | "yes" | "no"; label: string };

const activeOptions: ActiveFilterOption[] = [
  { value: "all", label: "Összes" },
  { value: "yes", label: "Aktív" },
  { value: "no", label: "Inaktív" },
];

export default function PeopleListPage() {
  const [rows, setRows] = useState<PersonRow[]>([]);
  const [total, setTotal] = useState(0);
  const [loading, setLoading] = useState(true);

  const [page, setPage] = useState(1);
  const [pageSize, setPageSize] = useState(25);
  const [sortField, setSortField] = useState<"display_name" | "is_active">(
    "display_name"
  );
  const [sortDir, setSortDir] = useState<"asc" | "desc">("asc");
  const [showCreate, setShowCreate] = useState(false);
  // filters
  const [search, setSearch] = useState("");
  const [activeFilter, setActiveFilter] = useState<ActiveFilterOption>(
    activeOptions[0]
  );

  // expanded row
  const [expandedRowId, setExpandedRowId] = useState<string | null>(null);

  const [statusOptions, setStatusOptions] = useState<HrOption[]>([]);
  const [departmentOptions, setDepartmentOptions] = useState<HrOption[]>([]);
  const [companyOptions, setCompanyOptions] = useState<HrOption[]>([]);
  const ellipsis = (text?: string | null) => {
    const v = text ?? "";
    return (
      <div
        title={v}
        style={{
          overflow: "hidden",
          whiteSpace: "nowrap",
          textOverflow: "ellipsis",
        }}>
        {v}
      </div>
    );
  };

  // Build query string to mirror BE listPeople(req.query...)
  const queryString = useMemo(() => {
    const p = new URLSearchParams();
    p.set("page", String(page));
    p.set("pageSize", String(pageSize));
    if (search.trim()) p.set("search", search.trim());
    if (activeFilter.value !== "all") p.set("active", activeFilter.value);
    p.set("sortField", sortField);
    p.set("sortDir", sortDir);
    return `?${p.toString()}`;
  }, [page, pageSize, search, activeFilter, sortField, sortDir]);

  const fetchData = useCallback(async () => {
    setLoading(true);
    try {
      const r = await ApiCall.peoples.getList(queryString);
      console.log("r", r);
      const list: PersonRow[] = r.data?.people ?? r.data ?? [];
      const totalRecords: number | undefined = r.data?.totalRecords;

      setRows(list);
      setTotal(totalRecords ?? list.length);
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.message || "Felhasználók betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  }, [queryString]);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  useEffect(() => {
    const loadHrOptions = async () => {
      try {
        const [statusRes, deptRes, companiesRes] = await Promise.all([
          ApiCall.peoples.getStatuses(),
          ApiCall.peoples.getDepartments(),
          ApiCall.peoples.getOwnCompanies(), // NEW
        ]);

        const sItems: HrOption[] = (statusRes.data?.items ?? []).map(
          (s: any) => ({
            value: s.value,
            label: s.label,
            code: s.code,
          })
        );
        const dItems: HrOption[] = (deptRes.data?.items ?? []).map(
          (d: any) => ({
            value: d.value,
            label: d.label,
          })
        );

        const cItems: HrOption[] = (companiesRes.data?.items ?? []).map(
          (c: any) => ({
            value: c.value,
            label: c.label,
            code: c.short_code,
          })
        );

        setStatusOptions(sItems);
        setDepartmentOptions(dItems);
        setCompanyOptions(cItems);
      } catch (e: any) {
        console.error(e);
        toast.error(
          e?.response?.data?.message ||
            "HR törzsadatok (státuszok, részlegek, cégek) betöltése sikertelen"
        );
      }
    };

    loadHrOptions();
  }, []);

  const columns = [
    {
      id: "display_name",
      name: "Név",
      selector: (r: PersonRow) => r.display_name,
      cell: (r: PersonRow) => (
        <div className="flex items-center gap-2">
          <span className="truncate" title={r.display_name}>
            {r.display_name}
          </span>
          {r.open_hr_tasks && r.open_hr_tasks > 0 ? (
            <span className="inline-flex items-center px-1.5 py-0.5 rounded-full text-[10px] bg-amber-100 text-amber-700">
              {r.open_hr_tasks} HR feladat
            </span>
          ) : null}
        </div>
      ),
      sortable: true,
      grow: 2,
    },
    {
      id: "department_name",
      name: "Részleg",
      selector: (r: PersonRow) => r.department_name ?? "",
      cell: (r: PersonRow) => ellipsis(r.department_name ?? ""),
      sortable: true,
      grow: 2,
    },
    {
      id: "position_title",
      name: "Beosztás",
      selector: (r: PersonRow) => r.position_title ?? "",
      cell: (r: PersonRow) => ellipsis(r.position_title ?? ""),
      sortable: true,
      grow: 2,
    },
    {
      id: "manager_name",
      name: "Közvetlen felettes",
      selector: (r: PersonRow) => r.manager_name ?? "",
      cell: (r: PersonRow) => ellipsis(r.manager_name ?? ""),
      sortable: true,
      grow: 2,
    },
    {
      id: "is_active",
      name: "Aktív?",
      selector: (r: PersonRow) => (r.is_active ? "Igen" : "Nem"),
      cell: (r: PersonRow) => (r.is_active ? "✅ Aktív" : "—"),
      sortable: true,
      width: "120px",
    },
    {
      id: "hired_at",
      name: "Belépés",
      selector: (r: PersonRow) => r.hired_at ?? "",
      cell: (r: PersonRow) => (
        <div style={{ textAlign: "right" }}>
          {r.hired_at ? new Date(r.hired_at).toLocaleDateString() : "—"}
        </div>
      ),
      sortable: true,
      width: "130px",
    },
    {
      id: "exited_at",
      name: "Kilépés",
      selector: (r: PersonRow) => r.exited_at ?? "",
      cell: (r: PersonRow) => (
        <div style={{ textAlign: "right" }}>
          {r.exited_at ? new Date(r.exited_at).toLocaleDateString() : "—"}
        </div>
      ),
      sortable: true,
      width: "130px",
    },
  ];

  const renderExpanded = useCallback(
    (row: PersonRow, onSaved: () => void) => {
      if (!expandedRowId || row.id !== expandedRowId) return null;
      return (
        <div className="p-3">
          <ExpandedPerson
            id={row.id}
            onUpdated={() => {
              onSaved?.();
              fetchData();
            }}
          />
        </div>
      );
    },
    [expandedRowId, fetchData]
  );

  return (
    <main className="p-4 space-y-4">
      {/* Controls */}
      <div className="flex flex-wrap items-center gap-2">
        <input
          className="border rounded px-3 py-2 min-w-72"
          placeholder="Keresés (név, részleg, felettes)"
          value={search}
          onChange={(e) => {
            setSearch(e.target.value);
            setPage(1);
          }}
        />
        <Select
          className="min-w-40"
          options={activeOptions}
          value={activeFilter}
          onChange={(v) => {
            if (!v) return;
            setActiveFilter(v);
            setPage(1);
          }}
        />
        <button
          className="px-3 py-1.5 rounded bg-emerald-600 text-white text-xs"
          onClick={() => setShowCreate(true)}>
          Új személy
        </button>
        {/* you can add more filters here later (department dropdown, etc.) */}
      </div>

      <DataTableWrapper
        tableTitle="Felhasználók / HR nyilvántartás"
        data={rows}
        columns={columns as any}
        progressPending={loading}
        paginationServer
        paginationDefaultPage={page}
        paginationTotalRows={total}
        paginationPerPage={pageSize}
        onChangePage={(p) => setPage(p)}
        onChangeRowsPerPage={(s) => {
          setPageSize(s);
          setPage(1);
        }}
        sortServer
        defaultSortFieldId={sortField}
        onSort={(col: any, dir: "asc" | "desc") => {
          setSortField((col?.id as any) ?? "display_name");
          setSortDir(dir);
          setPage(1);
        }}
        expandableRows
        keyField="id"
        expandableRowExpanded={(r: PersonRow) =>
          !!expandedRowId && r.id === expandedRowId
        }
        onRowExpandToggled={(open: boolean, row: PersonRow) => {
          if (open) {
            if (expandedRowId && expandedRowId !== row.id)
              setExpandedRowId(null);
            setExpandedRowId(row.id);
          } else {
            setExpandedRowId(null);
          }
        }}
        renderExpandableRows={(row: PersonRow) =>
          renderExpanded(row, () => {
            toast.success("Mentve");
            fetchData();
          })
        }
      />
      <CreatePersonModal
        isOpen={showCreate}
        onClose={() => setShowCreate(false)}
        onCreated={fetchData}
        statusOptions={statusOptions}
        departmentOptions={departmentOptions}
        companyOptions={companyOptions} // NEW
      />
    </main>
  );
}
