// src/app/admin/phone-registry/list/page.tsx
"use client";

import { useCallback, useEffect, useMemo, useRef, useState } from "react";
import toast from "react-hot-toast";
import Select from "react-select";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import ExpandedPhoneRegistry from "@/components/phone-registry/ExpandedPhoneRegistry";
import Link from "next/link";

export type PhoneRow = {
  id: string;
  sort_order: number | null;
  phone_number: string;
  phone_number_norm?: string;
  segment_name?: string | null; // Csoport/Személy
  owner_name?: string | null;
  status_name?: string | null; // pl. "Aktív", "Szüneteltetett"
  pause_start?: string | null;
  pause_end?: string | null;
  display_name?: string | null; // Telefon neve
  issued_at?: string | null; // Kiadás dátuma (ISO)
  location_text?: string | null;
  device_brand_name?: string | null;
  device_model_name?: string | null;
  device_serial?: string | null;
  imeis?: string[] | null;
  sim_pins?: string[] | null;
  sim_puks?: string[] | null;
  sim_card_serial?: string | null;
  data_plan?: string | null; // legacy text from BE (or name)
  current_cost_center?: string | null; // short code
  cost_bearer?: string | null;
  future_cost_center?: string | null;
  notes?: string | null;
  updated_at?: string | null;
};

type Option = { value: string; label: string };

export default function PhoneRegistryListPage() {
  const [rows, setRows] = useState<PhoneRow[]>([]);
  const [total, setTotal] = useState<number>(0);
  const [loading, setLoading] = useState(true);
  const [expandedRow, setExpandedRow] = useState<PhoneRow | null>(null);
  const expandedRowId = expandedRow?.id ?? null;

  // paging + sorting
  const [page, setPage] = useState(1);
  const [pageSize, setPageSize] = useState(25);
  const [sortField, setSortField] = useState<
    | "sort_order"
    | "phone_number"
    | "segment_name"
    | "status_name"
    | "issued_at"
    | "data_plan"
    | "current_cost_center"
    | "updated_at"
  >("sort_order");
  const [sortDir, setSortDir] = useState<"asc" | "desc">("asc");

  // filters
  const [search, setSearch] = useState("");

  const [statusOptions, setStatusOptions] = useState<Option[]>([]);
  const [selectedStatuses, setSelectedStatuses] = useState<Option[]>([]);

  const [ccOptions, setCcOptions] = useState<Option[]>([]);
  const [selectedCostCenters, setSelectedCostCenters] = useState<Option[]>([]);

  const [planOptions, setPlanOptions] = useState<Option[]>([]);
  const [selectedPlans, setSelectedPlans] = useState<Option[]>([]);
  const [exporting, setExporting] = useState(false);
  const fileRef = useRef<HTMLInputElement | null>(null);

  // --------- load filter options (once) ----------
  useEffect(() => {
    (async () => {
      try {
        // Short codes (cost centers)
        const [sc, ip] = await Promise.all([
          ApiCall.phoneRegistry.shortCodeOptions("", 200),
          ApiCall.phoneRegistry.internetPackageOptions("", 200),
        ]);

        setCcOptions(
          (sc.data?.items ?? []).map((r: any) => ({
            value: r.code,
            label: r.description ? `${r.code} — ${r.description}` : r.code,
          }))
        );
        setPlanOptions(
          (ip.data?.items ?? []).map((r: any) => ({
            value: r.name,
            label: r.name,
          }))
        );
      } catch (e) {
        console.error(e);
        toast.error("Szűrők betöltése sikertelen");
      }
    })();
  }, []);

  // Build query string from UI state
  const queryString = useMemo(() => {
    const p = new URLSearchParams();
    p.set("page", String(page));
    p.set("pageSize", String(pageSize));
    if (search.trim()) p.set("search", search.trim());

    if (selectedStatuses.length) {
      p.set("status", selectedStatuses.map((s) => s.value).join(",")); // BE: comma-separated names
    }
    if (selectedCostCenters.length) {
      p.set("costCenters", selectedCostCenters.map((c) => c.value).join(",")); // BE: comma-separated short codes
    }
    if (selectedPlans.length) {
      p.set("dataPlans", selectedPlans.map((d) => d.value).join(",")); // BE: comma-separated names
    }

    p.set("sortField", sortField);
    p.set("sortDir", sortDir);
    return `?${p.toString()}`;
  }, [
    page,
    pageSize,
    search,
    selectedStatuses,
    selectedCostCenters,
    selectedPlans,
    sortField,
    sortDir,
  ]);

  // Fetch table data
  const fetchData = async () => {
    setLoading(true);
    try {
      const r = await ApiCall.phoneRegistry.getList(queryString);

      // Preferred shape (BE updated):
      const tableData: PhoneRow[] | undefined = r.data?.phone_lines;
      const totalRecords: number | undefined = r.data?.totalRecords;

      if (Array.isArray(tableData)) {
        setRows(tableData);
        setTotal(totalRecords ?? tableData.length);
      } else {
        // Fallback for legacy BE returning { phone_lines }
        const list: PhoneRow[] = r.data?.phone_lines ?? r.data ?? [];
        setRows(list);
        setTotal(list.length);
      }

      // Populate status options dynamically (first load only, or when empty)
      if (statusOptions.length === 0) {
        const uniq = Array.from(
          new Set(
            (r.data?.tableData ?? r.data?.phone_lines ?? [])
              .map((x: PhoneRow) => x?.status_name)
              .filter(Boolean)
          )
        ) as string[];
        if (uniq.length) {
          setStatusOptions(uniq.map((s) => ({ value: s, label: s })));
        }
      }
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.message || "Telefonlista betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchData();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [queryString]);

  const onExportXlsx = async () => {
    try {
      setExporting(true);
      const resp = await ApiCall.phoneRegistry.exportXlsx(queryString);

      const blob = resp.data as Blob;
      if (!blob) {
        toast.error("Üres export fájl");
        return;
      }

      const url = window.URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      // optional: include date
      a.download = `telefon-nyilvantartas_${new Date()
        .toISOString()
        .slice(0, 10)}.xlsx`;
      document.body.appendChild(a);
      a.click();
      a.remove();
      window.URL.revokeObjectURL(url);

      toast.success("Export kész (XLSX letöltve)");
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.message || "Export sikertelen");
    } finally {
      setExporting(false);
    }
  };

  // Upload (XLS/XLSX)
  const onFileClick = () => {
    if (!fileRef.current) return;
    fileRef.current.value = "";
    fileRef.current.click();
  };
  const onFileChange: React.ChangeEventHandler<HTMLInputElement> = async (
    e
  ) => {
    const f = e.target.files?.[0];
    if (!f) return;
    try {
      const resp = await ApiCall.phoneRegistry.uploadXlsx(f);
      const { inserted = 0, updated = 0, skipped = 0 } = resp.data || {};
      toast.success(
        `Import kész • Új: ${inserted}, Frissítve: ${updated}, Kihagyva: ${skipped}`
      );
      fetchData();
    } catch (err: any) {
      console.error(err);
      toast.error(err?.response?.data?.message || "Import sikertelen");
    }
  };

  // Utilities
  const ellipsis = (text?: string | null) => {
    const v = text ?? "";
    return (
      <div
        title={v}
        style={{
          overflow: "hidden",
          whiteSpace: "nowrap",
          textOverflow: "ellipsis",
        }}>
        {v}
      </div>
    );
  };

  // Columns
  const columns = [
    {
      id: "sort_order",
      name: "Sorszám",
      selector: (r: PhoneRow) => r.sort_order ?? "",
      sortable: true,
      width: "100px",
    },
    {
      id: "phone_number",
      name: "Telefonszám",
      selector: (r: PhoneRow) => r.phone_number,
      cell: (r: PhoneRow) => ellipsis(r.phone_number),
      sortable: true,
      width: "160px",
    },
    {
      id: "segment_name",
      name: "Felelős csoport",
      selector: (r: PhoneRow) => r.segment_name ?? "",
      cell: (r: PhoneRow) => ellipsis(r.segment_name ?? ""),
      sortable: true,
      width: "220px",
    },
    {
      id: "owner_name",
      name: "Kiadva (személy)",
      selector: (r: PhoneRow) => r.owner_name ?? "",
      cell: (r: PhoneRow) => ellipsis(r.owner_name ?? ""),
      sortable: true,
      width: "220px",
    },
    {
      id: "status_name",
      name: "Státusz",
      selector: (r: PhoneRow) => r.status_name ?? "",
      cell: (r: PhoneRow) => ellipsis(r.status_name ?? ""),
      sortable: true,
      width: "140px",
    },
    {
      id: "issued_at",
      name: "Kiadás dátuma",
      selector: (r: PhoneRow) => r.issued_at ?? "",
      cell: (r: PhoneRow) =>
        r.issued_at ? new Date(r.issued_at).toLocaleDateString() : "-",
      sortable: true,
      width: "140px",
    },
    {
      id: "data_plan",
      name: "Internet csomag",
      selector: (r: PhoneRow) => r.data_plan ?? "",
      cell: (r: PhoneRow) => ellipsis(r.data_plan ?? ""),
      sortable: true,
      grow: 2,
    },
    {
      id: "current_cost_center",
      name: "Költséghely",
      selector: (r: PhoneRow) => r.current_cost_center ?? "",
      cell: (r: PhoneRow) => ellipsis(r.current_cost_center ?? ""),
      sortable: true,
      width: "160px",
    },
    {
      id: "cost_bearer",
      name: "Költségviselő",
      selector: (r: PhoneRow) => r.cost_bearer ?? "",
      cell: (r: PhoneRow) => ellipsis(r.cost_bearer ?? ""),
      sortable: true,
      width: "160px",
    },
    {
      id: "updated_at",
      name: "Módosítva",
      selector: (r: PhoneRow) => r.updated_at ?? "",
      cell: (r: PhoneRow) => (
        <div style={{ textAlign: "right" }}>{ellipsis(r.updated_at ?? "")}</div>
      ),
      sortable: true,
      width: "160px",
    },
  ];

  // Expanded row
  const renderExpanded = useCallback(
    (row: PhoneRow, onSaved: () => void) => {
      // render nothing unless this is the selected row
      if (!expandedRowId || row.id !== expandedRowId) return null;
      return (
        <div className="p-3">
          <ExpandedPhoneRegistry
            row={row}
            onSaved={() => {
              onSaved?.();
              fetchData();
            }}
            onDeleted={() => {
              setExpandedRow(null);
              fetchData();
            }}
          />
        </div>
      );
    },
    [expandedRowId] // ← important so the guard updates
  );

  return (
    <main className="p-4 space-y-4">
      {/* Controls */}
      <div className="flex flex-wrap items-center gap-2">
        <input
          className="border rounded px-3 py-2"
          placeholder="Keresés (telefonszám, csoport, megjegyzés)"
          value={search}
          onChange={(e) => {
            setSearch(e.target.value);
            setPage(1);
          }}
        />

        <Select
          className="min-w-48"
          isMulti
          options={statusOptions}
          value={selectedStatuses}
          placeholder="Státusz szűrő"
          onChange={(v) => {
            setSelectedStatuses(v as Option[]);
            setPage(1);
          }}
        />

        <Select
          className="min-w-56"
          isMulti
          options={ccOptions}
          value={selectedCostCenters}
          placeholder="Költséghely szűrő"
          onChange={(v) => {
            setSelectedCostCenters(v as Option[]);
            setPage(1);
          }}
        />

        <Select
          className="min-w-56"
          isMulti
          options={planOptions}
          value={selectedPlans}
          placeholder="Internet csomag szűrő"
          onChange={(v) => {
            setSelectedPlans(v as Option[]);
            setPage(1);
          }}
        />

        <div className="ml-auto flex items-center gap-2">
          <Link
            href="/admin/phone-registry/create"
            className="px-3 py-2 bg-emerald-600/80 hover:bg-emerald-600 text-white rounded">
            Új telefonszám
          </Link>
          <button
            type="button"
            onClick={onFileClick}
            className="px-3 py-2 bg-gray-700/80 hover:bg-gray-700 text-white rounded">
            Feltöltés (XLSX)
          </button>
          <input
            ref={fileRef}
            type="file"
            accept=".xls,.xlsx,application/vnd.ms-excel,application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
            onChange={onFileChange}
            className="sr-only"
          />
          <button
            type="button"
            onClick={onExportXlsx}
            disabled={exporting}
            className={`px-3 py-2 rounded text-white ${
              exporting
                ? "bg-blue-400 cursor-wait"
                : "bg-blue-600/80 hover:bg-blue-600"
            }`}>
            {exporting ? "Exportálás..." : "Export (XLSX)"}
          </button>
        </div>
      </div>

      {/* Table */}
      <DataTableWrapper
        tableTitle="Telefon nyilvántartás"
        data={rows}
        columns={columns as any}
        progressPending={loading}
        paginationServer
        paginationDefaultPage={page}
        paginationPerPage={pageSize}
        paginationTotalRows={total}
        onChangePage={(p) => setPage(p)}
        onChangeRowsPerPage={(s) => {
          setPageSize(s);
          setPage(1);
        }}
        sortServer
        defaultSortFieldId={sortField}
        onSort={(col: any, dir: "asc" | "desc") => {
          setSortField((col?.id as any) ?? "sort_order");
          setSortDir(dir);
          setPage(1);
        }}
        expandableRows
        // ✅ Only mark a row expanded if we actually have a selected id
        expandableRowExpanded={(r: PhoneRow) =>
          !!expandedRowId && r.id === expandedRowId
        }
        onRowExpandToggled={(open: boolean, row: PhoneRow) => {
          if (open) {
            // collapse previous, then open the clicked one
            if (expandedRowId && expandedRowId !== row.id) setExpandedRow(null);
            setExpandedRow(row);
          } else {
            setExpandedRow(null);
          }
        }}
        // ✅ Render expanded content only for the active row
        renderExpandableRows={(row: PhoneRow) =>
          renderExpanded(row, () => {
            toast.success("Mentve");
            fetchData();
          })
        }
        // (optional but helpful if your wrapper supports it)
        keyField="id"
      />
    </main>
  );
}
