"use client";

import { useEffect, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import type { Playbook } from "@/api/routes/playbooks";
import {
  Page,
  Card,
  CardHeader,
  CardBody,
  CardTitle,
  CardMeta,
  ToolbarRow,
  TextInput,
  Button,
  SecondaryButton,
  Select,
  Field,
  Chip,
  EmptyState,
} from "@/components/ui/CardBits";
import { useRouter } from "next/navigation";

export default function PlaybooksAdminPage() {
  const router = useRouter();

  const [items, setItems] = useState<Playbook[]>([]);
  const [loading, setLoading] = useState(false);

  // create
  const [key, setKey] = useState("");
  const [name, setName] = useState("");
  const [desc, setDesc] = useState("");
  const [defaultCaseType, setDefaultCaseType] = useState("");
  const [defaultFlowType, setDefaultFlowType] = useState<
    "projekt" | "folyamatos" | ""
  >("");
  const [creating, setCreating] = useState(false);

  async function load() {
    setLoading(true);
    try {
      const r = await ApiCall.playbooks.list({ includeInactive: 0 });
      setItems(r.data.playbooks ?? []);
    } catch (e: any) {
      toast.error(
        e?.response?.data?.message ?? "Playbooks betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  }

  useEffect(() => {
    void load();
  }, []);

  async function onCreate() {
    if (!key.trim() || !name.trim()) return;
    setCreating(true);
    try {
      const r = await ApiCall.playbooks.create({
        key: key.trim(),
        name: name.trim(),
        description: desc.trim() || undefined,
        defaultCaseType: defaultCaseType.trim() || null,
        defaultFlowType: defaultFlowType || null,
        isActive: true,
      });
      toast.success("Playbook létrehozva");
      setKey("");
      setName("");
      setDesc("");
      setDefaultCaseType("");
      setDefaultFlowType("");

      router.push(`/admin/playbooks/${r.data.playbook.id}`);
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Létrehozás sikertelen");
    } finally {
      setCreating(false);
    }
  }

  return (
    <Page
      title="Playbookok"
      subtitle={loading ? "Betöltés…" : `${items.length} aktív`}
      actions={
        <SecondaryButton onClick={() => void load()} disabled={loading}>
          {loading ? "..." : "Frissít"}
        </SecondaryButton>
      }>
      <Card className="mb-3">
        <CardHeader>
          <div>
            <CardTitle>Új Playbook (A opció)</CardTitle>
            <CardMeta>
              irányt ad + checklist, nem kényszerít workflow-t
            </CardMeta>
          </div>
        </CardHeader>
        <CardBody>
          <div className="grid gap-3">
            <div className="grid gap-3 md:grid-cols-2">
              <Field label="Key" hint="pl. playbook_a">
                <TextInput
                  value={key}
                  onChange={(e) => setKey(e.target.value)}
                />
              </Field>
              <Field label="Név">
                <TextInput
                  value={name}
                  onChange={(e) => setName(e.target.value)}
                />
              </Field>
            </div>
            <Field
              label="Leírás"
              hint="pl. Cadence / Roles sorok is mehetnek ide">
              <TextInput
                value={desc}
                onChange={(e) => setDesc(e.target.value)}
              />
            </Field>

            <div className="grid gap-3 md:grid-cols-2">
              <Field label="Default case_type (opcionális)">
                <TextInput
                  value={defaultCaseType}
                  onChange={(e) => setDefaultCaseType(e.target.value)}
                  placeholder="pl. panasz"
                />
              </Field>
              <Field label="Default flow_type (opcionális)">
                <Select
                  value={defaultFlowType}
                  onChange={(e) => setDefaultFlowType(e.target.value as any)}>
                  <option value="">(nincs)</option>
                  <option value="transient">Projekt</option>
                  <option value="continuous">Folyamatos</option>
                </Select>
              </Field>
            </div>

            <ToolbarRow>
              <Button
                onClick={() => void onCreate()}
                disabled={creating || !key.trim() || !name.trim()}>
                {creating ? "..." : "Létrehoz"}
              </Button>
            </ToolbarRow>
          </div>
        </CardBody>
      </Card>

      {items.length ? (
        <div className="grid gap-3">
          {items.map((p) => (
            <Card
              key={p.id}
              className="cursor-pointer"
              onClick={() => router.push(`/admin/playbooks/${p.id}`)}>
              <CardHeader>
                <div style={{ minWidth: 0 }}>
                  <CardTitle>{p.name}</CardTitle>
                  <CardMeta>{p.key}</CardMeta>
                </div>
                <div className="flex gap-2 flex-wrap">
                  {p.is_active ? (
                    <Chip tone="good">active</Chip>
                  ) : (
                    <Chip tone="warn">inactive</Chip>
                  )}
                </div>
              </CardHeader>
            </Card>
          ))}
        </div>
      ) : (
        <EmptyState title="Nincs playbook" hint="Hozz létre egyet fent." />
      )}
    </Page>
  );
}
