"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import toast from "react-hot-toast";
import { useRouter } from "next/navigation";
import ApiCall from "@/api/api-calls";
import Modal from "@/components/shared/modal/Modal";
import useModalMap from "@/hooks/useModalMap";
import CategoryModal from "@/components/qms/CategoryModal";
import DocumentModal from "@/components/qms/DocumentModal";
import QmsAclEditorModal from "@/components/qms/QmsAclEditorModal"; // <-- add this file below
import {
  DndContext,
  PointerSensor,
  useSensor,
  useSensors,
  DragEndEvent,
  closestCenter,
} from "@dnd-kit/core";
import {
  SortableContext,
  useSortable,
  verticalListSortingStrategy,
  arrayMove,
} from "@dnd-kit/sortable";
import { CSS } from "@dnd-kit/utilities";

type Category = {
  id: string;
  parent_id: string | null;
  name: string;
  sort_order?: number;
  is_active?: boolean;
  children?: Category[];
};

function SortableRow({
  node,
  depth,
  selectedId,
  onSelect,
}: {
  node: Category;
  depth: number;
  selectedId: string | null;
  onSelect: (id: string) => void;
}) {
  const { attributes, listeners, setNodeRef, transform, transition } =
    useSortable({ id: node.id });

  const style: React.CSSProperties = {
    transform: CSS.Transform.toString(transform),
    transition,
  };

  const selected = node.id === selectedId;
  const inactive = node.is_active === false;

  return (
    <div ref={setNodeRef} style={style}>
      <div
        className={[
          "w-full px-2 py-1 rounded flex items-center gap-2",
          selected ? "bg-blue-600 text-white" : "hover:bg-gray-100",
          inactive ? "opacity-60" : "",
        ].join(" ")}
        style={{ paddingLeft: 8 + depth * 12 }}>
        {/* drag handle */}
        <button
          className="cursor-grab active:cursor-grabbing select-none px-1"
          title="Húzd a sorrendhez"
          {...attributes}
          {...listeners}
          onClick={(e) => e.preventDefault()}>
          ⠿
        </button>

        {/* select */}
        <button
          className="flex-1 text-left truncate"
          onClick={() => onSelect(node.id)}
          title={node.name}>
          {node.name}
          {inactive ? <span className="text-xs ml-2">(inactive)</span> : null}
        </button>
      </div>
    </div>
  );
}

function SortableSiblings({
  parentId,
  items,
  depth,
  selectedId,
  onSelect,
  onReorderSiblings,
}: {
  parentId: string | null;
  items: Category[];
  depth: number;
  selectedId: string | null;
  onSelect: (id: string) => void;
  onReorderSiblings: (
    parentId: string | null,
    orderedIds: string[]
  ) => Promise<void> | void;
}) {
  const sensors = useSensors(
    useSensor(PointerSensor, { activationConstraint: { distance: 4 } })
  );

  // local ordering for THIS sibling group only
  const [localItems, setLocalItems] = useState(items);
  useEffect(() => setLocalItems(items), [items]);

  const ids = useMemo(() => localItems.map((x) => x.id), [localItems]);

  return (
    <DndContext
      sensors={sensors}
      collisionDetection={closestCenter}
      onDragEnd={async (e) => {
        const { active, over } = e;
        if (!over) return;
        if (active.id === over.id) return;

        const oldIndex = ids.indexOf(String(active.id));
        const newIndex = ids.indexOf(String(over.id));
        if (oldIndex < 0 || newIndex < 0) return;

        const next = arrayMove(localItems, oldIndex, newIndex);
        setLocalItems(next);

        // IMPORTANT: parentId here is the parentId of THIS sibling group.
        await onReorderSiblings(
          parentId,
          next.map((x) => x.id)
        );
      }}>
      <SortableContext items={ids} strategy={verticalListSortingStrategy}>
        <div className="mt-1 space-y-1">
          {localItems.map((c) => (
            <div key={c.id}>
              <SortableRow
                node={c}
                depth={depth}
                selectedId={selectedId}
                onSelect={onSelect}
              />
              {c.children?.length ? (
                <SortableSiblings
                  parentId={c.id}
                  items={c.children}
                  depth={depth + 1}
                  selectedId={selectedId}
                  onSelect={onSelect}
                  onReorderSiblings={onReorderSiblings}
                />
              ) : null}
            </div>
          ))}
        </div>
      </SortableContext>
    </DndContext>
  );
}
function CategoryTreeSortable(props: {
  tree: Category[];
  selectedId: string | null;
  onSelect: (id: string) => void;
  onReorderSiblings: (
    parentId: string | null,
    orderedIds: string[]
  ) => Promise<void> | void;
}) {
  return (
    <SortableSiblings
      parentId={null}
      items={props.tree}
      depth={0}
      selectedId={props.selectedId}
      onSelect={props.onSelect}
      onReorderSiblings={props.onReorderSiblings}
    />
  );
}

type DocRow = {
  id: string;
  category_id: string;
  code: string | null;
  title: string;
  status: string;
  resolved_version_no: number | null;
  resolved_uploaded_at: string | null;
};

function findCategoryById(
  tree: Category[],
  id: string | null
): Category | null {
  if (!id) return null;
  const stack = [...tree];
  while (stack.length) {
    const n = stack.pop()!;
    if (n.id === id) return n;
    if (n.children?.length) stack.push(...n.children);
  }
  return null;
}

function findParentAndSiblings(
  tree: Category[],
  selectedId: string | null
): { parentId: string | null; siblings: Category[]; index: number } | null {
  if (!selectedId) return null;

  // top-level
  const topIdx = tree.findIndex((c) => c.id === selectedId);
  if (topIdx >= 0) {
    return { parentId: null, siblings: tree, index: topIdx };
  }

  // search recursively for parent whose children contain selectedId
  const stack: Category[] = [...tree];
  while (stack.length) {
    const n = stack.pop()!;
    const idx = (n.children ?? []).findIndex((c) => c.id === selectedId);
    if (idx >= 0) {
      return { parentId: n.id, siblings: n.children ?? [], index: idx };
    }
    if (n.children?.length) stack.push(...n.children);
  }

  return null;
}

function CategoryTree({
  tree,
  selectedId,
  onSelect,
}: {
  tree: Category[];
  selectedId: string | null;
  onSelect: (id: string) => void;
}) {
  const Node = ({ node, depth }: { node: Category; depth: number }) => {
    const selected = node.id === selectedId;
    const inactive = node.is_active === false;

    return (
      <div>
        <button
          className={[
            "w-full text-left px-2 py-1 rounded flex items-center gap-2",
            selected ? "bg-blue-600 text-white" : "hover:bg-gray-100",
            inactive ? "opacity-60" : "",
          ].join(" ")}
          style={{ paddingLeft: 8 + depth * 12 }}
          onClick={() => onSelect(node.id)}
          title={node.name}>
          <span className="truncate">{node.name}</span>
          {inactive ? <span className="text-xs">(inactive)</span> : null}
        </button>

        {node.children?.length ? (
          <div className="mt-1 space-y-1">
            {node.children.map((c) => (
              <Node key={c.id} node={c} depth={depth + 1} />
            ))}
          </div>
        ) : null}
      </div>
    );
  };

  return (
    <div className="space-y-1">
      {tree.map((n) => (
        <Node key={n.id} node={n} depth={0} />
      ))}
    </div>
  );
}

export default function QmsPage() {
  const router = useRouter();
  const { isModalOpen, toggleModal } = useModalMap();

  const [catTree, setCatTree] = useState<Category[]>([]);
  const [selectedCategoryId, setSelectedCategoryId] = useState<string | null>(
    null
  );

  const selectedCategory = useMemo(
    () => findCategoryById(catTree, selectedCategoryId),
    [catTree, selectedCategoryId]
  );

  const [docs, setDocs] = useState<DocRow[]>([]);
  const [loadingDocs, setLoadingDocs] = useState(false);
  const [loadingCats, setLoadingCats] = useState(false);

  // filters/paging
  const [search, setSearch] = useState("");
  const [page, setPage] = useState(1);
  const [pageSize, setPageSize] = useState(25);
  const [total, setTotal] = useState(0);

  // modals
  const [editCategory, setEditCategory] = useState<Category | null>(null);
  const [editDoc, setEditDoc] = useState<DocRow | null>(null);

  const [createParentId, setCreateParentId] = useState<string | null>(null);

  // ACL modal context (category-level from this page)
  const [aclObject, setAclObject] = useState<{
    objectType: "category" | "document";
    objectId: string;
    title: string;
  } | null>(null);

  // IMPORTANT: global search:
  // - if search is set -> omit categoryId to search across all docs
  // - if search is empty -> require categoryId (normal browsing)
  const queryString = useMemo(() => {
    const p = new URLSearchParams();
    p.set("page", String(page));
    p.set("pageSize", String(pageSize));

    const q = search.trim();
    if (q) {
      p.set("q", q);
      // no categoryId -> global search
    } else {
      if (selectedCategoryId) p.set("categoryId", selectedCategoryId);
    }

    p.set("sortField", "title");
    p.set("sortDir", "asc");
    return `?${p.toString()}`;
  }, [page, pageSize, selectedCategoryId, search]);

  const fetchCategories = useCallback(async () => {
    setLoadingCats(true);
    try {
      const r = await ApiCall.qms.getCategories();
      const tree = (r.data?.tree ?? []) as Category[];
      setCatTree(tree);

      if (!selectedCategoryId) {
        if (tree?.[0]?.id) setSelectedCategoryId(tree[0].id);
      } else {
        const stillExists = !!findCategoryById(tree, selectedCategoryId);
        if (!stillExists) setSelectedCategoryId(tree?.[0]?.id ?? null);
      }
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Kategóriák betöltése sikertelen"
      );
    } finally {
      setLoadingCats(false);
    }
  }, [selectedCategoryId]);

  const fetchDocs = useCallback(async () => {
    const q = search.trim();

    // If no search and no selected category -> show empty state
    if (!q && !selectedCategoryId) {
      setDocs([]);
      setTotal(0);
      return;
    }

    // If browsing (no search) but category missing -> also empty
    if (!q && !selectedCategoryId) {
      setDocs([]);
      setTotal(0);
      return;
    }

    setLoadingDocs(true);
    try {
      const r = await ApiCall.qms.listDocuments(queryString);
      setDocs((r.data?.items ?? []) as DocRow[]);
      setTotal(Number(r.data?.totalRecords ?? 0));
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Dokumentumok betöltése sikertelen"
      );
    } finally {
      setLoadingDocs(false);
    }
  }, [queryString, selectedCategoryId, search]);

  useEffect(() => {
    fetchCategories();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  useEffect(() => {
    fetchDocs();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [queryString]);

  // -------- actions --------

  const openCreateCategory = () => {
    setEditCategory(null);
    toggleModal("categoryModal", true);
  };

  const openEditSelectedCategory = () => {
    if (!selectedCategory) return toast.error("Nincs kiválasztott kategória");
    setEditCategory(selectedCategory);
    toggleModal("categoryModal", true);
  };

  const deleteSelectedCategory = async () => {
    if (!selectedCategory) return toast.error("Nincs kiválasztott kategória");
    if (!confirm(`Biztos törlöd? (deactivate)\n\n${selectedCategory.name}`))
      return;

    try {
      await ApiCall.qms.deleteCategory(selectedCategory.id);
      toast.success("Kategória deaktiválva");
      await fetchCategories();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Törlés sikertelen");
    }
  };

  const openCreateDocument = () => {
    // in global search mode: you should force category selection for creating
    if (!selectedCategoryId) return toast.error("Válassz kategóriát");
    setEditDoc(null);
    toggleModal("docModal", true);
  };

  const openEditDocument = (row: DocRow) => {
    setEditDoc(row);
    toggleModal("docModal", true);
  };

  const deleteDocument = async (row: DocRow) => {
    if (!confirm(`Biztos törlöd? (deactivate)\n\n${row.title}`)) return;
    try {
      await ApiCall.qms.deleteDocument(row.id);
      toast.success("Dokumentum deaktiválva");
      fetchDocs();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Törlés sikertelen");
    }
  };
  const openCreateRootCategory = () => {
    setEditCategory(null);
    setCreateParentId(null);
    toggleModal("categoryModal", true);
  };

  const openCreateSubCategory = () => {
    if (!selectedCategoryId) return toast.error("Válassz kategóriát");
    setEditCategory(null);
    setCreateParentId(selectedCategoryId);
    toggleModal("categoryModal", true);
  };

  // -------- reorder (siblings) --------

  const moveSelectedCategory = async (dir: "up" | "down") => {
    if (!selectedCategoryId) return;
    const ctx = findParentAndSiblings(catTree, selectedCategoryId);
    if (!ctx) return;

    const { parentId, siblings, index } = ctx;

    const targetIndex = dir === "up" ? index - 1 : index + 1;
    if (targetIndex < 0 || targetIndex >= siblings.length) return;

    const ordered = siblings.map((s) => s.id);
    // swap
    [ordered[index], ordered[targetIndex]] = [
      ordered[targetIndex],
      ordered[index],
    ];

    try {
      await ApiCall.qms.reorderCategories({ parentId, orderedIds: ordered });
      toast.success("Sorrend mentve");
      await fetchCategories();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Sorrend mentése sikertelen");
    }
  };

  const openAclForSelectedCategory = () => {
    if (!selectedCategoryId || !selectedCategory)
      return toast.error("Nincs kiválasztott kategória");
    setAclObject({
      objectType: "category",
      objectId: selectedCategoryId,
      title: `ACL • ${selectedCategory.name}`,
    });
    toggleModal("aclModal", true);
  };

  return (
    <main className="p-4 grid grid-cols-12 gap-4">
      {/* LEFT: Category tree */}
      <aside className="col-span-12 md:col-span-4 lg:col-span-3 border rounded p-3">
        <div className="flex items-center gap-2 mb-2 flex-wrap">
          <h1 className="font-semibold w-full text-center">
            Minőségirányítási Rendszer kategóriák
          </h1>
          <div className="w-full flex justify-between">
            <button
              onClick={openCreateRootCategory}
              className="px-2 py-1 rounded bg-blue-600 text-white cursor-pointer hover:bg-white hover:text-blue-600 hover:border hover:border-blue-600">
              + Fő kategória
            </button>

            <button
              onClick={openCreateSubCategory}
              className="px-2 py-1 rounded border cursor-pointer hover:bg-blue-400 hover:text-white">
              + Alkategória
            </button>
          </div>
        </div>

        <div className="flex flex-wrap items-center gap-2 mb-3">
          <button
            className="px-2 py-1 rounded border disabled:opacity-50"
            onClick={openEditSelectedCategory}
            disabled={!selectedCategory}>
            Szerkeszt
          </button>
          <button
            className="px-2 py-1 rounded border disabled:opacity-50"
            onClick={deleteSelectedCategory}
            disabled={!selectedCategory}>
            Töröl
          </button>

          <button
            className="px-2 py-1 rounded border disabled:opacity-50"
            onClick={openAclForSelectedCategory}
            disabled={!selectedCategory}>
            ACL
          </button>

          <button
            className="ml-auto px-2 py-1 rounded border"
            onClick={fetchCategories}>
            Frissít
          </button>
        </div>
        <div className="max-h-[70vh] overflow-y-auto pt-2 border-t border-t-gray-300">
          {loadingCats ? (
            <div className="py-6 text-sm text-gray-500">Betöltés…</div>
          ) : (
            <CategoryTreeSortable
              tree={catTree}
              selectedId={selectedCategoryId}
              onSelect={(id) => {
                setSelectedCategoryId(id);
                setPage(1);
              }}
              onReorderSiblings={async (parentId, orderedIds) => {
                try {
                  await ApiCall.qms.reorderCategories({ parentId, orderedIds });
                  toast.success("Sorrend mentve");
                  await fetchCategories();
                } catch (e: any) {
                  console.error(e);
                  toast.error(
                    e?.response?.data?.error ?? "Sorrend mentése sikertelen"
                  );
                }
              }}
            />
          )}
        </div>
      </aside>

      {/* RIGHT: Documents */}
      <section className="col-span-12 md:col-span-8 lg:col-span-9 border rounded p-3">
        <div className="flex flex-wrap items-center gap-2 mb-3">
          <div className="min-w-0">
            <h2 className="font-semibold">Dokumentumok</h2>
            <div className="text-xs text-gray-500 truncate">
              {search.trim() ? "Globális keresés" : `Kategória: `}
              {!search.trim() ? <b>{selectedCategory?.name ?? "—"}</b> : null}
            </div>
          </div>

          {/* ✅ QMS shortcuts */}
          <div className="flex items-center gap-2 ml-2">
            <button
              className="px-2 py-1 rounded border"
              onClick={() => router.push("/admin/qms/training/packages")}>
              Training admin
            </button>

            <button
              className="px-2 py-1 rounded border"
              onClick={() => router.push("/admin/qms/approvals")}
              title="Jóváhagyási kérelmek">
              Approvals
            </button>
            <button
              className="px-2 py-1 rounded border"
              onClick={() => router.push("/admin/qms/work-items")}
              title="Feladatok">
              Work items
            </button>
            <button
              className="px-2 py-1 rounded border"
              onClick={() => router.push("/admin/qms/training")}
              title="Saját képzéseim">
              Training
            </button>
          </div>

          <input
            className="border rounded px-3 py-2 ml-2"
            placeholder="Keresés (cím, kód, leírás) — üresen kategória szerint listáz"
            value={search}
            onChange={(e) => {
              setSearch(e.target.value);
              setPage(1);
            }}
          />

          <button
            onClick={openCreateDocument}
            className="ml-auto px-3 py-2 rounded bg-emerald-600 text-white">
            + Dokumentum
          </button>
        </div>

        <div className="text-sm text-gray-600 mb-2">Összes: {total}</div>

        {loadingDocs ? (
          <div className="py-6 text-sm text-gray-500">Betöltés…</div>
        ) : (
          <div className="overflow-auto">
            <table className="min-w-full text-sm">
              <thead className="text-left border-b">
                <tr>
                  <th className="py-2 pr-3">Kód</th>
                  <th className="py-2 pr-3">Cím</th>
                  <th className="py-2 pr-3">Státusz</th>
                  <th className="py-2 pr-3">Verzió</th>
                  <th className="py-2 pr-3">Módosítva</th>
                  <th className="py-2 pr-3 text-right">Műveletek</th>
                </tr>
              </thead>

              <tbody>
                {docs.map((d) => (
                  <tr key={d.id} className="border-b hover:bg-gray-50">
                    <td
                      className="py-2 pr-3 cursor-pointer"
                      onClick={() =>
                        router.push(`/admin/qms/document/${d.id}`)
                      }>
                      {d.code ?? "—"}
                    </td>
                    <td
                      className="py-2 pr-3 cursor-pointer"
                      onClick={() =>
                        router.push(`/admin/qms/document/${d.id}`)
                      }>
                      {d.title}
                    </td>
                    <td className="py-2 pr-3">{d.status}</td>
                    <td className="py-2 pr-3">
                      {d.resolved_version_no ?? "—"}
                    </td>
                    <td className="py-2 pr-3">
                      {d.resolved_uploaded_at ?? "—"}
                    </td>
                    <td className="py-2 pr-3">
                      <div className="flex justify-end gap-2">
                        <button
                          className="px-2 py-1 rounded border"
                          onClick={() => openEditDocument(d)}>
                          Szerkeszt
                        </button>
                        <button
                          className="px-2 py-1 rounded border"
                          onClick={() => deleteDocument(d)}>
                          Töröl
                        </button>
                        <button
                          className="px-2 py-1 rounded bg-gray-800 text-white"
                          onClick={() =>
                            router.push(`/admin/qms/document/${d.id}`)
                          }>
                          Megnyit
                        </button>
                      </div>
                    </td>
                  </tr>
                ))}

                {!docs.length && (
                  <tr>
                    <td colSpan={6} className="py-6 text-center text-gray-500">
                      {search.trim()
                        ? "Nincs találat"
                        : selectedCategoryId
                        ? "Nincs találat ebben a kategóriában"
                        : "Válassz kategóriát vagy keress"}
                    </td>
                  </tr>
                )}
              </tbody>
            </table>
          </div>
        )}

        {/* Pager */}
        <div className="flex items-center gap-2 mt-3">
          <button
            className="px-2 py-1 border rounded disabled:opacity-50"
            disabled={page <= 1}
            onClick={() => setPage((p) => Math.max(1, p - 1))}>
            ◀
          </button>

          <div className="text-sm">
            Oldal <b>{page}</b> • Összes: <b>{total}</b>
          </div>

          <button
            className="px-2 py-1 border rounded disabled:opacity-50"
            disabled={page * pageSize >= total}
            onClick={() => setPage((p) => p + 1)}>
            ▶
          </button>

          <select
            className="ml-auto border rounded px-2 py-1"
            value={pageSize}
            onChange={(e) => {
              setPageSize(Number(e.target.value));
              setPage(1);
            }}>
            {[10, 25, 50, 100].map((s) => (
              <option key={s} value={s}>
                {s} / oldal
              </option>
            ))}
          </select>
        </div>
      </section>

      {/* Category Modal */}
      <Modal
        show={isModalOpen("categoryModal")}
        onClose={() => toggleModal("categoryModal", false)}
        title={editCategory ? "Kategória szerkesztés" : "Új kategória"}
        height={"auto"}
        width={"640px"}>
        <CategoryModal
          mode={editCategory ? "edit" : "create"}
          category={editCategory ?? undefined}
          parentId={editCategory ? editCategory.parent_id : createParentId}
          onClose={() => toggleModal("categoryModal", false)}
          onSaved={async () => {
            await fetchCategories();
          }}
        />
      </Modal>

      {/* Document Modal */}
      <Modal
        show={isModalOpen("docModal")}
        onClose={() => toggleModal("docModal", false)}
        title={editDoc ? "Dokumentum szerkesztés" : "Új dokumentum"}
        height={"auto"}
        width={"720px"}>
        <DocumentModal
          mode={editDoc ? "edit" : "create"}
          categoryId={selectedCategoryId ?? ""}
          doc={editDoc ?? undefined}
          onClose={() => toggleModal("docModal", false)}
          onSaved={async (createdId?: string) => {
            await fetchDocs();
            if (createdId) router.push(`/admin/qms/document/${createdId}`);
          }}
        />
      </Modal>

      {/* ACL Modal */}
      <Modal
        show={isModalOpen("aclModal")}
        onClose={() => toggleModal("aclModal", false)}
        title={aclObject?.title ?? "ACL"}
        height={"auto"}
        width={"860px"}>
        {aclObject ? (
          <QmsAclEditorModal
            objectType={aclObject.objectType}
            objectId={aclObject.objectId}
            onClose={() => toggleModal("aclModal", false)}
          />
        ) : null}
      </Modal>
    </main>
  );
}
