"use client";

import { useCallback, useEffect, useState } from "react";
import toast from "react-hot-toast";
import { useRouter } from "next/navigation";
import ApiCall from "@/api/api-calls";
import Modal from "@/components/shared/modal/Modal";
import useModalMap from "@/hooks/useModalMap";
import PackageModal from "@/components/qms/PackageModal";

type PackageRow = {
  id: string;
  name: string;
  description: string | null;
  is_active: boolean;
  item_count?: number;
  created_at?: string;
  updated_at?: string;
};

export default function TrainingPackagesPage() {
  const router = useRouter();
  const { isModalOpen, toggleModal } = useModalMap();

  const [items, setItems] = useState<PackageRow[]>([]);
  const [loading, setLoading] = useState(false);
  const [edit, setEdit] = useState<PackageRow | null>(null);

  const load = useCallback(async () => {
    setLoading(true);
    try {
      const r = await ApiCall.qms.listPackages();
      setItems((r.data?.items ?? []) as PackageRow[]);
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Csomagok betöltése sikertelen");
    } finally {
      setLoading(false);
    }
  }, []);

  useEffect(() => {
    load();
  }, [load]);

  return (
    <main className="p-4 space-y-3">
      <div className="flex items-center gap-2">
        <h1 className="text-lg font-semibold">Training packages</h1>
        <div className="ml-auto flex items-center gap-2">
          <button className="px-2 py-1 rounded border" onClick={load}>
            Frissít
          </button>
          <button
            className="px-2 py-1 rounded bg-emerald-600 text-white"
            onClick={() => {
              setEdit(null);
              toggleModal("pkgModal", true);
            }}>
            + Package
          </button>
        </div>
      </div>

      {loading ? (
        <div className="py-6 text-sm text-gray-500">Betöltés…</div>
      ) : (
        <div className="overflow-auto border rounded">
          <table className="min-w-full text-sm">
            <thead className="text-left border-b bg-gray-50">
              <tr>
                <th className="py-2 px-3">Név</th>
                <th className="py-2 px-3">Aktív</th>
                <th className="py-2 px-3">Items</th>
                <th className="py-2 px-3 text-right">Művelet</th>
              </tr>
            </thead>
            <tbody>
              {items.map((p) => (
                <tr key={p.id} className="border-b hover:bg-gray-50">
                  <td className="py-2 px-3">
                    <div className="font-medium">{p.name}</div>
                    <div className="text-xs text-gray-500">
                      {p.description ?? "—"}
                    </div>
                  </td>
                  <td className="py-2 px-3">{p.is_active ? "yes" : "no"}</td>
                  <td className="py-2 px-3">{p.item_count ?? "—"}</td>
                  <td className="py-2 px-3">
                    <div className="flex justify-end gap-2">
                      <button
                        className="px-2 py-1 rounded border"
                        onClick={() => {
                          setEdit(p);
                          toggleModal("pkgModal", true);
                        }}>
                        Szerkeszt
                      </button>
                      <button
                        className="px-2 py-1 rounded bg-gray-800 text-white"
                        onClick={() =>
                          router.push(`/admin/qms/training/packages/${p.id}`)
                        }>
                        Megnyit
                      </button>
                    </div>
                  </td>
                </tr>
              ))}

              {!items.length && (
                <tr>
                  <td colSpan={4} className="py-8 text-center text-gray-500">
                    Nincs csomag.
                  </td>
                </tr>
              )}
            </tbody>
          </table>
        </div>
      )}

      <Modal
        show={isModalOpen("pkgModal")}
        onClose={() => toggleModal("pkgModal", false)}
        title={edit ? `Package • ${edit.name}` : "Új package"}
        height="auto"
        width="720px">
        <PackageModal
          mode={edit ? "edit" : "create"}
          pkg={edit ?? undefined}
          onClose={() => toggleModal("pkgModal", false)}
          onSaved={async (createdId?: string) => {
            await load();
            if (createdId)
              router.push(`/admin/qms/training/packages/${createdId}`);
          }}
        />
      </Modal>
    </main>
  );
}
