"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import toast from "react-hot-toast";
import { useRouter } from "next/navigation";
import ApiCall from "@/api/api-calls";
import StatusBadge from "@/components/qms/StatusBadge";

type EnrollmentRow = {
  id: string;
  status: string;
  due_at: string | null;

  package_name?: string;
  assignment_due_at?: string | null;

  // depending on your listEnrollments payload
  required_count?: number;
  completed_required_count?: number;
  is_overdue?: boolean;
};

export default function QmsTrainingPage() {
  const router = useRouter();

  const [items, setItems] = useState<EnrollmentRow[]>([]);
  const [loading, setLoading] = useState(false);

  const load = useCallback(async () => {
    setLoading(true);
    try {
      const r = await ApiCall.qms.listEnrollments({ mine: true });
      setItems((r.data?.items ?? []) as EnrollmentRow[]);
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Képzések betöltése sikertelen");
    } finally {
      setLoading(false);
    }
  }, []);

  useEffect(() => {
    load();
  }, [load]);

  const overdueCount = useMemo(
    () => items.filter((x) => x.is_overdue).length,
    [items]
  );

  return (
    <main className="p-4 space-y-3">
      <div className="flex items-center gap-2 flex-wrap">
        <h1 className="text-lg font-semibold">My training</h1>
        <div className="ml-auto flex items-center gap-2">
          <button className="px-2 py-1 rounded border" onClick={load}>
            Frissít
          </button>
        </div>
      </div>

      <div className="text-sm text-gray-600">
        Összes: <b>{items.length}</b> • Overdue: <b>{overdueCount}</b>
      </div>

      {loading ? (
        <div className="py-6 text-sm text-gray-500">Betöltés…</div>
      ) : (
        <div className="overflow-auto border rounded">
          <table className="min-w-full text-sm">
            <thead className="text-left border-b bg-gray-50">
              <tr>
                <th className="py-2 px-3">Csomag</th>
                <th className="py-2 px-3">Státusz</th>
                <th className="py-2 px-3">Határidő</th>
                <th className="py-2 px-3">Progress</th>
                <th className="py-2 px-3 text-right">Művelet</th>
              </tr>
            </thead>
            <tbody>
              {items.map((x) => (
                <tr key={x.id} className="border-b hover:bg-gray-50">
                  <td className="py-2 px-3">
                    <div className="font-medium">{x.package_name ?? "—"}</div>
                    {x.is_overdue ? (
                      <div className="text-xs text-red-600">Overdue</div>
                    ) : null}
                  </td>
                  <td className="py-2 px-3">
                    <StatusBadge value={x.status} />
                  </td>
                  <td className="py-2 px-3">
                    {x.due_at ?? x.assignment_due_at ?? "—"}
                  </td>
                  <td className="py-2 px-3">
                    {x.required_count != null ? (
                      <span className="text-xs">
                        {x.completed_required_count ?? 0}/{x.required_count}
                      </span>
                    ) : (
                      <span className="text-xs text-gray-500">—</span>
                    )}
                  </td>
                  <td className="py-2 px-3">
                    <div className="flex justify-end">
                      <button
                        className="px-2 py-1 rounded bg-gray-800 text-white"
                        onClick={() =>
                          router.push(`/admin/qms/training/enrollment/${x.id}`)
                        }>
                        Megnyit
                      </button>
                    </div>
                  </td>
                </tr>
              ))}

              {!items.length && (
                <tr>
                  <td colSpan={5} className="py-8 text-center text-gray-500">
                    Nincs kiosztott képzés.
                  </td>
                </tr>
              )}
            </tbody>
          </table>
        </div>
      )}
    </main>
  );
}
