"use client";
import ApiCall from "@/api/api-calls";
import { useAuth } from "@/context/AuthContext";
import { useState, useEffect } from "react";
import Cookies from "js-cookie";
import { useRouter } from "next/navigation";
import { ensureTokenFresh } from "@/lib/ensureTokenFresh";

const DeviceManagement = () => {
  const { user, logout } = useAuth();
  const [devices, setDevices] = useState([]);
  const [status, setStatus] = useState(null);
  const token = Cookies.get("edSId");
  const router = useRouter();

  useEffect(() => {
    if (user) setDevices(user.activeDevices);
  }, [user, router]);

  const handleSecureAccount = async () => {
    try {
      const res = await ApiCall.auth.secureAccount(
        user.details.user.email.value
      );
      setStatus(res.message);
      router.push("/");
    } catch (error) {
      setStatus(error.message || "Something went wrong.");
      router.push("/"); // optional fallback
    }
  };

  const logoutDevice = async (sessionId) => {
    try {
      if (sessionId === user.currentSessionId) {
        const confirmLogout = window.confirm(
          "This is your current session. Log out?"
        );
        if (!confirmLogout) return;
      }

      await ApiCall.auth.postDeviceLogout(sessionId);
      setDevices(devices.filter((d) => d.sessionId !== sessionId));

      if (sessionId === user.currentSessionId) {
        logout();
        router.push("/");
      }
    } catch (error) {
      setStatus(error.message || "Logout device failed.");
    }
  };

  return (
    <div>
      <h2>Aktív bejelentkezések</h2>
      <p
        className="text-red-500 font-semibold cursor-pointer py-4"
        onClick={handleSecureAccount}>
        Kijelentkezés az összes eszközről
      </p>
      {status ? <p>{status}</p> : null}
      <table className="w-full border-collapse border border-gray-300 text-center">
        <thead>
          <tr>
            <th>Device</th>
            <th>Browser</th>
            <th>OS</th>
            <th>IP</th>
            <th>Login Time</th>
            <th>Actions</th>
          </tr>
        </thead>
        <tbody>
          {devices?.map((device) => (
            <tr key={device.sessionId} className="border border-gray-300">
              <td>{device.deviceType}</td>
              <td>{device.browser}</td>
              <td>{device.os}</td>
              <td>{device.ip}</td>
              <td>{new Date(device.createdAt).toLocaleString()}</td>
              <td className="flex flex-col justify-center items-center">
                {device.sessionId === user.currentSessionId
                  ? "(Current Session)"
                  : ""}
                <button
                  className="px-4 py-2 bg-red-500 text-white rounded m-2"
                  onClick={() => logoutDevice(device.sessionId)}>
                  {device.sessionId === user.currentSessionId
                    ? "Logout (Self)"
                    : "Logout"}
                </button>
              </td>
            </tr>
          ))}
        </tbody>
      </table>
    </div>
  );
};

export default DeviceManagement;
