"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";
import Link from "next/link";
import { slugifyTitle, timestampForFilename } from "@/utils/exportFileSlugify";

import ApiCall from "@/api/api-calls";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";

type Row = {
  id: string;
  title: string;
  created_at?: string | null;
  updated_at?: string | null;
  template_type_id?: string | null;
  template_kind_code?: string | null;
  template_kind_label?: string | null;
  header_template_id?: string | null;
  footer_template_id?: string | null;
  header_template_name?: string | null;
  footer_template_name?: string | null;
  header_image_name?: string | null;
  footer_image_name?: string | null;
  is_active: boolean;
};

type DocTemplateKind = {
  id: string;
  code: string;
  label: string;
};

const DEFAULT_KIND_CODES = "generic,medical-centers, it, mir";

export default function TemplateDocumentListPage() {
  const router = useRouter();

  const [rows, setRows] = useState<Row[]>([]);
  const [loading, setLoading] = useState(true);
  const [search, setSearch] = useState("");

  const [selectedIds, setSelectedIds] = useState<string[]>([]);
  const [clearSelectionToggle, setClearSelectionToggle] = useState(false);

  const [onlyActive, setOnlyActive] = useState<"all" | "yes" | "no">("all");
  const [kindFilter, setKindFilter] = useState<string>(""); // '' = all within DEFAULT_KIND_CODES

  const [kinds, setKinds] = useState<DocTemplateKind[]>([]);
  const [kindsLoading, setKindsLoading] = useState(false);

  const fetchData = useCallback(async () => {
    setLoading(true);
    try {
      const params: any = {};

      // kind_code: if specific kind selected, use that; otherwise default set
      if (kindFilter) {
        params.kind_code = kindFilter;
      } else {
        params.kind_code = DEFAULT_KIND_CODES;
      }

      if (onlyActive !== "all") {
        params.is_active = onlyActive; // backend interprets yes/no
      }

      const res = await ApiCall.templateDocuments.getList(params);
      if (res.status !== 200) throw new Error("Failed to load list");
      setRows(res.data ?? []);
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Lista betöltése sikertelen");
    } finally {
      setLoading(false);
    }
  }, [onlyActive, kindFilter]);

  // Load list on filters change
  useEffect(() => {
    fetchData();
  }, [fetchData]);

  // Load kinds for the "Típus" select
  useEffect(() => {
    (async () => {
      try {
        setKindsLoading(true);
        const res = await ApiCall.docTemplateKinds.getList({
          onlyActive: "yes",
        });
        setKinds(res.data ?? []);
      } catch (e: any) {
        console.error(e);
        toast.error(
          e?.response?.data?.error ?? "Sablon típusok betöltése sikertelen"
        );
      } finally {
        setKindsLoading(false);
      }
    })();
  }, []);

  const filteredRows = useMemo(() => {
    const term = search.trim().toLowerCase();
    if (!term) return rows;
    return rows.filter((r) => r.title.toLowerCase().includes(term));
  }, [rows, search]);

  const handleDownloadDocx = useCallback(async (templateDocument: Row) => {
    try {
      const res = await ApiCall.templateDocuments.exportDocx(
        templateDocument.id
      );
      const blob = new Blob([res.data], {
        type: "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
      });

      const url = window.URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;

      const slug = slugifyTitle(templateDocument.title);
      const ts = timestampForFilename();
      a.download = `${slug}-${ts}.docx`;

      document.body.appendChild(a);
      a.click();
      a.remove();
      window.URL.revokeObjectURL(url);
    } catch (e) {
      console.error(e);
      toast.error("DOCX letöltés sikertelen");
    }
  }, []);

  const ellipsis = (text?: string | null) => {
    const v = String(text ?? "");
    return (
      <div
        title={v}
        style={{
          overflow: "hidden",
          whiteSpace: "nowrap",
          textOverflow: "ellipsis",
        }}>
        {v}
      </div>
    );
  };

  const handleSelectedRowsChange = (state: { selectedRows: Row[] }) => {
    setSelectedIds(state.selectedRows.map((r) => r.id));
  };

  const handleBulkDelete = useCallback(async () => {
    if (!selectedIds.length) return;

    if (
      !confirm(
        `Biztosan törölni szeretnéd a kijelölt ${selectedIds.length} sablont?`
      )
    ) {
      return;
    }

    try {
      const res = await ApiCall.templateDocuments.removeMany(selectedIds);
      const { deleted, blocked, error } = res.data || {};

      if (deleted?.length) {
        toast.success(`Törölve: ${deleted.length} sablon`);
      }
      if (blocked?.length) {
        toast.error(
          error ||
            `Nem törölhető: ${blocked.length} sablon (még használatban vannak)`
        );
      }

      setSelectedIds([]);
      setClearSelectionToggle((prev) => !prev);
      fetchData();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Tömeges törlés sikertelen");
    }
  }, [selectedIds, fetchData]);

  const handleDeleteOne = useCallback(
    async (row: Row) => {
      if (!confirm(`Biztosan törlöd ezt a sablont?\n\n${row.title}`)) {
        return;
      }

      try {
        const res = await ApiCall.templateDocuments.removeOne(row.id);
        const data = res.data || {};

        if (data?.blocked?.length) {
          toast.error(
            data.error ??
              "A sablon nem törölhető, mert jelenleg használatban van."
          );
          return;
        }

        toast.success("Sablon törölve");
        fetchData();
      } catch (e: any) {
        console.error(e);
        toast.error(e?.response?.data?.error ?? "Törlés sikertelen");
      }
    },
    [fetchData]
  );

  const columns = useMemo(
    () => [
      {
        id: "title",
        name: "Cím",
        selector: (r: Row) => r.title,
        cell: (r: Row) => (
          <button
            type="button"
            onClick={() =>
              router.push(`/admin/template-documents/${r.id}/view`)
            }
            className="text-blue-600 hover:underline text-left cursor-pointer">
            {r.title}
          </button>
        ),
        sortable: true,
        grow: 2,
      },
      {
        id: "kind",
        name: "Típus",
        selector: (r: Row) => r.template_kind_label,
        cell: (r: Row) => ellipsis(r.template_kind_label),
        sortable: true,
        width: "160px",
      },
      {
        id: "header_image",
        name: "Fejléc",
        selector: (r: Row) => r.header_image_name ?? "",
        cell: (r: Row) =>
          r.header_image_name ? (
            <img
              src={ApiCall.mediaStore.getResized(
                "/300/200/0",
                r.header_image_name,
                { format: "webp", mode: "pixel" }
              )}
              alt="Fejléc"
              className="max-h-[100px] w-full rounded border border-gray-300 object-cover"
            />
          ) : (
            <span className="text-xs text-gray-400">Nincs</span>
          ),
        sortable: false,
        width: "300px",
      },
      {
        id: "footer_image",
        name: "Lábléc",
        selector: (r: Row) => r.footer_image_name ?? "",
        cell: (r: Row) =>
          r.footer_image_name ? (
            <img
              src={ApiCall.mediaStore.getResized(
                "/300/200/0",
                r.footer_image_name,
                { format: "webp", mode: "pixel" }
              )}
              alt="Lábléc"
              className="max-h-[100px] w-full rounded border border-gray-300 object-cover"
            />
          ) : (
            <span className="text-xs text-gray-400">Nincs</span>
          ),
        sortable: false,
        width: "300px",
      },
      {
        id: "is_active",
        name: "Aktív",
        selector: (r: Row) => (r.is_active ? "Igen" : "Nem"),
        cell: (r: Row) => (
          <span className={r.is_active ? "text-emerald-600" : "text-gray-400"}>
            {r.is_active ? "Igen" : "Nem"}
          </span>
        ),
        sortable: true,
        width: "100px",
      },
      {
        id: "updated_at",
        name: "Módosítva",
        selector: (r: Row) => r.updated_at ?? "",
        cell: (r: Row) => (
          <span className="text-xs text-gray-600">
            {r.updated_at ? new Date(r.updated_at).toLocaleDateString() : "-"}
          </span>
        ),
        sortable: true,
        width: "140px",
      },
      {
        id: "actions",
        name: "Műveletek",
        selector: () => "",
        cell: (r: Row) => (
          <div className="flex flex-wrap justify-end gap-2">
            <button
              type="button"
              onClick={() =>
                router.push(`/admin/template-documents/${r.id}/edit`)
              }
              className="px-2 py-1 text-xs rounded border border-gray-300 hover:bg-gray-50">
              Szerkesztés
            </button>
            <button
              type="button"
              onClick={() =>
                router.push(`/admin/template-documents/${r.id}/send`)
              }
              className="px-2 py-1 text-xs rounded border border-blue-300 text-blue-700 hover:bg-blue-50">
              Kitöltés / Küldés
            </button>
            <button
              type="button"
              onClick={() => handleDownloadDocx(r)}
              className="px-2 py-1 text-xs rounded border border-gray-300 hover:bg-gray-50">
              DOCX letöltés
            </button>
            <button
              type="button"
              onClick={() => handleDeleteOne(r)}
              className="px-2 py-1 text-xs rounded border border-red-300 text-red-700 hover:bg-red-50">
              Törlés
            </button>
          </div>
        ),
        ignoreRowClick: true,
        width: "280px",
      },
    ],
    [router, handleDownloadDocx, handleDeleteOne]
  );

  return (
    <main className="p-4 space-y-4">
      <div className="flex flex-wrap items-center gap-3">
        <div>
          <h1 className="text-lg font-semibold">Általános sablonok</h1>
          <p className="text-sm text-gray-600">
            Fejléces dokumentum sablonok általános felhasználásra.
          </p>
        </div>

        {/* Filters */}
        <div className="flex flex-wrap items-center gap-2 ml-auto">
          <select
            className="border rounded px-3 py-2 text-sm"
            value={onlyActive}
            onChange={(e) => setOnlyActive(e.target.value as any)}>
            <option value="all">Aktív: mind</option>
            <option value="yes">Csak aktív</option>
            <option value="no">Csak inaktív</option>
          </select>

          <select
            className="border rounded px-3 py-2 text-sm min-w-[160px]"
            value={kindFilter}
            onChange={(e) => setKindFilter(e.target.value)}>
            <option value="">Típus: összes ({DEFAULT_KIND_CODES})</option>
            {kinds.map((k) => (
              <option key={k.id} value={k.code}>
                {k.label} ({k.code})
              </option>
            ))}
          </select>

          {kindsLoading && (
            <span className="text-xs text-gray-500">Típusok…</span>
          )}

          {selectedIds.length > 0 && (
            <button
              onClick={handleBulkDelete}
              className="px-3 py-2 text-sm rounded bg-red-600/80 hover:bg-red-600 text-white">
              Kijelöltek törlése ({selectedIds.length})
            </button>
          )}

          <input
            className="border rounded px-3 py-2 text-sm"
            placeholder="Keresés cím alapján…"
            value={search}
            onChange={(e) => setSearch(e.target.value)}
          />

          <Link href="/admin/template-documents/new/edit">
            <button className="px-3 py-2 text-sm rounded bg-emerald-600/80 hover:bg-emerald-600 text-white">
              + Új sablon
            </button>
          </Link>
        </div>
      </div>

      <DataTableWrapper
        keyField="id"
        tableTitle="Dokumentum sablon lista"
        data={filteredRows}
        columns={columns as any}
        progressPending={loading}
        pagination
        defaultSortFieldId="title"
        selectableRows
        onSelectedRowsChange={handleSelectedRowsChange}
        clearSelectedRows={clearSelectionToggle}
      />
    </main>
  );
}

// "use client";

// import { useCallback, useEffect, useMemo, useState } from "react";
// import { useRouter } from "next/navigation";
// import toast from "react-hot-toast";
// import Link from "next/link";
// import { slugifyTitle, timestampForFilename } from "@/utils/exportFileSlugify";

// import ApiCall from "@/api/api-calls";
// import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";

// type Row = {
//   id: string;
//   title: string;
//   created_at?: string | null;
//   updated_at?: string | null;
//   template_type_id?: string | null;
//   template_kind_code?: string | null;
//   template_kind_label?: string | null;
//   header_template_id?: string | null;
//   footer_template_id?: string | null;
//   header_template_name?: string | null;
//   footer_template_name?: string | null;
//   header_image_name?: string | null;
//   footer_image_name?: string | null;
//   is_active: boolean;
// };

// export default function TemplateDocumentListPage() {
//   const router = useRouter();

//   const [rows, setRows] = useState<Row[]>([]);
//   const [loading, setLoading] = useState(true);
//   const [search, setSearch] = useState("");

//   const [selectedIds, setSelectedIds] = useState<string[]>([]);
//   const [clearSelectionToggle, setClearSelectionToggle] = useState(false);

//   const fetchData = useCallback(async () => {
//     setLoading(true);
//     try {
//       const res = await ApiCall.templateDocuments.getList({
//         kind_code: "generic, medical-centers, it",
//       });
//       if (res.status !== 200) throw new Error("Failed to load list");
//       setRows(res.data ?? []);
//     } catch (e: any) {
//       console.error(e);
//       toast.error(e?.response?.data?.error ?? "Lista betöltése sikertelen");
//     } finally {
//       setLoading(false);
//     }
//   }, []);

//   useEffect(() => {
//     fetchData();
//   }, [fetchData]);

//   const filteredRows = useMemo(() => {
//     const term = search.trim().toLowerCase();
//     if (!term) return rows;
//     return rows.filter((r) => r.title.toLowerCase().includes(term));
//   }, [rows, search]);

//   const handleDownloadDocx = useCallback(async (templateDocument: Row) => {
//     try {
//       const res = await ApiCall.templateDocuments.exportDocx(
//         templateDocument.id
//       );
//       const blob = new Blob([res.data], {
//         type: "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
//       });

//       const url = window.URL.createObjectURL(blob);
//       const a = document.createElement("a");
//       a.href = url;

//       const slug = slugifyTitle(templateDocument.title);
//       const ts = timestampForFilename();
//       a.download = `${slug}-${ts}.docx`;

//       document.body.appendChild(a);
//       a.click();
//       a.remove();
//       window.URL.revokeObjectURL(url);
//     } catch (e) {
//       console.error(e);
//       toast.error("DOCX letöltés sikertelen");
//     }
//   }, []);

//   const ellipsis = (text?: string | null) => {
//     const v = String(text ?? "");
//     return (
//       <div
//         title={v}
//         style={{
//           overflow: "hidden",
//           whiteSpace: "nowrap",
//           textOverflow: "ellipsis",
//         }}>
//         {v}
//       </div>
//     );
//   };

//   const handleSelectedRowsChange = (state: { selectedRows: Row[] }) => {
//     setSelectedIds(state.selectedRows.map((r) => r.id));
//   };

//   const handleBulkDelete = useCallback(async () => {
//     if (!selectedIds.length) return;

//     if (
//       !confirm(
//         `Biztosan törölni szeretnéd a kijelölt ${selectedIds.length} sablont?`
//       )
//     ) {
//       return;
//     }

//     try {
//       const res = await ApiCall.templateDocuments.removeMany(selectedIds);
//       const { deleted, blocked, error } = res.data || {};

//       if (deleted?.length) {
//         toast.success(`Törölve: ${deleted.length} sablon`);
//       }
//       if (blocked?.length) {
//         toast.error(
//           error ||
//             `Nem törölhető: ${blocked.length} sablon (még használatban vannak)`
//         );
//       }

//       setSelectedIds([]);
//       setClearSelectionToggle((prev) => !prev);
//       fetchData();
//     } catch (e: any) {
//       console.error(e);
//       toast.error(e?.response?.data?.error ?? "Tömeges törlés sikertelen");
//     }
//   }, [selectedIds, fetchData]);

//   const handleDeleteOne = useCallback(
//     async (row: Row) => {
//       if (!confirm(`Biztosan törlöd ezt a sablont?\n\n${row.title}`)) {
//         return;
//       }

//       try {
//         // you can use bulk endpoint with [id] or the single delete
//         const res = await ApiCall.templateDocuments.removeOne(row.id);
//         const data = res.data || {};

//         if (data?.blocked?.length) {
//           toast.error(
//             data.error ??
//               "A sablon nem törölhető, mert jelenleg használatban van."
//           );
//           return;
//         }

//         toast.success("Sablon törölve");
//         fetchData();
//       } catch (e: any) {
//         console.error(e);
//         toast.error(e?.response?.data?.error ?? "Törlés sikertelen");
//       }
//     },
//     [fetchData]
//   );

//   const columns = useMemo(
//     () => [
//       {
//         id: "title",
//         name: "Cím",
//         selector: (r: Row) => r.title,
//         cell: (r: Row) => (
//           <button
//             type="button"
//             onClick={() =>
//               router.push(`/admin/template-documents/${r.id}/view`)
//             }
//             className="text-blue-600 hover:underline text-left cursor-pointer">
//             {r.title}
//           </button>
//         ),
//         sortable: true,
//         grow: 2,
//       },
//       {
//         id: "kind",
//         name: "Típus",
//         selector: (r: Row) => r.template_kind_label,
//         cell: (r: Row) => ellipsis(r.template_kind_label),
//         sortable: true,
//         width: "140px",
//       },
//       {
//         id: "header_image",
//         name: "Fejléc",
//         selector: (r: Row) => r.header_image_name ?? "",
//         cell: (r: Row) =>
//           r.header_image_name ? (
//             <img
//               src={ApiCall.mediaStore.getResized(
//                 "/300/200/0",
//                 r.header_image_name,
//                 { format: "webp", mode: "pixel" }
//               )}
//               alt="Fejléc"
//               className="max-h-[100px] w-full rounded border border-gray-300 object-cover"
//             />
//           ) : (
//             <span className="text-xs text-gray-400">Nincs</span>
//           ),
//         sortable: false,
//         width: "300px",
//       },
//       {
//         id: "footer_image",
//         name: "Lábléc",
//         selector: (r: Row) => r.footer_image_name ?? "",
//         cell: (r: Row) =>
//           r.footer_image_name ? (
//             <img
//               src={ApiCall.mediaStore.getResized(
//                 "/300/200/0",
//                 r.footer_image_name,
//                 { format: "webp", mode: "pixel" }
//               )}
//               alt="Lábléc"
//               className="max-h-[100px] w-full rounded border border-gray-300 object-cover"
//             />
//           ) : (
//             <span className="text-xs text-gray-400">Nincs</span>
//           ),
//         sortable: false,
//         width: "300px",
//       },
//       {
//         id: "is_active",
//         name: "Aktív",
//         selector: (r: Row) => (r.is_active ? "Igen" : "Nem"),
//         cell: (r: Row) => (
//           <span className={r.is_active ? "text-emerald-600" : "text-gray-400"}>
//             {r.is_active ? "Igen" : "Nem"}
//           </span>
//         ),
//         sortable: true,
//         width: "100px",
//       },
//       {
//         id: "updated_at",
//         name: "Módosítva",
//         selector: (r: Row) => r.updated_at ?? "",
//         cell: (r: Row) => (
//           <span className="text-xs text-gray-600">
//             {r.updated_at ? new Date(r.updated_at).toLocaleDateString() : "-"}
//           </span>
//         ),
//         sortable: true,
//         width: "140px",
//       },
//       {
//         id: "actions",
//         name: "Műveletek",
//         selector: () => "",
//         cell: (r: Row) => (
//           <div className="flex justify-end gap-2">
//             <button
//               type="button"
//               onClick={() =>
//                 router.push(`/admin/template-documents/${r.id}/edit`)
//               }
//               className="px-2 py-1 text-xs rounded border border-gray-300 hover:bg-gray-50">
//               Szerkesztés
//             </button>
//             <button
//               type="button"
//               onClick={() => handleDownloadDocx(r)}
//               className="px-2 py-1 text-xs rounded border border-gray-300 hover:bg-gray-50">
//               DOCX letöltés
//             </button>
//             <button
//               type="button"
//               onClick={() => handleDeleteOne(r)}
//               className="px-2 py-1 text-xs rounded border border-red-300 text-red-700 hover:bg-red-50">
//               Törlés
//             </button>
//           </div>
//         ),
//         ignoreRowClick: true,
//         width: "280px",
//       },
//     ],
//     [router, handleDownloadDocx, handleDeleteOne]
//   );

//   return (
//     <main className="p-4 space-y-4">
//       <div className="flex flex-wrap items-center gap-3">
//         <div>
//           <h1 className="text-lg font-semibold">Általános sablonok</h1>
//           <p className="text-sm text-gray-600">
//             Fejléces dokumentum sablonok általános felhasználásra.
//           </p>
//         </div>

//         <div className="flex items-center gap-2 ml-auto">
//           {selectedIds.length > 0 && (
//             <button
//               onClick={handleBulkDelete}
//               className="px-3 py-2 text-sm rounded bg-red-600/80 hover:bg-red-600 text-white">
//               Kijelöltek törlése ({selectedIds.length})
//             </button>
//           )}

//           <input
//             className="border rounded px-3 py-2 text-sm"
//             placeholder="Keresés cím alapján…"
//             value={search}
//             onChange={(e) => setSearch(e.target.value)}
//           />

//           <Link href="/admin/template-documents/new/edit">
//             <button className="px-3 py-2 text-sm rounded bg-emerald-600/80 hover:bg-emerald-600 text-white">
//               + Új sablon
//             </button>
//           </Link>
//         </div>
//       </div>

//       <DataTableWrapper
//         keyField="id"
//         tableTitle="Dokumentum sablon lista"
//         data={filteredRows}
//         columns={columns as any}
//         progressPending={loading}
//         pagination
//         defaultSortFieldId="title"
//         selectableRows
//         onSelectedRowsChange={handleSelectedRowsChange}
//         clearSelectedRows={clearSelectionToggle}
//       />
//     </main>
//   );
// }
