"use client";

import { useEffect, useMemo, useState, useCallback, useRef } from "react";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ExpandedProductGroups from "@/components/product-groups/expanded/ExpandedProductGroups";
import ApiCall from "@/api/api-calls";
import Loader from "@/components/shared/loader/global-loading";
import Image from "next/image";
import { TextField } from "@mui/material";
import Autocomplete from "@mui/material/Autocomplete";
import useTableDebounce from "@/hooks/useTableDebounce";
import ExpandedUsers from "@/components/users/expanded/ExpandedUsers";

const UserListComponent = ({ tableData, totalRecords, token }) => {
  const [data, setData] = useState(tableData);
  const [page, setPage] = useState(1);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [expandedRow, setExpandedRow] = useState(null);
  const expandedRowRef = useRef(null);
  const [pageSize, setPageSize] = useState(25);
  const [sortField, setSortField] = useState("createdAt");
  const [sortDirection, setSortDirection] = useState("desc");
  const [inputValue, setInputValue] = useState("");
  const debouncedSearchTerm = useTableDebounce(inputValue, 500); // ✅ Use custom hook

  const query = `?page=${page}&pageSize=${pageSize}&sortField=${sortField}&sortDirection=${sortDirection}&searchTerm=${debouncedSearchTerm.trim()}`;

  const fetchData = async () => {
    setLoading(true);
    try {
      const res = await ApiCall.users.getList(query);
      setData(res?.data?.tableData ?? []);
    } catch (err) {
      console.error(err);
    }
    setLoading(false);
  };
  console.log("tableData", tableData);
  useEffect(() => {
    fetchData();
  }, [query]);

  const columns = useMemo(
    () => [
      {
        id: "name",
        name: "Felhasználó neve",
        selector: (row) => (
          <p className="font-bold text-xs whitespace-break-spaces">
            {row.name}
          </p>
        ),
        sortable: true,
      },
      {
        id: "username",
        name: "E-mail cím",
        selector: (row) => {
          return (
            <p className="font-bold text-xs whitespace-break-spaces">
              {row.username}
            </p>
          );
        },
        sortable: true,
      },
      {
        id: "telefon",
        name: "Telefonszám",
        selector: (row) => {
          return (
            <p className="font-bold text-xs whitespace-pre-wrap">
              {row.telefon}
            </p>
          );
        },
        sortable: true,
      },
      {
        id: "iranyitoszam",
        name: "Irányítószám",
        selector: (row) => {
          return (
            <p className="font-bold text-xs whitespace-pre-wrap">
              {row.iranyitoszam}
            </p>
          );
        },
        sortable: true,
      },
      {
        id: "telepules",
        name: "Település",
        selector: (row) => {
          return <p className="font-bold text-xs">{row.telepules}</p>;
        },
        sortable: true,
      },
      {
        id: "ordersList",
        name: "Rendelések száma",
        selector: (row) => {
          return (
            <li className="w-full text-xs text-center">{row.ordersList}</li>
          );
        },
        sortable: true,
      },

      {
        id: "createdAt",
        name: "Létrehozva",
        selector: (row) => row.createdAt,
        sortable: true,
      },
      {
        id: "updatedAt",
        name: "Frissítve",
        selector: (row) => row.updatedAt,
        sortable: true,
      },
    ],
    []
  );

  const handleUserUpdated = () => {
    fetchData(); // Refetches updated data, including updated image
  };
  const renderExpandedGroup = useCallback(
    (data, onUserUpdated) => {
      return (
        <div ref={expandedRowRef}>
          <ExpandedUsers
            id={data?._id}
            token={token}
            onUserUpdated={onUserUpdated}
          />
        </div>
      );
    },
    [token] // Proper dependencies
  );

  const subHeaderComponent = (
    <Autocomplete
      freeSolo
      id="subHeaderComponent"
      className="w-72"
      disableClearable
      options={[]} // could be search suggestions
      inputValue={inputValue} // controlled from the start
      onInputChange={(event, newInputValue) => {
        setInputValue(newInputValue);
      }}
      renderInput={(params) => (
        <TextField
          {...params}
          label="Keresés"
          type="search"
          variant="outlined"
        />
      )}
    />
  );

  return (
    <>
      <div className="flex justify-between items-center mb-4"></div>
      {data && (
        <DataTableWrapper
          tableTitle={
            <p className="flex flex-col">
              Regisztrált felhasználók
              <span className="text-sm font-normal">
                Kereshető mezők: Név, e-mail, cégnév, adószám, irányítószám,
                település, utca, telefonszám
              </span>
            </p>
          }
          data={data}
          columns={columns}
          tableOptionsPath="users/create"
          progressPending={loading}
          paginationDefaultPage={page}
          defaultSortFieldId="createdAt"
          paginationServer
          onChangePage={(pageNumber) => {
            setPage(pageNumber);
          }}
          //   conditionalRowStyles={conditionalRowStyles}
          progressComponent={<Loader />}
          paginationTotalRows={totalRecords}
          paginationPerPage={pageSize}
          onSort={(sortId, direction) => {
            setSortField(sortId.id);
            setSortDirection(direction);
            setPage(1); // reset to first page when sorting
          }}
          onChangeRowsPerPage={(size) => {
            setPageSize(size);
            setPage(1); // reset page
          }}
          onFilterClear={() => fetchData()}
          subHeader={true}
          subHeaderComponent={subHeaderComponent}
          subHeaderAlign="right"
          subHeaderWrap={true}
          expandableRows
          expandableRowExpanded={(rowData) => rowData?._id === expandedRow?._id}
          onRowExpandToggled={(toggleState, rowIndex) => {
            if (toggleState) {
              // A row is being expanded, so close previously expanded row
              if (expandedRow !== null && expandedRow._id !== rowIndex._id) {
                setExpandedRow(null);
              }
              setExpandedRow(rowIndex);
            } else {
              // A row is being collapsed
              setExpandedRow(null);
            }
          }}
          renderExpandableRows={(rowData) =>
            renderExpandedGroup(rowData, handleUserUpdated)
          }
        />
      )}
    </>
  );
};

export default UserListComponent;
