import { write, utils } from "xlsx";

export async function POST(req) {
  try {
    const fd = await req.formData();
    const file = fd.get("file");

    if (!file) {
      return new Response(JSON.stringify({ error: "No file uploaded" }), {
        status: 400,
      });
    }

    const buf = Buffer.from(await file.arrayBuffer());
    const xlsx = require("xlsx");
    const wb = xlsx.read(buf, { type: "buffer" });
    const sheet = wb.Sheets[wb.SheetNames[0]];
    const rows = xlsx.utils.sheet_to_json(sheet);

    // Filter out deleted rows
    const validRows = rows.filter((r) => r.delete === 0);

    // Helpers
    const imgPath = (code) => (code ? `${code}` : "");

    const priceHUF = (r) =>
      r.productPriceType === "EUR"
        ? parseFloat(r.productPieceWebshopPrice || 0) * 400
        : r.productPieceWebshopPrice || 0;

    const prodType = (r) => {
      const hasSize = !!r.sizeOriginal;
      const hasColor = !!r.color;
      if (hasSize && hasColor) return "Méret - Szín alapján";
      if (hasSize) return "Méret alapján";
      if (hasColor) return "Szín alapján";
      return "Alapértelmezett terméktípus";
    };

    const categoryPath = (r) =>
      [
        r.productCategory,
        r.productSubCategory,
        r.productChildCategory,
        r.productChildSubCategory,
      ]
        .filter(Boolean)
        .join("/");

    // Group top-down
    const output = [];
    let currentParent = null;
    let childCounter = 1;

    for (const r of validRows) {
      if (r._id) {
        currentParent = r;

        output.push({
          Cikkszám: r.productItemNumberIdentifier,
          "Kategória név/nevek": categoryPath(r),
          "Terméknév (hu)": r.productGroupTitle || "",
          "Hosszú leírás (hu)": r.productDescription || "",
          "További termékképek": "",
          "Gyártói cikkszám": r.productItemNumberIdentifier,
          "Vonalkód/GTIN Azonosító": r.productEAN || "",
          "Elsődleges termékkép": imgPath(r.imageCode),
          Súly: r.productWeight || 0,
          Alapár: priceHUF(r),
          Gyártó: r.productGroupMark || "",
          Terméktípus: prodType(r),
          "Szülőtermék cikkszáma": "-",
          "Tulajdonság: Méret (a_meret)": r.sizeOriginal || "",
          "Tulajdonság: Szín (a_szin)": r.color || "",
        });
      } else if (currentParent) {
        const childSku = `10000${String(childCounter++).padStart(4, "0")}`;
        console.log("gallery", r.productGallery);
        const gallery =
          r.productGallery?.length > 0
            ? r.productGallery.split(/\s*,\s*/).join("|||")
            : "";
        output.push({
          Cikkszám: childSku,
          "Kategória név/nevek": categoryPath(currentParent),
          "Terméknév (hu)": r.productGroupTitle || "",
          "Hosszú leírás (hu)": r.productMetaDescription || "",
          "További termékképek": gallery,
          "Gyártói cikkszám": r.productItemNumberIdentifier,
          "Vonalkód/GTIN Azonosító": r.productEAN || "",
          "Elsődleges termékkép": imgPath(r.imageCode),
          Súly: r.productWeight || 0,
          Alapár: priceHUF(r),
          Gyártó: r.productGroupMark || "",
          Terméktípus: prodType(r),
          "Szülőtermék cikkszáma": currentParent.productItemNumberIdentifier,
          "Tulajdonság: Méret (a_meret)": r.sizeOriginal || "",
          "Tulajdonság: Szín (a_szin)": r.color || "",
        });
      }
    }

    // Export
    const ws = utils.json_to_sheet(output);
    const wbOut = utils.book_new();
    utils.book_append_sheet(wbOut, ws, "Sheet1");
    const outBuf = write(wbOut, { bookType: "xlsx", type: "buffer" });

    return new Response(outBuf, {
      status: 200,
      headers: {
        "Content-Disposition":
          'attachment; filename="termekcsoportok_b_final.xlsx"',
        "Content-Type":
          "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
      },
    });
  } catch (err) {
    console.error("Conversion error:", err);
    return new Response(JSON.stringify({ error: "Internal Server Error" }), {
      status: 500,
    });
  }
}
