"use client";

import { useEffect, useMemo, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import { getTranslation } from "@/i18n";
import { useRouter } from "next/navigation";

import {
  Card,
  CardBody,
  CardHeader,
  CardTitle,
  CardMeta,
  Chip,
  SecondaryButton,
  Button,
  ToolbarRow,
  EmptyState,
  Divider,
  SectionTitle,
} from "@/components/ui/CardBits";

function fmt(iso?: string | null) {
  if (!iso) return "-";
  try {
    return new Date(iso).toLocaleString();
  } catch {
    return iso ?? "-";
  }
}

type Props = { caseRef: string };

export default function CaseTreeOverview({ caseRef }: Props) {
  const router = useRouter();
  const { t } = getTranslation();

  const [loading, setLoading] = useState(false);
  const [data, setData] = useState<any>(null);

  async function load() {
    setLoading(true);
    try {
      const r = await ApiCall.cases.overviewTree(caseRef);
      setData(r.data);
    } catch (e: any) {
      toast.error(
        e?.response?.data?.message ?? "Összegző betöltése sikertelen"
      );
      setData(null);
    } finally {
      setLoading(false);
    }
  }

  useEffect(() => {
    void load();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [caseRef]);

  // ✅ backend shape:
  const threadNodes = data?.threads ?? [];
  const decisions = data?.decisions ?? [];
  const meetings = data?.meetings ?? []; // flat meeting list
  const agendaItems = data?.meetingsMeta?.agendaItems ?? []; // agenda points

  const totals = useMemo(() => {
    const threadCount = threadNodes.length;
    const taskCount = threadNodes.reduce(
      (acc: number, n: any) => acc + (n.actionItems?.length ?? 0),
      0
    );
    const commentCount = threadNodes.reduce(
      (acc: number, n: any) => acc + (n.comments?.length ?? 0),
      0
    );
    const attachCount = threadNodes.reduce(
      (acc: number, n: any) => acc + (n.attachments?.length ?? 0),
      0
    );
    return { threadCount, taskCount, commentCount, attachCount };
  }, [threadNodes]);

  const isEmpty =
    !loading &&
    !threadNodes.length &&
    !decisions.length &&
    (meetings?.length ?? 0) === 0;

  return (
    <div className="space-y-3">
      <Card>
        <CardHeader>
          <div>
            <CardTitle>
              {t?.("case_tree.title", { defaultValue: "Áttekintő fa" })}
            </CardTitle>
            <CardMeta>
              {t?.("case_tree.subtitle", {
                defaultValue:
                  "Szálak → feladatok → reakciók/dokumentumok • döntések • meetingek/agenda",
              })}
            </CardMeta>
          </div>

          <ToolbarRow>
            <SecondaryButton onClick={() => void load()} disabled={loading}>
              {loading
                ? "..."
                : t?.("common.refresh", { defaultValue: "Frissít" })}
            </SecondaryButton>
            <Button
              onClick={() => router.push(`/admin/cases/${caseRef}/threads`)}>
              + {t?.("case_tree.new_thread", { defaultValue: "Új szál" })}
            </Button>
          </ToolbarRow>
        </CardHeader>

        <CardBody className="flex flex-wrap gap-2">
          <Chip tone="info">szál: {totals.threadCount}</Chip>
          <Chip tone="info">feladat: {totals.taskCount}</Chip>
          <Chip tone="info">reakció: {totals.commentCount}</Chip>
          <Chip tone="info">doksi: {totals.attachCount}</Chip>
          <Chip tone="neutral">döntés: {decisions.length}</Chip>
          <Chip tone="neutral">meeting: {meetings.length}</Chip>
        </CardBody>
      </Card>

      {isEmpty ? (
        <EmptyState
          title="Nincs még tartalom"
          hint="Hozz létre egy szálat, adj hozzá feladatokat, kommenteket vagy csatolmányokat."
        />
      ) : null}

      <Card>
        <CardBody className="space-y-3">
          <SectionTitle>Szálak</SectionTitle>

          {!threadNodes.length ? (
            <div className="text-sm text-slate-600">Nincs szál.</div>
          ) : (
            <div className="grid gap-2">
              {threadNodes.map((node: any) => {
                const th = node.thread;
                const comments = node.comments ?? [];
                const actionItems = node.actionItems ?? [];
                const attachments = node.attachments ?? [];

                return (
                  <details
                    key={th.id}
                    className="rounded-xl border border-slate-200 bg-white">
                    <summary className="cursor-pointer list-none px-4 py-3 flex items-start justify-between gap-3">
                      <div className="min-w-0">
                        <div className="font-semibold text-slate-900 truncate">
                          {th.title}{" "}
                          {th.code ? (
                            <span className="opacity-60">({th.code})</span>
                          ) : null}
                        </div>
                        <div className="text-xs text-slate-500">
                          created: {fmt(th.created_at)} · input due:{" "}
                          {fmt(th.input_due_at)}
                        </div>
                      </div>

                      <div className="flex flex-wrap gap-2 items-center">
                        <Chip tone="info">feladat: {actionItems.length}</Chip>
                        <Chip tone="info">reakció: {comments.length}</Chip>
                        <Chip tone="info">doksi: {attachments.length}</Chip>

                        <SecondaryButton
                          onClick={(e) => {
                            e.preventDefault();
                            e.stopPropagation();
                            router.push(`/admin/threads/${th.code ?? th.id}`);
                          }}>
                          Megnyit
                        </SecondaryButton>
                      </div>
                    </summary>

                    <div className="px-4 pb-4 space-y-3">
                      {th.description ? (
                        <div className="text-sm text-slate-700 whitespace-pre-wrap">
                          {th.description}
                        </div>
                      ) : null}

                      <Divider />

                      <div className="grid gap-3 md:grid-cols-3">
                        <div>
                          <div className="text-sm font-semibold mb-2">
                            Feladatok
                          </div>
                          {actionItems.length ? (
                            <ul className="text-sm text-slate-700 space-y-1">
                              {actionItems.slice(0, 8).map((ai: any) => (
                                <li
                                  key={ai.id}
                                  className="flex items-center justify-between gap-2">
                                  <span className="truncate">
                                    {ai.title ?? ai.name ?? ai.id}
                                  </span>
                                  <span className="text-xs text-slate-500 whitespace-nowrap">
                                    {ai.status ?? ""}
                                  </span>
                                </li>
                              ))}
                              {actionItems.length > 8 ? (
                                <li className="text-xs text-slate-500">
                                  +{actionItems.length - 8} további…
                                </li>
                              ) : null}
                            </ul>
                          ) : (
                            <div className="text-sm text-slate-600">
                              Nincs feladat.
                            </div>
                          )}
                        </div>

                        <div>
                          <div className="text-sm font-semibold mb-2">
                            Reakciók
                          </div>
                          {comments.length ? (
                            <ul className="text-sm text-slate-700 space-y-2">
                              {comments.slice(0, 5).map((c: any) => (
                                <li
                                  key={c.id}
                                  className="border border-slate-200 rounded-lg p-2">
                                  <div className="text-xs text-slate-500">
                                    {c.author_name ?? c.author_id ?? "—"} ·{" "}
                                    {fmt(c.created_at)}
                                  </div>
                                  <div className="mt-1 whitespace-pre-wrap line-clamp-3">
                                    {c.body}
                                  </div>
                                </li>
                              ))}
                              {comments.length > 5 ? (
                                <li className="text-xs text-slate-500">
                                  +{comments.length - 5} további…
                                </li>
                              ) : null}
                            </ul>
                          ) : (
                            <div className="text-sm text-slate-600">
                              Nincs hozzászólás.
                            </div>
                          )}
                        </div>

                        <div>
                          <div className="text-sm font-semibold mb-2">
                            Dokumentumok
                          </div>
                          {attachments.length ? (
                            <ul className="text-sm text-slate-700 space-y-1">
                              {attachments.slice(0, 8).map((a: any) => (
                                <li key={a.id} className="truncate">
                                  {a.original_name ?? a.name ?? a.id}
                                </li>
                              ))}
                              {attachments.length > 8 ? (
                                <li className="text-xs text-slate-500">
                                  +{attachments.length - 8} további…
                                </li>
                              ) : null}
                            </ul>
                          ) : (
                            <div className="text-sm text-slate-600">
                              Nincs csatolmány.
                            </div>
                          )}
                        </div>
                      </div>
                    </div>
                  </details>
                );
              })}
            </div>
          )}

          <Divider />

          <SectionTitle>Döntések</SectionTitle>
          {decisions.length ? (
            <div className="grid gap-2">
              {decisions.slice(0, 10).map((d: any) => (
                <div
                  key={d.id}
                  className="rounded-xl border border-slate-200 p-3 flex items-center justify-between gap-3">
                  <div className="min-w-0">
                    <div className="font-semibold truncate">
                      {d.title}{" "}
                      {d.code ? (
                        <span className="opacity-60">({d.code})</span>
                      ) : null}
                    </div>
                    <div className="text-xs text-slate-500">
                      státusz: {d.status} · created: {fmt(d.created_at)}
                    </div>
                  </div>
                  <SecondaryButton
                    onClick={() =>
                      router.push(`/admin/decisions/${d.code ?? d.id}`)
                    }>
                    Megnyit
                  </SecondaryButton>
                </div>
              ))}
              {decisions.length > 10 ? (
                <div className="text-xs text-slate-500">
                  +{decisions.length - 10} további…
                </div>
              ) : null}
            </div>
          ) : (
            <div className="text-sm text-slate-600">Nincs döntés.</div>
          )}

          <Divider />

          <SectionTitle>Meeting agenda</SectionTitle>
          {agendaItems.length ? (
            <div className="grid gap-2">
              {agendaItems.slice(0, 12).map((a: any) => (
                <div
                  key={a.id}
                  className="rounded-xl border border-slate-200 p-3 flex items-center justify-between gap-3">
                  <div className="min-w-0">
                    <div className="font-semibold truncate">
                      {a.title ?? "Agenda pont"}
                      <span className="opacity-60">
                        {" "}
                        · #{a.sort_order ?? "-"}
                      </span>
                    </div>
                    <div className="text-xs text-slate-500">
                      meeting:{" "}
                      {a.meeting_title ?? a.meeting_code ?? a.meeting_id} ·{" "}
                      {fmt(a.starts_at ?? a.meeting_created_at)}
                    </div>
                  </div>

                  <SecondaryButton
                    onClick={() =>
                      router.push(
                        `/admin/meetings/${a.meeting_code ?? a.meeting_id}`
                      )
                    }>
                    Meeting
                  </SecondaryButton>
                </div>
              ))}
              {agendaItems.length > 12 ? (
                <div className="text-xs text-slate-500">
                  +{agendaItems.length - 12} további agenda…
                </div>
              ) : null}
            </div>
          ) : (
            <div className="text-sm text-slate-600">Nincs meeting agenda.</div>
          )}
        </CardBody>
      </Card>
    </div>
  );
}
