"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import AsyncSelect from "react-select/async";
import ApiCall from "@/api/api-calls";

export type CompanySegmentUpsertPayload = {
  name: string;
  slug: string;
  normalized_name: string;
  company_id?: string | null;
};

export default function CompanySegmentForm({
  initial,
  onSubmit,
}: {
  initial?: any;
  onSubmit: (payload: CompanySegmentUpsertPayload) => void | Promise<void>;
}) {
  const [name, setName] = useState(initial?.name ?? "");
  const [slug, setSlug] = useState(initial?.slug ?? "");
  const [normalizedName, setNormalizedName] = useState(
    initial?.normalized_name ?? initial?.normalizedName ?? ""
  );

  const [companyId, setCompanyId] = useState<string | null>(
    initial?.company_id ?? null
  );
  const [companyLabel, setCompanyLabel] = useState<string>(
    initial?.company_name ?? ""
  );

  // simple slugify helper
  const slugify = (v: string) =>
    v
      .trim()
      .toLowerCase()
      .replace(/\s+/g, "-")
      .replace(/[^a-z0-9-]+/g, "")
      .replace(/--+/g, "-")
      .replace(/^-+|-+$/g, "");

  // You can replace this with a proper search endpoint if you have one
  const loadCompanyOptions = useCallback(async (input: string) => {
    const res = await ApiCall.ownCompanies.getAllNames();
    const items = res.data?.companies ?? [];
    const lower = (input || "").toLowerCase();
    return items
      .filter(
        (o: any) =>
          !lower ||
          String(o.label).toLowerCase().includes(lower) ||
          String(o.value).toLowerCase().includes(lower)
      )
      .map((o: any) => ({
        value: o.value,
        label: o.label,
      }));
  }, []);

  useEffect(() => {
    if (!initial) return;
    setName(initial.name ?? "");
    setSlug(initial.slug ?? "");
    setNormalizedName(initial.normalized_name ?? initial.normalizedName ?? "");
    setCompanyId(initial.company_id ?? null);
    setCompanyLabel(initial.company_name ?? "");
  }, [initial]);

  // auto-generate slug + normalized if empty
  useEffect(() => {
    if (!name) return;
    if (!slug) setSlug(slugify(name));
    if (!normalizedName)
      setNormalizedName(name.toLowerCase().replace(/\s+/g, " "));
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [name]);

  const canSubmit = useMemo(() => name.trim().length > 1, [name]);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    const payload: CompanySegmentUpsertPayload = {
      name: name.trim(),
      slug: slug.trim() || slugify(name),
      normalized_name:
        normalizedName.trim() || name.trim().toLowerCase().replace(/\s+/g, " "),
      company_id: companyId ?? null,
    };
    await onSubmit(payload);
  };

  return (
    <form
      onSubmit={handleSubmit}
      className="grid grid-cols-1 md:grid-cols-2 gap-3">
      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Szegmens név</label>
        <input
          className="border rounded px-3 py-2"
          value={name}
          onChange={(e) => setName(e.target.value)}
          placeholder="pl. MARKETING"
          required
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Slug</label>
        <input
          className="border rounded px-3 py-2"
          value={slug}
          onChange={(e) => setSlug(e.target.value)}
          placeholder="pl. marketing"
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Normalizált név</label>
        <input
          className="border rounded px-3 py-2"
          value={normalizedName}
          onChange={(e) => setNormalizedName(e.target.value)}
          placeholder="pl. marketing"
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Kapcsolt cég</label>
        <AsyncSelect
          cacheOptions
          defaultOptions
          loadOptions={loadCompanyOptions}
          value={
            companyId
              ? { value: companyId, label: companyLabel || "—" }
              : companyLabel
              ? { value: "", label: companyLabel }
              : null
          }
          onChange={(opt: any) => {
            setCompanyId(opt?.value ?? null);
            setCompanyLabel(opt?.label ?? "");
          }}
          isClearable
          placeholder="Válassz céget"
        />
      </div>

      <div className="md:col-span-2 flex justify-end gap-2 mt-2">
        <button
          type="submit"
          disabled={!canSubmit}
          className="px-4 py-2 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white disabled:opacity-50">
          Mentés
        </button>
      </div>
    </form>
  );
}
