"use client";

import { useEffect, useState } from "react";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";
import CompanySegmentForm, {
  CompanySegmentUpsertPayload,
} from "./CompanySegmentForm";

export default function ExpandedCompanySegment({
  id,
  onSaved,
  onDeleted,
}: {
  id: string;
  onSaved?: () => void;
  onDeleted?: () => void;
}) {
  const [loading, setLoading] = useState(true);
  const [item, setItem] = useState<any>(null);

  const load = async () => {
    setLoading(true);
    try {
      // expects GET /company-segment/edit/:id or similar
      const r = await ApiCall.companySegments.getOne(id);
      // shape expected: { segment: { ... } } or direct item
      const data = r.data?.segment ?? r.data;
      setItem(data ?? null);
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Betöltés sikertelen");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (id) load();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [id]);

  const handleSave = async (payload: CompanySegmentUpsertPayload) => {
    try {
      // expects POST /company-segment/edit/:id
      await ApiCall.companySegments.update(id, payload);
      toast.success("Szegmens frissítve");
      onSaved?.();
      await load();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Mentés sikertelen");
    }
  };

  const handleDelete = async () => {
    if (!confirm("Biztosan törli ezt a szegmenst?")) return;
    try {
      // reuse bulk delete endpoint with single id
      await ApiCall.companySegments.delete([id]);
      toast.success("Szegmens törölve");
      onDeleted?.();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Törlés sikertelen");
    }
  };

  if (loading)
    return <div className="p-3 text-sm text-gray-500">Betöltés…</div>;
  if (!item)
    return <div className="p-3 text-sm text-red-600">Nem található.</div>;

  return (
    <div className="rounded border p-3 bg-white/50 space-y-3 max-w-[80vw]">
      <CompanySegmentForm
        initial={{
          id: item.id ?? item._id,
          name: item.name,
          slug: item.slug,
          normalized_name: item.normalized_name,
          company_id: item.company_id ?? null,
          company_name: item.company_name ?? null,
        }}
        onSubmit={handleSave}
      />

      <div className="flex items-center justify-between">
        <span className="text-xs text-gray-500">
          Utoljára módosítva: {item.updated_at ?? item.updatedAt ?? "-"}
        </span>
        <button
          onClick={handleDelete}
          className="px-3 py-2 bg-red-600/80 hover:bg-red-600 text-white rounded">
          Törlés
        </button>
      </div>
    </div>
  );
}
