// components/AdminChat.tsx
"use client";

import React, { useContext, useEffect, useRef, useState, useMemo } from "react";
import {
  SocketContext,
  AdminUsersListContext,
  ChatUnreadContext,
} from "@/context/SocketProvider";
import { useAuth } from "@/context/AuthContext";
import ApiCall from "@/api/api-calls";
import { showTypedToast } from "@/lib/typedToast";

const GLOBAL_CONVERSATION_ID = "00000000-0000-0000-0000-000000000001";
const API_BASE_URL = process.env.SERVER_URL || "http://localhost:8110";

const resolveFileUrl = (url: string) => {
  if (!url) return "";
  // if it's already absolute (starts with http), return as is
  if (/^https?:\/\//i.test(url)) return url;
  // otherwise treat it as a backend-relative path like /uploads/...
  return `${API_BASE_URL}${url}`;
};
function getConversationLabel(c: any, myName: string | undefined | null) {
  if (!c) return "";
  if (c.id === GLOBAL_CONVERSATION_ID) return "Mindenki";

  if (c.is_group) {
    const base = c.name || "Csoport";
    return `Csoport: ${base}`;
  }

  const others =
    c.members
      ?.filter((m: any) => m.username && m.username !== myName)
      .map((m: any) => m.username) || [];

  if (others.length > 0) return others.join(", ");

  return "Privát chat";
}

function getConversationSubtitle(c: any, myName: string | undefined | null) {
  if (!c) return "Nincs kiválasztott beszélgetés";

  if (c.id === GLOBAL_CONVERSATION_ID) {
    return "Mindenki (csoportos beszélgetés)";
  }

  if (c.is_group) {
    return `Csoport: ${c.name || "Csoportos beszélgetés"}`;
  }

  const others =
    c.members
      ?.filter((m: any) => m.username && m.username !== myName)
      .map((m: any) => m.username) || [];

  return `Privát: ${others.length ? others.join(", ") : "Ismeretlen"}`;
}

function getConversationDisplayWithUnread(
  c: any,
  myName: string | undefined | null,
  unreadByConversation: Record<string, number>
) {
  if (!c) return "";
  const base = getConversationLabel(c, myName);
  const count = unreadByConversation[c.id] || 0;
  return count > 0 ? `${base} (${count})` : base;
}

export default function AdminChat({ isOpen = true }) {
  const socket = useContext(SocketContext);
  const onlineUsers = useContext(AdminUsersListContext);
  const { setUnread } = useContext(ChatUnreadContext);
  const { user } = useAuth();

  const [conversations, setConversations] = useState<any[]>([]);
  const [messages, setMessages] = useState<any[]>([]);
  const [text, setText] = useState("");
  const [activeConversationId, setActiveConversationId] = useState<
    string | null
  >(GLOBAL_CONVERSATION_ID);
  const [typingUsers, setTypingUsers] = useState<Record<string, boolean>>({});
  const [creatingGroup, setCreatingGroup] = useState(false);
  const [groupMembers, setGroupMembers] = useState<string[]>([]);
  const [groupName, setGroupName] = useState("");
  const [pendingAttachments, setPendingAttachments] = useState<any[]>([]);

  const [conversationSearch, setConversationSearch] = useState("");

  // 🔔 per-conversation unread counts
  const [unreadByConversation, setUnreadByConversation] = useState<
    Record<string, number>
  >({});

  const listRef = useRef<HTMLDivElement | null>(null);
  const typingTimeoutRef = useRef<NodeJS.Timeout | null>(null);
  const fileInputRef = useRef<HTMLInputElement | null>(null);
  const activeConversationRef = useRef<string | null>(activeConversationId);

  const myName = user?.username || user?.email;

  useEffect(() => {
    activeConversationRef.current = activeConversationId;
  }, [activeConversationId]);

  useEffect(() => {
    if (isOpen) {
      setUnreadByConversation({});
    }
  }, [isOpen]);

  // ---- Initial load: conversations + messages ----
  useEffect(() => {
    const load = async () => {
      try {
        const convRes = await ApiCall.chat.getConversations();
        const convs = convRes.data?.body || convRes.data || [];

        setConversations(convs);

        let initialId = GLOBAL_CONVERSATION_ID;
        const foundGlobal = convs.find(
          (c: any) => c.id === GLOBAL_CONVERSATION_ID
        );
        if (!foundGlobal && convs.length > 0) {
          initialId = convs[0].id;
        }
        setActiveConversationId(initialId);
        activeConversationRef.current = initialId;

        const msgRes = await ApiCall.chat.getMessages(initialId);
        const msgs = msgRes.data?.body || msgRes.data || [];
        setMessages(msgs);

        // starting state: everything read
        setUnreadByConversation({});
        setUnread(0);

        socket?.emit("chat:joinConversation", { conversationId: initialId });
      } catch (err) {
        console.error("[AdminChat] Failed to load:", err);
        showTypedToast({
          type: "error",
          message: "Nem sikerült betölteni a chatet",
        });
      }
    };

    if (socket) {
      load();
    }
  }, [socket, setUnread]);

  // ---- File upload ----
  const handleFileChange = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    try {
      const formData = new FormData();
      formData.append("file", file);

      const res = await ApiCall.chat.uploadFile(formData);
      const body = res.data?.body || res.data;

      setPendingAttachments((prev) => [...prev, body]);
    } catch (err) {
      console.error("[AdminChat] file upload error:", err);
      showTypedToast({
        type: "error",
        message: "Nem sikerült a fájl feltöltése",
      });
    } finally {
      e.target.value = "";
    }
  };

  // ---- Load messages when conversation changes & mark as read ----
  useEffect(() => {
    if (!activeConversationId) return;

    const loadMessages = async () => {
      try {
        const msgRes = await ApiCall.chat.getMessages(activeConversationId);
        const msgs = msgRes.data?.body || msgRes.data || [];
        setMessages(msgs);

        // join room
        socket?.emit("chat:joinConversation", {
          conversationId: activeConversationId,
        });

        // mark this conversation as read in the local map
        setUnreadByConversation((prev) => {
          if (!prev[activeConversationId]) return prev;
          const { [activeConversationId]: _, ...rest } = prev;
          return rest;
        });
      } catch (err) {
        console.error("[AdminChat] Failed to load messages:", err);
      }
    };

    loadMessages();
  }, [activeConversationId, socket]);

  // ---- Auto-scroll ----
  useEffect(() => {
    if (!isOpen) return;
    if (!listRef.current) return;
    listRef.current.scrollTop = listRef.current.scrollHeight;
  }, [messages, activeConversationId, isOpen]);

  // ---- Socket listeners ----
  useEffect(() => {
    if (!socket) return;

    const handleChatMessage = (message: any) => {
      // dedupe by id
      setMessages((prev) => {
        if (prev.some((m) => m.id === message.id)) return prev;
        return [...prev, message];
      });

      const isMe = message.from === myName;
      if (isMe) return;

      const currentConvId = activeConversationRef.current;
      const sameConversation = message.conversationId === currentConvId;

      const shouldIncrementUnread =
        !sameConversation || document.hidden || !isOpen;

      if (shouldIncrementUnread) {
        const convId = message.conversationId;
        if (!convId) return;

        setUnreadByConversation((prev) => {
          const prevCount = prev[convId] ?? 0;
          const nextMap: Record<string, number> = {
            ...prev,
            [convId]: prevCount + 1,
          };
          return nextMap;
        });
      }
    };

    const handleSystemMessage = (msg: any) => {
      setMessages((prev) => {
        const id = msg.id || `system-${msg.ts || Date.now()}-${Math.random()}`;
        if (prev.some((m) => m.id === id)) return prev;
        return [...prev, { ...msg, id, from: "SYSTEM", isSystem: true }];
      });
    };

    const handleTyping = (payload: any) => {
      const { from, isTyping, conversationId } = payload;
      if (!from || from === myName) return;

      const currentConvId = activeConversationRef.current;
      if (conversationId !== currentConvId) return;

      setTypingUsers((prev) => {
        const next = { ...prev };
        if (isTyping) next[from] = true;
        else delete next[from];
        return next;
      });
    };

    const handleConversationCreated = (conv: any) => {
      if (!conv || !conv.id) return;

      setConversations((prev) => {
        if (prev.some((c) => c.id === conv.id)) return prev;
        return [...prev, conv];
      });

      socket.emit("chat:joinConversation", { conversationId: conv.id });

      if (conv.created_by === myName) {
        setActiveConversationId(conv.id);
        activeConversationRef.current = conv.id;
      }
    };

    socket.on("chat:message", handleChatMessage);
    socket.on("chat:system", handleSystemMessage);
    socket.on("chat:typing", handleTyping);
    socket.on("chat:conversation:created", handleConversationCreated);

    return () => {
      socket.off("chat:message", handleChatMessage);
      socket.off("chat:system", handleSystemMessage);
      socket.off("chat:typing", handleTyping);
      socket.off("chat:conversation:created", handleConversationCreated);
    };
  }, [socket, myName, isOpen, setUnread]);

  // ---- Typing ----
  const emitTyping = (isTyping: boolean) => {
    if (!socket || !activeConversationId) return;

    socket.emit("chat:typing", {
      conversationId: activeConversationId,
      isTyping,
    });
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setText(e.target.value);

    emitTyping(true);
    if (typingTimeoutRef.current) {
      clearTimeout(typingTimeoutRef.current);
    }
    typingTimeoutRef.current = setTimeout(() => {
      emitTyping(false);
    }, 1500);
  };

  // ---- Send message ----
  const handleSend = (e: React.FormEvent) => {
    e.preventDefault();
    if (
      !socket ||
      (!text.trim() && !pendingAttachments.length) ||
      !activeConversationId
    )
      return;

    socket.emit("chat:message", {
      text: text.trim(),
      conversationId: activeConversationId,
      attachments: pendingAttachments,
    });

    setText("");
    setPendingAttachments([]);
    emitTyping(false);
  };

  // ---- Filter messages for active conversation ----
  const filteredMessages = useMemo(() => {
    if (!activeConversationId) return [];
    return messages.filter((m) => m.conversationId === activeConversationId);
  }, [messages, activeConversationId]);

  const typingList = Object.keys(typingUsers);
  const typingText =
    typingList.length === 1
      ? `${typingList[0]} éppen ír...`
      : typingList.length > 1
      ? `${typingList.join(", ")} éppen írnak...`
      : "";

  // ---- Online users list (for starting DMs / groups) ----
  const selectableUsers = onlineUsers
    .map((u: any) => u.name)
    .filter((name: string | undefined) => name && name !== myName);

  const activeConversation = conversations.find(
    (c) => c.id === activeConversationId
  );
  const globalConversation = conversations.find(
    (c) => c.id === GLOBAL_CONVERSATION_ID
  );

  const activeSubtitle = getConversationSubtitle(activeConversation, myName);

  const toggleMember = (username: string) => {
    setGroupMembers((prev) =>
      prev.includes(username)
        ? prev.filter((u) => u !== username)
        : [...prev, username]
    );
  };

  const handleCreateGroup = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!groupMembers.length) {
      showTypedToast({
        type: "warning",
        message: "Válassz legalább egy résztvevőt",
      });
      return;
    }

    const allMembers = Array.from(new Set([...groupMembers, myName]));

    const existingGroup = conversations.find((c) => {
      if (c.id === GLOBAL_CONVERSATION_ID) return false;
      if (!c.is_group) return false;
      const memberUsernames = (c.members || []).map((m: any) => m.username);
      const uniq = Array.from(new Set(memberUsernames));
      if (uniq.length !== allMembers.length) return false;
      return allMembers.every((u) => uniq.includes(u));
    });

    if (existingGroup) {
      setActiveConversationId(existingGroup.id);
      activeConversationRef.current = existingGroup.id;
      setGroupMembers([]);
      setGroupName("");
      setCreatingGroup(false);

      showTypedToast({
        type: "info",
        message: "Már létezik ilyen csoport, azt nyitottuk meg.",
      });
      return;
    }

    try {
      const res = await ApiCall.chat.createConversation({
        name: groupName || "Új csoport",
        members: groupMembers,
      });
      const conv = res.data?.body || res.data;

      if (conv?.id) {
        setActiveConversationId(conv.id);
        activeConversationRef.current = conv.id;
        socket?.emit("chat:joinConversation", {
          conversationId: conv.id,
        });
      }

      setGroupMembers([]);
      setGroupName("");
      setCreatingGroup(false);

      showTypedToast({
        type: "success",
        message: "Csoportos beszélgetés létrehozva",
      });
    } catch (err) {
      console.error("[AdminChat] create group error:", err);
      showTypedToast({
        type: "error",
        message: "Nem sikerült létrehozni a csoportot",
      });
    }
  };

  const ensurePrivateConversation = async (targetUsername: string) => {
    if (!targetUsername || targetUsername === myName) return;

    const existing = conversations.find((c) => {
      if (c.is_group) return false;
      const members = (c.members || []).map((m: any) => m.username);
      const uniq = Array.from(new Set(members));
      return (
        uniq.length === 2 &&
        uniq.includes(myName) &&
        uniq.includes(targetUsername)
      );
    });

    if (existing) {
      setActiveConversationId(existing.id);
      activeConversationRef.current = existing.id;
      return;
    }

    try {
      const res = await ApiCall.chat.createConversation({
        members: [targetUsername],
      });
      const conv = res.data?.body || res.data;

      if (conv?.id) {
        setActiveConversationId(conv.id);
        activeConversationRef.current = conv.id;
        socket?.emit("chat:joinConversation", {
          conversationId: conv.id,
        });
      }

      showTypedToast({
        type: "success",
        message: `Privát chat indítva: ${targetUsername}`,
      });
    } catch (err) {
      console.error("[AdminChat] ensurePrivateConversation error:", err);
      showTypedToast({
        type: "error",
        message: "Nem sikerült privát beszélgetést indítani",
      });
    }
  };

  // Recompute global unread count whenever per-conversation map changes
  useEffect(() => {
    const values = Object.values(unreadByConversation) as number[];
    const globalCount = values.reduce((sum, n) => sum + n, 0);
    setUnread(globalCount);
  }, [unreadByConversation, setUnread]);

  // Map of last message per conversation (for sidebar preview + sorting)
  const lastMessagesByConversation = useMemo(() => {
    const map: Record<string, any> = {};
    for (const m of messages) {
      map[m.conversationId] = m; // assumes messages array is roughly chronological
    }
    return map;
  }, [messages]);

  const normalizedSearch = conversationSearch.toLowerCase();

  if (!socket) return null;

  return (
    <div
      className={`flex h-96 rounded-2xl border border-gray-200 bg-white/90 shadow-xl backdrop-blur-sm transition-transform transition-opacity duration-200 ${
        isOpen
          ? "translate-y-0 opacity-100"
          : "pointer-events-none translate-y-4 opacity-0"
      }`}>
      {/* LEFT SIDEBAR: conversations + online users */}
      <aside className="w-64 border-r border-gray-100 flex flex-col">
        {/* Sidebar header */}
        <div className="px-3 py-3 border-b border-gray-100 flex items-center justify-between gap-2">
          <div className="flex items-center gap-2">
            <div className="relative">
              <div className="w-2.5 h-2.5 rounded-full bg-emerald-500" />
              <div className="absolute inset-0 w-2.5 h-2.5 rounded-full bg-emerald-500/40 animate-ping" />
            </div>
            <div className="flex flex-col">
              <span className="text-xs font-semibold text-gray-800">
                Admin chat
              </span>
              <span className="text-[11px] text-gray-400">
                {myName || "Ismeretlen felhasználó"}
              </span>
            </div>
          </div>

          <button
            type="button"
            onClick={() => setCreatingGroup((p) => !p)}
            className="px-2 py-1 rounded-full border border-indigo-100 text-[11px] text-indigo-600 hover:bg-indigo-50">
            + Új csoport
          </button>
        </div>

        {/* Search */}
        <div className="px-3 py-2 border-b border-gray-100">
          <input
            className="w-full rounded-full border border-gray-200 px-3 py-1.5 text-xs bg-white focus:outline-none focus:ring-1 focus:ring-indigo-500"
            placeholder="Beszélgetések keresése…"
            value={conversationSearch}
            onChange={(e) => setConversationSearch(e.target.value)}
          />
        </div>

        {/* Conversations list */}
        <div className="flex-1 overflow-auto">
          {/* Global conversation pinned (if exists) */}
          {globalConversation && (
            <>
              <div className="px-3 pt-2 pb-1 text-[10px] font-semibold text-gray-400 uppercase">
                Általános
              </div>
              <button
                type="button"
                onClick={() => {
                  setActiveConversationId(globalConversation.id);
                  activeConversationRef.current = globalConversation.id;
                }}
                className={`w-full flex items-center gap-2 px-3 py-2 text-left text-xs hover:bg-gray-50 ${
                  activeConversationId === globalConversation.id
                    ? "bg-indigo-50/70"
                    : ""
                }`}>
                <div className="w-7 h-7 rounded-full bg-indigo-100 text-[11px] flex items-center justify-center text-indigo-700 font-medium">
                  🌍
                </div>
                <div className="flex-1 min-w-0">
                  <div className="flex items-center justify-between gap-1">
                    <span className="truncate text-gray-800">
                      {getConversationDisplayWithUnread(
                        globalConversation,
                        myName,
                        unreadByConversation
                      )}
                    </span>
                    {lastMessagesByConversation[globalConversation.id]?.ts && (
                      <span className="text-[10px] text-gray-400 shrink-0">
                        {new Date(
                          lastMessagesByConversation[globalConversation.id].ts
                        ).toLocaleTimeString([], {
                          hour: "2-digit",
                          minute: "2-digit",
                        })}
                      </span>
                    )}
                  </div>
                  <span className="text-[11px] text-gray-400 truncate">
                    {lastMessagesByConversation[globalConversation.id]?.text ||
                      "Nincs üzenet"}
                  </span>
                </div>
              </button>
            </>
          )}

          {/* Other conversations */}
          <div className="px-3 pt-2 pb-1 text-[10px] font-semibold text-gray-400 uppercase">
            Beszélgetések
          </div>
          {conversations
            .filter((c) => c.id !== GLOBAL_CONVERSATION_ID)
            .filter((c) => {
              if (!normalizedSearch) return true;
              const label = getConversationLabel(c, myName)
                .toLowerCase()
                .trim();
              const last =
                lastMessagesByConversation[c.id]?.text?.toLowerCase() || "";
              return (
                label.includes(normalizedSearch) ||
                last.includes(normalizedSearch)
              );
            })
            .sort((a, b) => {
              const ua = unreadByConversation[a.id] ?? 0;
              const ub = unreadByConversation[b.id] ?? 0;
              if (ua !== ub) return ub - ua; // unread first
              const la = lastMessagesByConversation[a.id]?.ts || 0;
              const lb = lastMessagesByConversation[b.id]?.ts || 0;
              return lb - la; // then by last activity
            })
            .map((c) => {
              const unread = unreadByConversation[c.id] ?? 0;
              const last = lastMessagesByConversation[c.id];
              const isActive = c.id === activeConversationId;
              const label = getConversationLabel(c, myName);
              const icon = c.is_group ? "👥" : "👤";

              return (
                <button
                  key={c.id}
                  type="button"
                  onClick={() => {
                    setActiveConversationId(c.id);
                    activeConversationRef.current = c.id;
                  }}
                  className={`w-full flex items-center gap-2 px-3 py-2 text-left text-xs hover:bg-gray-50 ${
                    isActive ? "bg-indigo-50/70" : ""
                  }`}>
                  <div className="w-7 h-7 rounded-full bg-indigo-100 text-[11px] flex items-center justify-center text-indigo-700 font-medium">
                    {icon}
                  </div>

                  <div className="flex-1 min-w-0">
                    <div className="flex items-center justify-between gap-1">
                      <span
                        className={`truncate ${
                          unread
                            ? "font-semibold text-gray-900"
                            : "text-gray-800"
                        }`}>
                        {label}
                      </span>
                      {last?.ts && (
                        <span className="text-[10px] text-gray-400 shrink-0">
                          {new Date(last.ts).toLocaleTimeString([], {
                            hour: "2-digit",
                            minute: "2-digit",
                          })}
                        </span>
                      )}
                    </div>
                    <div className="flex items-center justify-between gap-1">
                      <span className="text-[11px] text-gray-400 truncate">
                        {last?.text || "Nincs üzenet"}
                      </span>
                      {unread > 0 && (
                        <span className="ml-1 inline-flex items-center justify-center min-w-[16px] h-[16px] px-1 rounded-full bg-red-500 text-[10px] text-white font-semibold shrink-0">
                          {unread > 9 ? "9+" : unread}
                        </span>
                      )}
                    </div>
                  </div>
                </button>
              );
            })}

          {conversations.filter((c) => c.id !== GLOBAL_CONVERSATION_ID)
            .length === 0 && (
            <div className="px-3 py-4 text-[11px] text-gray-400">
              Még nincsenek beszélgetések.
            </div>
          )}
        </div>

        {/* Online users at bottom */}
        <div className="border-t border-gray-100 p-2">
          <div className="flex items-center justify-between text-[11px] text-gray-500 mb-1">
            <span>Online</span>
          </div>
          <div className="flex flex-wrap gap-1">
            {selectableUsers.map((name: string) => (
              <button
                key={name}
                type="button"
                onClick={() => ensurePrivateConversation(name)}
                className="inline-flex items-center gap-1 px-2 py-0.5 rounded-full bg-gray-50 border border-gray-200 text-[11px] text-gray-700 hover:bg-gray-100 max-w-full">
                <span className="w-1.5 h-1.5 rounded-full bg-emerald-500" />
                <span className="truncate max-w-[120px]">{name}</span>
              </button>
            ))}
            {selectableUsers.length === 0 && (
              <span className="text-[11px] text-gray-400">
                Nincs más online felhasználó.
              </span>
            )}
          </div>
        </div>
      </aside>

      {/* RIGHT: active conversation */}
      <div className="flex flex-col flex-1">
        {/* Header for active conversation */}
        <div className="px-4 py-3 border-b border-gray-100 flex items-center justify-between gap-3">
          <div className="flex items-center gap-2">
            <div className="w-8 h-8 rounded-full bg-gray-800 text-[12px] flex items-center justify-center text-white font-medium">
              {activeConversation
                ? getConversationLabel(activeConversation, myName)
                    .charAt(0)
                    .toUpperCase()
                : "?"}
            </div>
            <div className="flex flex-col">
              <span className="text-xs font-semibold text-gray-800">
                {activeConversation
                  ? getConversationLabel(activeConversation, myName)
                  : "Nincs kiválasztott beszélgetés"}
              </span>
              <span className="text-[11px] text-gray-400">
                {activeSubtitle}
              </span>
            </div>
          </div>

          {/* (Optional) show unread count for active conversation */}
          {activeConversationId &&
            unreadByConversation[activeConversationId] && (
              <div className="text-[11px] text-gray-500">
                Új üzenetek:{" "}
                <span className="font-semibold">
                  {unreadByConversation[activeConversationId]}
                </span>
              </div>
            )}
        </div>

        {/* Group create panel (on right side) */}
        {creatingGroup && (
          <form
            onSubmit={handleCreateGroup}
            className="border-b border-gray-100 px-4 py-2 text-[11px] space-y-2 bg-gray-50/60">
            <div>
              <label className="block mb-1 text-gray-600">Csoport neve</label>
              <input
                className="w-full border border-gray-200 rounded px-2 py-1 focus:outline-none focus:ring-1 focus:ring-indigo-500"
                value={groupName}
                onChange={(e) => setGroupName(e.target.value)}
                placeholder="Pl. Support csapat"
              />
            </div>

            <div>
              <label className="block mb-1 text-gray-600">
                Résztvevők (onlineból)
              </label>
              <div className="max-h-24 overflow-auto border border-gray-200 rounded p-2 space-y-1 bg-white">
                {selectableUsers.length === 0 && (
                  <div className="text-gray-400 text-[11px]">
                    Nincs más online felhasználó.
                  </div>
                )}
                {selectableUsers.map((name) => (
                  <label
                    key={name}
                    className="flex items-center gap-2 text-gray-700">
                    <input
                      type="checkbox"
                      checked={groupMembers.includes(name)}
                      onChange={() => toggleMember(name)}
                    />
                    <span>{name}</span>
                  </label>
                ))}
              </div>
            </div>
            <div className="flex justify-end gap-2">
              <button
                type="button"
                onClick={() => setCreatingGroup(false)}
                className="px-2 py-1 rounded-full border border-gray-200 text-gray-500">
                Mégse
              </button>
              <button
                type="submit"
                className="px-3 py-1 rounded-full bg-indigo-600 text-white">
                Létrehozás
              </button>
            </div>
          </form>
        )}

        {/* Messages */}
        <div
          ref={listRef}
          className="flex-1 overflow-auto px-3 py-2 space-y-1.5 bg-gradient-to-b from-gray-50/60 to-white">
          {filteredMessages.length === 0 && (
            <div className="flex h-full items-center justify-center">
              <div className="text-xs text-gray-400 text-center px-4">
                Még nincsenek üzenetek ebben a beszélgetésben.
                <br />
                Írj egy üzenetet a kezdéshez. 🙂
              </div>
            </div>
          )}

          {/* simple date separators */}
          {filteredMessages.map((m, idx) => {
            const isMe = m.from === myName;
            const isSystem = m.isSystem || m.from === "SYSTEM";
            const timeLabel = m.ts
              ? new Date(m.ts).toLocaleTimeString([], {
                  hour: "2-digit",
                  minute: "2-digit",
                })
              : "";
            const currentDate = m.ts
              ? new Date(m.ts).toLocaleDateString("hu-HU")
              : "";
            const prevMsg = filteredMessages[idx - 1];
            const prevDate = prevMsg?.ts
              ? new Date(prevMsg.ts).toLocaleDateString("hu-HU")
              : "";
            const showDateSeparator = currentDate && currentDate !== prevDate;

            if (isSystem) {
              return (
                <React.Fragment key={m.id}>
                  {showDateSeparator && (
                    <div className="my-2 flex items-center justify-center">
                      <span className="px-3 py-0.5 rounded-full bg-gray-100 text-[10px] text-gray-500">
                        {currentDate}
                      </span>
                    </div>
                  )}
                  <div className="my-1 text-center text-[11px] text-gray-400">
                    {m.text}
                  </div>
                </React.Fragment>
              );
            }

            return (
              <React.Fragment key={m.id}>
                {showDateSeparator && (
                  <div className="my-2 flex items-center justify-center">
                    <span className="px-3 py-0.5 rounded-full bg-gray-100 text-[10px] text-gray-500">
                      {currentDate}
                    </span>
                  </div>
                )}
                <div
                  className={`flex mb-1 ${
                    isMe ? "justify-end" : "justify-start"
                  }`}>
                  {!isMe && (
                    <div className="mr-2 mt-1 w-6 h-6 rounded-full bg-indigo-100 text-[10px] flex items-center justify-center text-indigo-700 font-medium">
                      {m.from?.charAt(0)?.toUpperCase() || "?"}
                    </div>
                  )}

                  <div
                    className={`max-w-[80%] rounded-2xl px-3 py-2 shadow-sm border text-xs ${
                      isMe
                        ? "bg-indigo-50 border-indigo-100 text-gray-800"
                        : "bg-white border-gray-100 text-gray-800"
                    }`}>
                    <div className="flex items-baseline justify-between gap-2 mb-0.5">
                      <span className="font-medium text-[11px] text-gray-700">
                        {isMe ? "Te" : m.from}
                      </span>
                      {timeLabel && (
                        <span className="text-[10px] text-gray-400">
                          {timeLabel}
                        </span>
                      )}
                    </div>
                    <div className="text-[13px] leading-snug whitespace-pre-wrap break-words">
                      {m.text}
                    </div>

                    {m.attachments?.map((a: any) => {
                      const isImage = a.mimeType?.startsWith("image/");
                      const fileUrl = resolveFileUrl(a.url);
                      if (isImage) {
                        return (
                          <a
                            key={a.id}
                            href={fileUrl}
                            target="_blank"
                            rel="noopener noreferrer"
                            className="block mt-1">
                            <img
                              src={fileUrl}
                              alt={a.fileName}
                              className="max-h-40 rounded-md border border-gray-200"
                            />
                          </a>
                        );
                      }

                      return (
                        <a
                          key={a.id}
                          href={fileUrl}
                          target="_blank"
                          rel="noopener noreferrer"
                          className="block text-[11px] text-indigo-600 underline mt-1 break-all">
                          📎 {a.fileName}
                        </a>
                      );
                    })}
                  </div>

                  {isMe && (
                    <div className="ml-2 mt-1 w-6 h-6 rounded-full bg-gray-800 text-[10px] flex items-center justify-center text-white font-medium">
                      {myName?.charAt(0)?.toUpperCase() || "M"}
                    </div>
                  )}
                </div>
              </React.Fragment>
            );
          })}
        </div>

        {/* Typing indicator */}
        {typingText && (
          <div className="px-4 pb-1 text-[11px] text-gray-500 flex items-center gap-2">
            <div className="flex gap-0.5 items-center">
              <span
                className="w-1.5 h-1.5 rounded-full bg-gray-400/80 animate-pulse"
                style={{ animationDelay: "0ms" }}
              />
              <span
                className="w-1.5 h-1.5 rounded-full bg-gray-400/60 animate-pulse"
                style={{ animationDelay: "120ms" }}
              />
              <span
                className="w-1.5 h-1.5 rounded-full bg-gray-400/40 animate-pulse"
                style={{ animationDelay: "240ms" }}
              />
            </div>
            <span>{typingText}</span>
          </div>
        )}

        {/* Pending attachments chips */}
        {pendingAttachments.length > 0 && (
          <div className="px-4 pt-1 pb-2 text-[11px] text-gray-600 flex flex-wrap gap-1 border-t border-gray-100 bg-gray-50/60">
            {pendingAttachments.map((a, idx) => (
              <span
                key={idx}
                className="inline-flex items-center gap-1 px-2 py-0.5 rounded-full bg-indigo-50 border border-indigo-100">
                📎 {a.fileName}
              </span>
            ))}
          </div>
        )}

        {/* Input */}
        <form
          onSubmit={handleSend}
          className="px-3 pb-3 pt-2 border-t border-gray-100 flex items-center gap-2">
          <input
            type="file"
            ref={fileInputRef}
            className="hidden"
            onChange={handleFileChange}
          />

          <button
            type="button"
            onClick={() => fileInputRef.current?.click()}
            className="inline-flex items-center justify-center px-2 py-1.5 rounded-full border border-gray-200 text-xs text-gray-600 hover:bg-gray-50">
            <span className="material-icons text-[16px]">attach_file</span>
          </button>

          <input
            className="flex-1 border border-gray-200 rounded-full px-3 py-1.5 text-xs bg-white focus:outline-none focus:ring-2 focus:ring-indigo-500/70 focus:border-indigo-500"
            value={text}
            onChange={handleInputChange}
            placeholder={
              activeConversation?.id === GLOBAL_CONVERSATION_ID
                ? "Üzenet mindenkinek…"
                : activeConversation?.is_group
                ? "Üzenet a csoportnak…"
                : "Üzenet…"
            }
          />

          <button
            type="submit"
            className="inline-flex items-center justify-center px-3 py-1.5 rounded-full text-xs bg-indigo-600 text-white hover:bg-indigo-700 transition-colors disabled:bg-gray-300 disabled:cursor-not-allowed"
            disabled={
              (!text.trim() && !pendingAttachments.length) ||
              !activeConversationId
            }>
            <span className="material-icons text-[16px] mr-1">send</span>
            Küldés
          </button>
        </form>
      </div>
    </div>
  );
}
