// src/components/registrars/RegistrarForm.tsx
"use client";

import { useEffect, useMemo, useState } from "react";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";

export type RegistrarPayload = {
  name?: string;
  email?: string | null;
  url?: string | null;
  password?: string | null; // send only when setting/changing; null to clear
};

export default function RegistrarForm({
  initial,
  onSubmit,
}: {
  initial?: {
    id?: string;
    name?: string;
    email?: string | null;
    url?: string | null;
    has_password?: boolean; // ← NEW
  };
  onSubmit: (payload: RegistrarPayload) => void | Promise<void>;
}) {
  const [name, setName] = useState(initial?.name ?? "");
  const [email, setEmail] = useState<string>(initial?.email ?? "");
  const [url, setUrl] = useState<string>(initial?.url ?? "");

  // password editing state
  const [password, setPassword] = useState<string>(""); // empty => no change
  const [showPw, setShowPw] = useState(false); // ← toggle visibility
  const [editPw, setEditPw] = useState(false); // ← show input section
  const [hasPw, setHasPw] = useState<boolean>(!!initial?.has_password);

  useEffect(() => {
    if (!initial) return;
    setName(initial.name ?? "");
    setEmail(initial.email ?? "");
    setUrl(initial.url ?? "");
    setPassword("");
    setShowPw(false);
    setEditPw(false);
    setHasPw(!!initial.has_password);
  }, [initial]);

  const canSubmit = useMemo(() => name.trim().length > 1, [name]);

  const handleGenerate = () => {
    // simple strong-ish generator; tweak as you like
    const bytes = crypto.getRandomValues(new Uint8Array(16));
    const base = btoa(String.fromCharCode(...Array.from(bytes)))
      .replace(/[^a-z0-9]/gi, "")
      .slice(0, 12);
    const candidate = `${base}#${Math.floor(100 + Math.random() * 900)}`;
    setPassword(candidate);
    setShowPw(true);
    setEditPw(true);
  };

  const handleClearPassword = () => {
    // send null to clear on backend
    setPassword("-");
    setEditPw(true); // reveal the section so user sees what's happening
    setShowPw(true);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    const payload: RegistrarPayload = {
      name: name.trim(),
      email: email.trim() ? email.trim().toLowerCase() : null,
      url: url.trim() ? url.trim() : null,
    };

    // Only include password if user typed something OR explicitly clear
    if (password === "") {
      // omit => keep existing
    } else if (password === "-") {
      payload.password = null; // clear
    } else {
      payload.password = password; // set new
    }

    await onSubmit(payload);
    // optimistic local flag updates
    if (payload.password === null) {
      setHasPw(false);
      setPassword("");
      setEditPw(false);
    } else if (payload.password) {
      setHasPw(true);
      setPassword("");
      setEditPw(false);
    }
  };

  return (
    <form
      onSubmit={handleSubmit}
      className="grid grid-cols-1 md:grid-cols-2 gap-3">
      <div className="flex flex-col md:col-span-2">
        <label className="text-xs text-gray-500">Név</label>
        <input
          className="border rounded px-3 py-2"
          value={name}
          onChange={(e) => setName(e.target.value)}
          placeholder="pl. Rackhost"
          required
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Email</label>
        <input
          className="border rounded px-3 py-2"
          value={email}
          onChange={(e) => setEmail(e.target.value)}
          type="email"
          placeholder="pl. support@example.com"
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">URL</label>
        <input
          className="border rounded px-3 py-2"
          value={url}
          onChange={(e) => setUrl(e.target.value)}
          type="url"
          placeholder="https://"
        />
      </div>

      {/* Password section */}

      <div className="md:col-span-2 border rounded p-3 bg-white/60 space-y-2">
        <div className="flex items-center justify-between">
          <div className="text-sm">
            <span className="mr-2 text-gray-700">Jelszó állapot:</span>
            {hasPw ? (
              <span className="px-2 py-0.5 rounded bg-emerald-100 text-emerald-800 text-xs">
                Be van állítva
              </span>
            ) : (
              <span className="px-2 py-0.5 rounded bg-gray-100 text-gray-800 text-xs">
                Nincs beállítva
              </span>
            )}
          </div>
          <div className="flex gap-2">
            <button
              type="button"
              onClick={() => {
                setEditPw((v) => !v);
                setShowPw(false);
                setPassword("");
              }}
              className="px-2 py-1 rounded border text-sm">
              {editPw
                ? "Mégse"
                : hasPw
                ? "Jelszó módosítása"
                : "Jelszó beállítása"}
            </button>

            {hasPw && (
              <button
                type="button"
                onClick={async () => {
                  try {
                    const r = await ApiCall.domainRegistrar.getPassword(
                      initial?.id!
                    );
                    const pw = r.data?.password ?? "";
                    if (!pw) {
                      // could be cleared
                      setHasPw(false);
                      return;
                    }
                    setPassword(pw);
                    setEditPw(true);
                    setShowPw(false); // default hidden
                  } catch (e: any) {
                    // 403/401/429/etc.
                    toast.error(
                      e?.response?.data?.error ?? "Jelszó nem kérhető le"
                    );
                  }
                }}
                className="px-2 py-1 rounded border text-sm"
                title="Meglévő jelszó megjelenítése (csak SUPERADMIN)">
                Jelszó megjelenítése
              </button>
            )}

            {hasPw && (
              <button
                type="button"
                onClick={handleClearPassword}
                className="px-2 py-1 rounded border border-red-300 text-red-700 text-sm hover:bg-red-50"
                title="Jelszó törlése (mentés után lép érvénybe)">
                Jelszó törlése
              </button>
            )}
          </div>
        </div>

        {editPw && (
          <div className="flex flex-col gap-2 md:flex-row md:items-end">
            <div className="flex-1">
              <label className="text-xs text-gray-500">
                {hasPw ? "Új jelszó (vagy meglévő szerkesztése)" : "Jelszó"}
              </label>
              <div className="flex">
                <input
                  className="border rounded-l px-3 py-2 grow"
                  value={password === "-" ? "" : password}
                  onChange={(e) => setPassword(e.target.value)}
                  type={showPw ? "text" : "password"}
                  placeholder={
                    hasPw ? "Hagyja üresen a változatlanhoz" : "Új jelszó"
                  }
                  autoComplete="new-password"
                />
                <button
                  type="button"
                  onClick={() => setShowPw((v) => !v)}
                  className="px-3 py-2 border-y border-r rounded-r text-sm"
                  title={showPw ? "Elrejtés" : "Megjelenítés"}>
                  {showPw ? "👁️‍🗨️" : "👁️"}
                </button>
              </div>
              <div className="mt-1 flex gap-2">
                <button
                  type="button"
                  onClick={async () => {
                    try {
                      await navigator.clipboard.writeText(password);
                      toast.success("Jelszó másolva");
                    } catch {
                      toast.error("Másolás nem sikerült");
                    }
                  }}
                  disabled={!password || password === "-"}
                  className="px-2 py-1 rounded border text-xs">
                  Másolás
                </button>
                <button
                  type="button"
                  onClick={handleGenerate}
                  className="px-2 py-1 rounded border text-xs">
                  Generálás
                </button>
              </div>
              <p className="text-[11px] text-gray-500 mt-1">
                A beküldött jelszó a szerveren hash-elésre és titkosításra
                kerül. A jelenlegi jelszó csak jogosult felhasználóknak kérhető
                le.
              </p>
            </div>
          </div>
        )}
      </div>

      <div className="md:col-span-2 flex justify-end gap-2">
        <button
          type="submit"
          disabled={!canSubmit}
          className="px-4 py-2 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white disabled:opacity-50">
          Mentés
        </button>
      </div>
    </form>
  );
}
