// components/domain-registry/ExpandedDomainRegistry.tsx
"use client";

import { useEffect, useState } from "react";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";
import DomainRegistryForm, { DomainUpsertPayload } from "./DomainRegistryForm";

export default function ExpandedDomainRegistry({
  id,
  onSaved,
  onDeleted,
}: {
  id: string;
  onSaved?: () => void;
  onDeleted?: () => void;
}) {
  const [loading, setLoading] = useState(true);
  const [item, setItem] = useState<any>(null);

  const load = async () => {
    setLoading(true);
    try {
      const r = await ApiCall.domainRegistry.getOne(id);
      setItem(r.data ?? null);
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Betöltés sikertelen");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (id) load();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [id]);

  const handleSave = async (payload: DomainUpsertPayload) => {
    try {
      await ApiCall.domainRegistry.update(id, payload);
      toast.success("Frissítve");
      onSaved?.();
      await load();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Mentés sikertelen");
    }
  };

  const handleDelete = async () => {
    if (!confirm("Biztosan törli?")) return;
    try {
      await ApiCall.domainRegistry.remove(id);
      toast.success("Törölve");
      onDeleted?.();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Törlés sikertelen");
    }
  };

  if (loading)
    return <div className="p-3 text-sm text-gray-500">Betöltés…</div>;
  if (!item)
    return <div className="p-3 text-sm text-red-600">Nem található.</div>;

  return (
    <div className="rounded border p-3 bg-white/50 space-y-3 max-w-[80vw]">
      <DomainRegistryForm
        initial={{
          id: item.id,
          domain_name: item.domain_name,
          current_company_id: item.current_company_id ?? null,
          current_company_name: item.current_company_name ?? null, // label seed
          new_biller_id: item.new_biller_id ?? null,
          new_biller_name: item.new_biller_name ?? null, // label seed
          registrar_id: item.registrar_id ?? null,
          registrar_name: item.registrar_name ?? null,
          expires_on: item.expires_on ?? null,
          auto_renew: !!item.auto_renew,
          cancel_flag: !!item.cancel_flag,
          notes: item.notes ?? "",
        }}
        onSubmit={handleSave}
      />

      <div className="flex items-center justify-between">
        <span className="text-xs text-gray-500">
          Utoljára módosítva: {item.updated_at ?? "-"}
        </span>
        <button
          onClick={handleDelete}
          className="px-3 py-2 bg-red-600/80 hover:bg-red-600 text-white rounded">
          Törlés
        </button>
      </div>
    </div>
  );
}
