import React, { useEffect, useState, useCallback, useRef } from "react";
import { useSelector, useDispatch } from "react-redux";
import {
  selectedElement,
  updateContainerContent,
} from "@/store/slices/pageEditorSlice";

import Select from "react-select";
import { initialData } from "@/data/data";
import Button from "@/components/shared/button/Button";
import dynamic from "next/dynamic";
import ApiCall from "@/api/api-calls";
const ColorPicker = dynamic(() => import("react-best-gradient-color-picker"), {
  ssr: false,
});

const EditorPeople = ({ toggleModal, modalName }) => {
  const [isValid, setIsValid] = useState(false);

  const [locations, setLocations] = useState(null);

  useEffect(() => {
    const fetchLocations = async () => {
      const response = await ApiCall.peoples.getLocations();
      setLocations(response);
    };
    fetchLocations();
  }, []);

  const editableContent = useSelector(
    (state) => state.pageEditor.current.content
  );
  const inputRefs = useRef([]);

  let index = 0;
  const [color, setColor] = useState(
    editableContent?.[0]?.content?.[0]?.backgroundColor ??
      "rgba(255, 255, 255, 1)"
  );

  const dispatch = useDispatch();

  useEffect(() => {
    // If there is data, the form is valid
    setIsValid(
      editableContent && editableContent?.[0]?.selectedColumn !== ""
        ? true
        : false
    );
  }, [editableContent]);

  const submitPeople = useCallback(() => {
    const content = {
      type: "people",
      title: "Személyek",
      backgroundColor: color,
      displayedProperties: [],
    };
    inputRefs.current.map((c, i) => {
      if (c) {
        if (c.props) {
          if (Array.isArray(c.props.value)) {
            if (c.props.name === "displayedProperties") {
              c.props.value.map((x, i) => {
                content.displayedProperties.push(x);
              });
            } else {
              content[c.props.name] = c.props.value?.[0].value;
            }
          } else {
            content[c.props.name] = c.props.value.value;
          }
        } else {
          content[c.name] = c.value;
        }
      }
    });

    dispatch(
      selectedElement({
        content: content,
      })
    );
    dispatch(updateContainerContent());
    toggleModal(modalName);
    toggleModal("showSelectElementModal");
  }, [color, inputRefs]);

  return (
    <div className="relative">
      <div className="product-wrapper flex flex-col w-auto p-4">
        {/* Layout select */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"peopleSelect"}
            className="w-sm-12 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Személy layout: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Válassza ki, hogy milyen formátumban jelenjen(ek) meg a
              személy(ek) - (alapértelmezett érték: Alap)
            </span>
          </label>

          <Select
            options={initialData.editor.people.layoutTypes}
            className="w-sm-36"
            menuPosition={"fixed"}
            placeholder={"Válasszon személy megjelenítést"}
            backspaceRemovesValue={false}
            name={"peopleSelect"}
            ref={(r) => (inputRefs.current[index++] = r)}
            defaultValue={
              editableContent?.[0]?.content?.[0]?.peopleSelect
                ? initialData.editor.people.layoutTypes.filter(
                    (x) =>
                      x.value === editableContent?.[0].content[0].peopleSelect
                  )
                : initialData.editor.people.layoutTypes[0]
            }
            onChange={(val, e) => {}}
            required
          />
        </div>
        {/* people count per stack */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor="productLength"
            className="w-sm-12 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Személyek oldalanként: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Válassza ki, az egyszerre megjelenítendő személyek számát
              (alapértelmezett érték: Összes)
            </span>
          </label>

          <Select
            options={initialData.editor.people.peopleLength}
            menuPosition={"fixed"}
            className="w-sm-36"
            placeholder={"Személyek száma oldalanként"}
            backspaceRemovesValue={false}
            name={"peopleLength"}
            ref={(r) => (inputRefs.current[index++] = r)}
            defaultValue={
              editableContent?.[0]?.content?.[0]?.peopleLength
                ? initialData.editor.people.peopleLength.filter(
                    (x) =>
                      x.value === editableContent?.[0].content[0].peopleLength
                  )
                : initialData.editor.people.peopleLength[0]
            } // default selected
            onChange={(val, e) => {}}
            required
          />
        </div>
        {/* order people by */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor="orderBy"
            className="w-sm-12 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Rendezés:</p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Válassza ki, hogy milyen érték alapján rendezve szeretné
              megjeleníteni a személyeket (alapértelmezett érték: Személy neve)
            </span>
          </label>

          <Select
            options={initialData.editor.people.orderBy}
            menuPosition={"fixed"}
            className="w-sm-36"
            placeholder={"Személyek rendezése"}
            backspaceRemovesValue={false}
            name={"orderBySelect"}
            ref={(r) => (inputRefs.current[index++] = r)}
            defaultValue={
              editableContent?.[0]?.content?.[0]?.orderBySelect
                ? initialData.editor.people.orderBy.filter(
                    (x) =>
                      x.value === editableContent?.[0].content[0].orderBySelect
                  )
                : initialData.editor.people.orderBy[1]
            } // default selected
            onChange={(val, e) => {
              // second parameter shows action triggered
              // dispatch(selectParentPage({ parent: val }));
            }}
          />
        </div>
        {/* locations select */}
        {locations && (
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor="selectedCategory"
              className="w-sm-12 text-normal text-gray-400 whitespace-pre flex flex-col flex-wrap pr-2">
              <p className="w-full font-bold">
                Megjelenítendő telephely kiválasztása:
              </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy mely telephelyhez tartozó személyeket kívánja
                megjeleníteni
              </span>
            </label>
            <div className="flex flex-col w-sm-36">
              <Select
                options={locations}
                menuPosition={"fixed"}
                className="w-full"
                placeholder={"Megjelenítendő telephely kiválasztása"}
                backspaceRemovesValue={false}
                name={"selectedLocation"}
                ref={(r) => (inputRefs.current[index++] = r)}
                defaultValue={
                  editableContent?.[0]?.content?.[0]?.selectedLocation
                    ? locations.filter(
                        (x) =>
                          x.value ===
                          editableContent?.[0].content[0].selectedLocation
                      )
                    : locations[0]
                }
                onChange={(val, e) => {
                  setIsValid(val ? true : false);
                }}
              />
            </div>
          </div>
        )}
        {/* Content display */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor="displayedProperties"
            className="w-sm-12 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Tartalom megjelenítése:</p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Válassza ki, hogy a személyeknél milyen tulajdonságokat szeretne
              megjeleníteni (alapértelmezett érték: Rövid szöveg)
            </span>
          </label>

          <Select
            options={initialData.editor.people.displayedProperties}
            menuPosition={"fixed"}
            className="w-sm-36"
            placeholder={"Tartalom megjelenítése"}
            backspaceRemovesValue={false}
            name={"displayedProperties"}
            ref={(r) => (inputRefs.current[index++] = r)}
            defaultValue={
              editableContent?.[0]?.content?.[0]?.displayedProperties ??
              initialData.editor.people.displayedProperties
            } // default selected
            isMulti
            onChange={(val, e) => {
              // second parameter shows action triggered
              // dispatch(selectParentPage({ parent: val }));
            }}
            required
          />
        </div>

        {/* people background color */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"backgroundColor"}
            className="w-sm-12 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Háttér szín: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Válassza ki, hogy milyen háttér színűek legyenek az egyes elemek -
              (alapértelmezett érték: rgba(255, 255, 255, 1) - Fehér).
            </span>
          </label>

          <ColorPicker value={color} onChange={setColor} />
        </div>
      </div>
      <div className="flex flex-row w-auto justify-end sticky bottom-0 bg-white p-2 border border-t">
        <Button
          onClick={submitPeople}
          className="px-4 py-2 uppercase border bg-green-500/75 text-white hover:border-green-500 hover:bg-green-500 disabled:bg-gray-500 disabled:pointer-events-none"
          btntext="Mentés"
          // disabled={!isValid}
        />
      </div>
    </div>
  );
};

export default EditorPeople;
