import React, { useState, useCallback, useRef } from "react";
import { initialData } from "@/data/data";
import Button from "@/components/shared/button/Button";
import icons from "@/assets/icons/icons";
import Modal from "@/components/shared/modal/Modal";
import ModalButton from "@/components/shared/modal/ModalButton";
import { useEditorSeparator } from "../../../hooks/useEditorSeparator";
import { handleInputMinMax } from "@/utils/handleInputMinMax";
import LabeledInput from "@/components/shared/labeledinput/LabeledInput";
import SelectInput from "@/components/shared/selectinput/SelectInput";
import ColorPickerComponent from "@/components/shared/colorpicker/ColorPickerComponent";
import TextInput from "@/components/shared/textinput/TextInput";
import useDynamicManager from "@/store/managers/useDynamicManager";
import ApiCall from "@/api/api-calls";
import ImageWithFallback from "@/components/shared/image/ImageWithFallback";
import MediaGalleryModal from "@/components/media-store/GalleryModal/MediaGalleryModal";

const EditorSeparator = ({ toggleModal, modalName, type }) => {
  const { handleImageRemove, handleSubmitEditorSeparator } = useEditorSeparator(
    { toggleModal, modalName, type: type }
  );

  const { state, setSelectedImage } = useDynamicManager(type);
  const [selectImageModal, showSelectImageModal] = useState(false);
  const [color, setColor] = useState(
    state?.current?.content?.[0]?.content?.[0]?.separatorColor ?? "#212121"
  );
  const [backgroundColor, setBackgroundColor] = useState(
    state?.current?.content?.[0]?.content?.[0]?.backgroundColor ?? "transparent"
  );
  const inputRefs = useRef([]);

  const generatSelectedImage = useCallback(() => {
    const i = 0;

    if (state?.selectedImage && state?.selectedImage?._id) {
      return state?.selectedImage?.props?.mimetype === "application/pdf" ? (
        <div
          className={`flex flex-col mr-2 mb-2 items-center border-1 w-full`}
          key={state?.selectedImage?._id}>
          {/* <MediaObject
            data={state?.selectedImage}
            index={i}
            data-index={i}
            className={"w-full"}
            id={state?.selectedImage?._id}
          /> */}
          <p className="break-all p-1">
            {" "}
            {state?.selectedImage.props.basename ??
              state?.selectedImage.originalname}
          </p>
          <input
            type="hidden"
            name="selectedFile"
            defaultValue={state?.selectedImage}
          />

          <button onClick={handleImageRemove} className="text-red-500 my-4">
            Eltávolítás
          </button>
        </div>
      ) : (
        <div
          className={`flex flex-col mr-2 mb-2 items-center border-1 w-full`}
          key={i}>
          <ImageWithFallback
            src={ApiCall.mediaStore.getResized(
              "/4/4/1",
              state?.selectedImage?.originalname
            )}
            width={150}
            height={100}
            alt={
              state?.selectedImage.originalname
                ? state?.selectedImage.originalname
                : ""
            }
            effect="blur"
            loading="lazy"
          />
          <p className="break-all p-1">{state?.selectedImage.props.filename}</p>
          <input
            type="hidden"
            name="selectedFile"
            defaultValue={state?.selectedImage}
          />
          <button onClick={handleImageRemove}>Eltávolítás</button>
        </div>
      );
    }
  }, [state?.selectedImage]);

  const submitEditorSeparator = useCallback(() => {
    handleSubmitEditorSeparator(
      {
        type: "separator",
        title: "Elválasztó",
        separatorColor: color,
        backgroundColor: backgroundColor,
        selectedImage: state?.selectedImage,
      },
      inputRefs
    );
  }, [color, inputRefs, state?.selectedImage, backgroundColor]);

  return (
    <div className="relative">
      <div className="image-wrapper flex flex-col w-full p-4">
        {/* separator design */}
        <LabeledInput
          label="Elválasztó stílusa:"
          description="Válassza ki, hogy milyen stílusa legyen az elválasztónak.
              Alapértelmezett: None">
          <SelectInput
            name="separatorDesign"
            options={initialData.separatorDesign}
            defaultValue={
              state?.current?.content?.[0]?.content?.[0]?.separatorDesign
                ? initialData.separatorDesign.find(
                    (x) =>
                      x.value ===
                      state?.current?.content?.[0]?.content?.[0]
                        ?.separatorDesign
                  )
                : initialData.separatorDesign[8]
            }
            placeholder="Elválasztó stílusa"
            ref={(r) => {
              inputRefs.current.separatorDesign = r;
              // addRef("imageLayers", r); // Explicitly add/update ref
            }}
          />
        </LabeledInput>
        {/* select image */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"imageSelect"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Kép kiválasztása: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Töltsön fel képet, vagy válasszon a médiatárból!
            </span>
          </label>
          <div className="flex flex-row w-3/4 justify-end items-center">
            {generatSelectedImage()}
            <Button
              onClick={() => showSelectImageModal(true)}
              className=" border border-gray-300 px-4 py-2 text-gray-500 hover:border-gray-600 hover:text-gray-900"
              btntext="Kép kiválasztása"></Button>
          </div>
        </div>
        {/* separator width % - default 100 */}
        <LabeledInput
          label={"Elválasztó szélessége:"}
          description={
            "Elválasztó szélességének beállítása. 0-100-ig választható érték %-ban. Alapértelmezett: 100"
          }>
          <TextInput
            type={"number"}
            name={"separatorWidth"}
            min={0}
            max={100}
            defaultValue={
              state?.current?.content?.length > 0
                ? state?.current?.content?.[0]?.content?.[0]?.separatorWidth ||
                  100
                : 100
            }
            ref={(r) => (inputRefs.current.separatorWidth = r)}
            onChange={(e) => handleInputMinMax(e)}
          />
        </LabeledInput>
        {/* separator height - default 0 */}
        <LabeledInput
          label={"Elválasztó magassága:"}
          description={
            " Elválasztó magasságának beállítása. Px, vh, vw, %, stb. Alapértelmezett: 0"
          }>
          <TextInput
            type={"text"}
            name={"separatorHeight"}
            defaultValue={
              state?.current?.content?.length > 0
                ? state?.current?.content?.[0]?.content?.[0]?.separatorHeight ||
                  0
                : 0
            }
            ref={(r) => (inputRefs.current.separatorHeight = r)}
          />
        </LabeledInput>
        {/* separator vastagsága px - default 0 */}
        <LabeledInput
          label={"Elválasztó vonal vastagsága:"}
          description={
            "Elválasztó vonal vastagságának beállítása. 0-100-ig választható érték px-ben. Alapértelmezett: 0"
          }>
          <TextInput
            type={"number"}
            name={"separatorThickness"}
            min={0}
            max={100}
            defaultValue={
              state?.current?.content?.length > 0
                ? state?.current?.content?.[0]?.content?.[0]
                    ?.separatorThickness || 0
                : 0
            }
            ref={(r) => (inputRefs.current.separatorThickness = r)}
            onChange={(e) => handleInputMinMax(e)}
          />
        </LabeledInput>
        {/* separator position: left - center - right */}
        <LabeledInput
          label="Elválasztó pozíciója:"
          description="Válassza ki, hogy milyen pozícióban legyen az elválasztó.
              Alapértelmezett: Közép">
          <SelectInput
            name="separatorPosition"
            options={initialData.separatorPosition}
            defaultValue={
              state?.current?.content?.[0]?.content?.[0]?.separatorPosition
                ? initialData.separatorPosition.find(
                    (x) =>
                      x.value ===
                      state?.current?.content?.[0]?.content?.[0]
                        ?.separatorPosition
                  )
                : initialData.separatorPosition[1]
            }
            placeholder="Elválasztó pozíciója"
            ref={(r) => {
              inputRefs.current.separatorPosition = r;
              // addRef("imageLayers", r); // Explicitly add/update ref
            }}
          />
        </LabeledInput>
        {/* separator padding top - bottom */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"textDistance"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Elválasztó eltartása: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Válassza ki, hogy az elválasztó milyen padding értéket vegyen fel
              a doboz tetejétől és aljától - (alapértelmezett érték: 0 0).
            </span>
          </label>
          <div className="flex flex-row flex-nowrap justify-evenly items-center">
            <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
              <span className="fill-gray-600 w-[2rem] h-[2rem]">
                {icons.up}
              </span>
              <input
                type="number"
                name="separatorPositionTop"
                id="separatorPositionTop"
                ref={(r) => (inputRefs.current.separatorPositionTop = r)}
                className="w-3/4 border h-max p-2"
                defaultValue={
                  state?.current?.content?.length > 0
                    ? state?.current?.content?.[0]?.content?.[0]
                        ?.separatorPositionTop || 0
                    : 0
                }
              />
            </div>
            <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
              <span className="fill-gray-600 w-[2rem] h-[2rem]">
                {icons.right}
              </span>
              <input
                type="number"
                name="separatorPositionRight"
                id="separatorPositionRight"
                ref={(r) => (inputRefs.current.separatorPositionRight = r)}
                className="w-3/4 border h-max p-2"
                defaultValue={
                  state?.current?.content?.length > 0
                    ? state?.current?.content?.[0]?.content?.[0]
                        ?.separatorPositionRight || 0
                    : 0
                }
              />
            </div>
            <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
              <span className="fill-gray-600 w-[2rem] h-[2rem]">
                {icons.down}
              </span>
              <input
                type="number"
                name="separatorPositionBottom"
                id="separatorPositionBottom"
                ref={(r) => (inputRefs.current.separatorPositionBottom = r)}
                className="w-3/4 border h-max p-2"
                defaultValue={
                  state?.current?.content?.length > 0
                    ? state?.current?.content?.[0]?.content?.[0]
                        ?.separatorPositionBottom || 0
                    : 0
                }
              />
            </div>
            <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
              <span className="fill-gray-600 w-[2rem] h-[2rem]">
                {icons.left}
              </span>
              <input
                type="number"
                name="separatorPositionLeft"
                id="separatorPositionLeft"
                ref={(r) => (inputRefs.current.separatorPositionLeft = r)}
                className="w-3/4 border h-max p-2"
                defaultValue={
                  state?.current?.content?.length > 0
                    ? state?.current?.content?.[0].content[0]
                        .separatorPositionLeft || 0
                    : 0
                }
              />
            </div>
          </div>
        </div>
        {/* separator color */}
        <LabeledInput
          label={"Elválasztó színének kiválasztása:"}
          description={
            "Megadhatja, hogy milyen színű legyen az elválasztó. Alapértelmezett: #212121"
          }>
          <ColorPickerComponent
            name={"separatorColor"}
            value={color}
            onChange={setColor}
          />
        </LabeledInput>
        {/* separator backgroundcolor */}
        <LabeledInput
          label={"Elválasztó háttér színének kiválasztása:"}
          description={
            "Megadhatja, hogy milyen színű legyen az elválasztó háttere. Alapértelmezett: transparent"
          }>
          <ColorPickerComponent
            name={"backgroundColor"}
            value={backgroundColor}
            onChange={setBackgroundColor}
          />
        </LabeledInput>
      </div>
      <ModalButton btntext={"Mentés"} callback={submitEditorSeparator} />
      <MediaGalleryModal
        isOpen={selectImageModal}
        onClose={() => showSelectImageModal(false)}
        action="separatorImage"
        onConfirmSelection={(selectedImages) =>
          setSelectedImage({ selectedImage: selectedImages })
        }
      />
    </div>
  );
};

export default EditorSeparator;
