import React, { useCallback, useState } from "react";
import Button from "@/components/shared/button/Button";
import LabeledInput from "@/components/shared/labeledinput/LabeledInput";
import TextInput from "@/components/shared/textinput/TextInput";
import SelectInput from "@/components/shared/selectinput/SelectInput";
import { initialData } from "@/data/data";
import Modal from "@/components/shared/modal/Modal";
import ModalButton from "@/components/shared/modal/ModalButton";
import { useEditorSlider } from "@/hooks/useEditorSlider";
import SliderImageList from "./SliderImageList";
import EditSliderImageOptions from "./EditSliderImageOptions";
import useDynamicManager from "@/store/managers/useDynamicManager";
import MediaGalleryModal from "@/components/media-store/GalleryModal/MediaGalleryModal";
import { useModalManager } from "@/hooks/useModalManager";

const EditorSlider = ({ toggleModal, modalName, type, lifecycleHandlers }) => {
  const [selectImageModal, setSelectImageModal] = useState(false);
  const { isModalOpen } = useModalManager();

  const { state, setSelectedSliderImages, resetSliderState } =
    useDynamicManager(type);
  const {
    editImageModal,
    editedImage,
    // Setters
    setEditImageModal,
    setEditedImage,
    // Handlers
    handleEditSingleImage,
    handleRemoveSingleImage,
    handleCloneSliderImage,
    handleUpdateSliderImage,
    handleUpdateSliderImageOrder,
    handleSliderSubmit,
    // Refs
    inputRefs,
  } = useEditorSlider({ toggleModal, modalName, type });

  const handleSetSelectedSliderImages = useCallback((images) => {
    const ret = [];
    images?.map((image) => ret.push({ image: image, options: {} }));

    setSelectedSliderImages({ selectedImages: ret });
  }, []);

  const displaySliderImages = useCallback(() => {
    return (
      <SliderImageList
        images={state?.sliderImages}
        onEdit={handleEditSingleImage}
        onClone={handleCloneSliderImage}
        onOrderChange={handleUpdateSliderImageOrder}
        onRemove={handleRemoveSingleImage}
      />
    );
  }, [state.sliderImages]);

  return (
    <div className="relative h-max">
      <div className="slider-wrapper flex flex-row flex-nowrap w-full h-full">
        {/* Left side: list of selected images */}
        <div className="w-1/3 bg-gray-100 h-full flex flex-col p-4">
          <h5 className="text-normal text-gray-400 w-full font-bold">
            Kép lista:
          </h5>
          <div className="flex flex-col gap-4 overflow-y-auto max-h-[80vh] pr-1">
            {displaySliderImages()}
          </div>
        </div>
        <div className="w-2/3 h-full flex flex-col gap-6 border-l p-6 bg-white overflow-y-auto max-h-[90vh] rounded">
          {/* image upload */}
          <div className="border border-gray-200 rounded-lg p-4 shadow-sm bg-gray-50">
            <p className="text-lg font-semibold text-gray-700 mb-2">
              📤 Képfeltöltés
            </p>
            <div className="flex flex-row justify-between items-start gap-4">
              <div className="flex-1">
                <p className="text-sm text-gray-600 mb-2 font-medium">
                  Töltsön fel képet, vagy válasszon a médiatárból!
                </p>
              </div>
              <Button
                onClick={() => setSelectImageModal(true)}
                className="border border-gray-300 px-4 py-2 text-gray-600 hover:border-gray-600 hover:text-gray-900"
                btntext="Kép(ek) kiválasztása"
              />
            </div>
          </div>
          <div className="border border-gray-200 rounded-lg p-4 shadow-sm bg-gray-50">
            <p className="text-lg font-semibold text-gray-700 mb-4">
              ⚙️ Alapbeállítások
            </p>
            <div className="grid grid-cols-1 gap-4">
              {/* image grayscale */}
              <LabeledInput
                label={"Kép(ek) szürkeárnyalat érték:"}
                description={
                  "Megadhatja, hogy a képeknek legyen-e szürkeárnyalata. Érték: 0-100. (Alapértelmezett: 0)"
                }>
                <TextInput
                  type={"text"}
                  name={"imageGrayScale"}
                  defaultValue={
                    state.current[0]?.content[0]?.imageGrayScale ?? 0
                  }
                  ref={(r) => (inputRefs.current.imageGrayScale = r)}
                />
              </LabeledInput>
              {/* image alignment */}
              <LabeledInput
                label={"Képek elrendezése:"}
                description={
                  "Válassza ki, hogy a slider képei hogyan legyenek elrendezve. (alapértelmezett: Középre zárt)"
                }>
                <SelectInput
                  name="alignment"
                  options={initialData.imageAlign}
                  defaultValue={
                    state.current?.content?.[0]?.content?.[0].alignment
                      ? initialData.imageAlign.find(
                          (x) =>
                            x.value ===
                            state.current?.content?.[0]?.content?.[0].alignment
                        )
                      : initialData.imageAlign[1]
                  }
                  placeholder="Válasszon kép elrendezést"
                  ref={(r) => {
                    inputRefs.current.alignment = r;
                  }}
                />
              </LabeledInput>
              {/* slider autoplay */}
              <LabeledInput
                label={"Slider automatikus lejátszása:"}
                description={
                  " Válassza ki, hogy a slider automatikusan elinduljon-e. (alapértelmezett: Igen)"
                }>
                <SelectInput
                  name="autoplay"
                  options={initialData.bool}
                  defaultValue={
                    state.current?.content?.[0]?.content?.[0].autoplay
                      ? initialData.bool.find(
                          (x) =>
                            x.value ===
                            state.current?.content?.[0]?.content?.[0].autoplay
                        )
                      : initialData.bool[0]
                  }
                  placeholder="Slider automatikus lejátszása"
                  ref={(r) => {
                    inputRefs.current.autoplay = r;
                  }}
                />
              </LabeledInput>
              {/* slider items navigation */}
              <LabeledInput
                label={"Slider navigáció jelzése:"}
                description={
                  "Válassza ki, hogy a slider navigációt megjelenítse-e a slider. (alapértelmezett: nem)"
                }>
                <SelectInput
                  name="navigation"
                  options={initialData.bool}
                  defaultValue={
                    state.current?.content?.[0]?.content?.[0].navigation
                      ? initialData.bool.find(
                          (x) =>
                            x.value ===
                            state.current?.content?.[0]?.content?.[0].navigation
                        )
                      : initialData.bool[1]
                  }
                  placeholder="Slider navigáció jelzése"
                  ref={(r) => {
                    inputRefs.current.navigation = r;
                  }}
                />
              </LabeledInput>
            </div>
          </div>
          {/* Responsive settings */}
          <div className="border border-gray-200 rounded-lg p-4 shadow-sm bg-gray-50">
            <p className="text-lg font-semibold text-gray-700 mb-4">
              📱 Reszponzív beállítások
            </p>
            {/* slider items per view */}
            <LabeledInput
              label={"Slider elemek száma:"}
              description={
                "Válassza ki, hogy a slider képei közel mennyi látszódjon egyszerre az oldalon. (alapértelmezett: 1)"
              }>
              <SelectInput
                name="slidesPerView"
                options={initialData.slidesPerView}
                defaultValue={
                  state.current?.content?.[0]?.content?.[0].slidesPerView
                    ? initialData.slidesPerView.find(
                        (x) =>
                          x.value ===
                          state.current?.content?.[0]?.content?.[0]
                            .slidesPerView
                      )
                    : initialData.slidesPerView[0]
                }
                placeholder="Slider elemek száma"
                ref={(r) => {
                  inputRefs.current.slidesPerView = r;
                }}
              />
            </LabeledInput>
            <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
              {["Xxs", "Xs", "Sm", "Md", "Lg", "Xl", "2Xl"].map((size) => (
                <LabeledInput
                  key={size}
                  label={`Slider elemek száma - ${size.toUpperCase()}:`}>
                  <SelectInput
                    name={`sliderBreakpoint${size}`}
                    options={initialData.slidesPerView}
                    defaultValue={
                      initialData.slidesPerView.find(
                        (x) =>
                          x.value ===
                          state.current?.content?.[0]?.content?.[0]?.[
                            `sliderBreakpoint${size}`
                          ]
                      ) ?? initialData.slidesPerView[0]
                    } // pull from `state.current`
                    placeholder={`Slider - ${size}`}
                    ref={(r) => {
                      inputRefs.current[`sliderBreakpoint${size}`] = r;
                    }}
                  />
                </LabeledInput>
              ))}
            </div>
          </div>
          {/* Advanced */}
          <div className="border border-gray-200 rounded-lg p-4 shadow-sm bg-gray-50">
            <p className="text-lg font-semibold text-gray-700 mb-4">
              🧩 Haladó beállítások
            </p>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {/* slider items smooth height */}
              <LabeledInput
                label={"Slider elemek azonos magassága:"}
                description={
                  "Válassza ki, hogy a slider képei azonos magasságúak legyenek-e. (alapértelmezett: Nem)"
                }>
                <SelectInput
                  name="smoothHeight"
                  options={initialData.bool}
                  defaultValue={
                    state.current?.content?.[0]?.content?.[0].smoothHeight
                      ? initialData.bool.find(
                          (x) =>
                            x.value ===
                            state.current?.content?.[0]?.content?.[0]
                              .smoothHeight
                        )
                      : initialData.bool[1]
                  }
                  placeholder="Slider elemek azonos magassága"
                  ref={(r) => {
                    inputRefs.current.smoothHeight = r;
                  }}
                />
              </LabeledInput>

              {/* slider items space between */}
              <LabeledInput
                label={"Slider elemek közötti távolság:"}
                description={
                  "Válassza ki, hogy a slider képei milyen távolságban legyenek egymástól. (alapértelmezett: 10)"
                }>
                <SelectInput
                  name="spaceBetween"
                  options={initialData.spaceBetween}
                  defaultValue={
                    state.current?.content?.[0]?.content?.[0].spaceBetween
                      ? initialData.spaceBetween.find(
                          (x) =>
                            x.value ===
                            state.current?.content?.[0]?.content?.[0]
                              .spaceBetween
                        )
                      : initialData.spaceBetween[0]
                  }
                  placeholder="Válassza ki a slider képek távolságát"
                  ref={(r) => {
                    inputRefs.current.spaceBetween = r;
                  }}
                />
              </LabeledInput>
              {/* slider items mouseWheelControl */}
              <LabeledInput
                label={"Slider egér görgővel vezérelhető:"}
                description={
                  " Válassza ki, hogy a slider egér görgővel vezérelhető legyen-e. (alapértelmezett: igen)"
                }>
                <SelectInput
                  name="mouseWheelControl"
                  options={initialData.bool}
                  defaultValue={
                    state.current?.content?.[0]?.content?.[0].mouseWheelControl
                      ? initialData.bool.find(
                          (x) =>
                            x.value ===
                            state.current?.content?.[0]?.content?.[0]
                              .mouseWheelControl
                        )
                      : initialData.bool[0]
                  }
                  placeholder="Slider egér görgővel vezérelhető"
                  ref={(r) => {
                    inputRefs.current.mouseWheelControl = r;
                  }}
                />
              </LabeledInput>
              {/* slider items keyboardControl */}
              <LabeledInput
                label={"Slider billentyűzettel vezérelhető:"}
                description={
                  "Válassza ki, hogy a slider billentyűzettel vezérelhető legyen-e. (alapértelmezett: igen)"
                }>
                <SelectInput
                  name="keyboardControl"
                  options={initialData.bool}
                  defaultValue={
                    state.current?.content?.[0]?.content?.[0].keyboardControl
                      ? initialData.bool.find(
                          (x) =>
                            x.value ===
                            state.current?.content?.[0]?.content?.[0]
                              .keyboardControl
                        )
                      : initialData.bool[0]
                  }
                  placeholder="Slider billentyűzettel vezérelhető"
                  ref={(r) => {
                    inputRefs.current.keyboardControl = r;
                  }}
                />
              </LabeledInput>
              {/* slider items pagination */}
              <LabeledInput
                label={"Slider oldalak jelzése:"}
                description={
                  "Válassza ki, hogy a slider oldalakat megjelenítse-e a slider. (alapértelmezett: igen)"
                }>
                <SelectInput
                  name="pagination"
                  options={initialData.bool}
                  defaultValue={
                    state.current?.content?.[0]?.content?.[0].pagination
                      ? initialData.bool.find(
                          (x) =>
                            x.value ===
                            state.current?.content?.[0]?.content?.[0].pagination
                        )
                      : initialData.bool[0]
                  }
                  placeholder="Slider oldalak jelzése"
                  ref={(r) => {
                    inputRefs.current.pagination = r;
                  }}
                />
              </LabeledInput>

              {/* slider items zoom */}
              <LabeledInput
                label={"Slider zoom:"}
                description={
                  "Válassza ki, hogy a slider képei zoomolhatóak legyenek-e. (alapértelmezett: igen)"
                }>
                <SelectInput
                  name="zoom"
                  options={initialData.bool}
                  defaultValue={
                    state.current?.content?.[0]?.content?.[0].zoom
                      ? initialData.bool.find(
                          (x) =>
                            x.value ===
                            state.current?.content?.[0]?.content?.[0].zoom
                        )
                      : initialData.bool[1]
                  }
                  placeholder="Slider zoom"
                  ref={(r) => {
                    inputRefs.current.zoom = r;
                  }}
                />
              </LabeledInput>
              {/* slider items animation */}
              <LabeledInput
                label={"Slider elemek animációja:"}
                description={
                  "Válassza ki a slider elemek animációját. (alapértelmezett: Nincs animáció)"
                }>
                <SelectInput
                  name="animationTypes"
                  options={initialData.animation}
                  defaultValue={
                    state.current?.content?.[0]?.content?.[0].animationTypes
                      ? initialData.animation.find(
                          (x) =>
                            x.value ===
                            state.current?.content?.[0]?.content?.[0]
                              .animationTypes
                        )
                      : initialData.animation[0]
                  }
                  placeholder="Slider elemek animációja"
                  ref={(r) => {
                    inputRefs.current.animationTypes = r;
                  }}
                />
              </LabeledInput>
            </div>{" "}
          </div>
        </div>
      </div>
      <ModalButton btntext={"Mentés"} callback={() => handleSliderSubmit()} />

      {/* image modal */}
      <MediaGalleryModal
        isOpen={selectImageModal}
        onClose={() => {
          toggleModal(modalName);
        }}
        action="pageEditorSliderImages"
        onConfirmSelection={(selectedImages) => {
          handleSetSelectedSliderImages(selectedImages);
          setSelectImageModal(false); // ✅ Close the modal manually
        }}
      />

      {/* edited image modal */}
      <Modal
        title={"Slider kép szerkesztése"}
        show={editImageModal}
        onClose={() => setEditImageModal(false)}
        height={"100%"}>
        {editedImage && Object.keys(editedImage).length > 0 && (
          <EditSliderImageOptions
            editedImage={editedImage}
            setEditedImage={setEditedImage}
            showEditImageModal={setEditImageModal}
            handleUpdateSliderImage={handleUpdateSliderImage}
          />
        )}
      </Modal>
    </div>
  );
};

export default EditorSlider;
