"use client";
import React, { useCallback, useEffect, useState } from "react";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";
import useDynamicManager from "@/store/managers/useDynamicManager";
import Modal from "@/components/shared/modal/Modal";
import { initialData } from "@/data/data";
import icons from "@/assets/icons/icons";
import { useModalManager } from "@/hooks/useModalManager";
import Button from "@/components/shared/button/Button";
import { useForm } from "@/hooks/custom-form-hook";
import Container from "@/components/editor/container/Container";
import useContainerActions from "@/hooks/useContainerActions";
import ArticleEditorFormFields from "../article-box/ArticleEditorFormFields";
import ImageWithFallback from "@/components/shared/image/ImageWithFallback";
import Input from "@/components/shared/input/Input";
import MediaGalleryModal from "@/components/media-store/GalleryModal/MediaGalleryModal";
import { useRouter, useParams } from "next/navigation";
import SuneditorComponent from "@/components/shared/texteditor/SuneditorComponent";

function formatLocalDateTime(date) {
  const pad = (n) => String(n).padStart(2, "0");
  const local = new Date(date);
  const year = local.getFullYear();
  const month = pad(local.getMonth() + 1);
  const day = pad(local.getDate());
  const hours = pad(local.getHours());
  const minutes = pad(local.getMinutes());
  return `${year}-${month}-${day}T${hours}:${minutes}`;
}

const EditArticle = () => {
  const [articleData, setArticleData] = useState(null);
  const [mediaStoreOpen, setMediaStoreOpen] = useState(false);
  const [boxColumns, setBoxColumns] = useState(null);
  const [excerpt, setExcerpt] = useState(articleData?.excerpt);
  const [formState, handleInput, handleSetInputData] = useForm([], false);
  const { toggleModal, isModalOpen } = useModalManager();
  const {
    state,
    resetFeaturedImageState,
    setFeaturedImage,
    loadArticleEditorInitialData,
    ...rest
  } = useDynamicManager("articleEditor");
  const params = useParams();
  const { handleAddContainer } = useContainerActions(
    initialData,
    toggleModal,
    "articleEditor"
  );
  const router = useRouter();
  useEffect(() => {
    const fetchArticleEditorData = async () => {
      const response = await ApiCall.articles.getEditArticle(params.id);
      if (response.status === 200) {
        loadArticleEditorInitialData(response?.data?.body?.data);
        setArticleData(response?.data?.body?.data);
        toast.success("Cikkszerkesztő adatok betöltve.");
      } else {
        toast.error("Cikkszerkesztő adatok betöltése sikertelen.");
      }
    };
    fetchArticleEditorData();
  }, [params.id]);

  useEffect(() => {
    const fetchBoxColumnsData = async () => {
      const response = await ApiCall.boxColumns.getAll();
      if (response.status === 200) {
        setBoxColumns(response?.data?.columnList);
        toast.success("Oszlop adatok betöltve.");
      } else {
        toast.error("Oszlop adatok betöltése sikertelen.");
      }
    };
    fetchBoxColumnsData();
  }, []);

  const [checkboxes] = useState([
    { title: "Publikus", name: "isPublished", checked: true },
  ]);
  const handleSetArticleFeaturedImage = useCallback((image) => {
    setFeaturedImage({ image: image });
  }, []);

  const handleRemoveFeaturedImage = useCallback(() => {
    resetFeaturedImageState();
  }, []);

  const handleUpdateArticle = useCallback(() => {
    const ret = {
      ...formState.inputs,
      ...state,
      excerpt,
    };
    const sendData = async () => {
      const response = await ApiCall.articles.postEditArticle(params?.id, ret);
      if (response.status === 200 && !response.data.error) {
        toast.success("Cikk sikeresen frissítve!");
        resetFeaturedImageState();
        router.push("/admin/articles/list");
      } else {
        toast.error(response.error || "Cikk mentése nem sikerült.");
      }
    };
    sendData();
    console.log("formState.inputs", formState.inputs, "state", state);
  }, [formState, state, router, excerpt, params]);
  console.log("state", state);
  return (
    <div className="w-full grid grid-cols-1 md:grid-cols-4 gap-4">
      {articleData && (
        <>
          {" "}
          <div className="col-span-3">
            {/* left */}
            <ArticleEditorFormFields
              data={articleData}
              handleInput={handleInput}
              initialValid={false}
            />
            <label
              htmlFor="excerpt"
              className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
              Kivonat
            </label>
            <SuneditorComponent
              lang={"en"}
              name={"excerpt"}
              defaultValue={articleData.excerpt}
              placeholder={"Ide gépelheti a szöveget..."}
              onChange={(value) => setExcerpt(value)}
              options={{
                buttonList: [
                  ["font", "fontSize", "formatBlock"],
                  [
                    "bold",
                    "underline",
                    "italic",
                    "strike",
                    "subscript",
                    "superscript",
                  ],
                  ["align", "list", "table"],
                  ["fontColor", "hiliteColor"],
                  ["outdent", "indent"],
                  ["undo", "redo"],
                  ["removeFormat"],
                  ["link", "image"],
                  ["preview", "print"],
                  ["fullScreen", "showBlocks", "codeView"],
                ],
                tableWidth: "100%",
                tableStyles: {
                  border: "1px solid black",
                },
              }}
            />
            {state &&
              state?.containerData?.map((c, idx) => {
                return (
                  <Container
                    key={idx}
                    data={c}
                    alldata={state?.containerData}
                    parent={idx}
                    handleTitleChange={rest.setContainerTitle}
                    type={"articleEditor"}
                  />
                );
              })}
            <Button
              className="w-full flex flex-row flex-nowrap justify-center items-center cursor-pointer border border-gray-600/50 text-white hover:bg-green-600 hover:text-white bg-gray-600 py-2 px-4 font-bold my-4"
              onClick={() => toggleModal("containerModal")}
              btntext="Konténer hozzáadása"
              icon={icons.add}
              iconclass="flex w-[1.5rem] h-[1.5rem] fill-white stroke-white ml-4"></Button>
          </div>
          {/* right */}
          <div className="col-span-1">
            <div className="w-full flex flex-col">
              {/* Featured Image */}
              <div className="w-full h-80 border-2 border-dashed relative mb-4">
                {state.featuredImage?.length > 0 && (
                  <ImageWithFallback
                    src={ApiCall.mediaStore.getResized(
                      "/4/4/1",
                      state.featuredImage[0].originalname
                    )}
                    alt="Featured"
                    fill
                    className="object-cover"
                  />
                )}
                <span
                  onClick={() => setMediaStoreOpen(true)}
                  className="absolute top-2 left-2 w-[3rem] h-[3rem] hover:border-1 bg-white rounded-full flex justify-center items-center cursor-pointer">
                  {icons.change}
                </span>
                <span
                  onClick={handleRemoveFeaturedImage}
                  className="absolute top-2 right-2 w-[2rem] h-[2rem] !fill-red !stroke-red rounded-full flex justify-center items-center cursor-pointer">
                  {icons.clear}
                </span>
              </div>
              {/* checkboxes */}
              <div className="w-full flex flex-row flex-wrap justify-center items-center">
                {checkboxes?.map((checkbox, index) => {
                  console.log(
                    "!!state[checkbox.name]",
                    !!state[checkbox.name],
                    checkbox.name
                  );
                  return (
                    <div
                      key={checkbox.name}
                      className="form-group w-1/2 flex flex-col my-4 justify-center items-center">
                      <Input
                        element="checkbox"
                        type="checkbox"
                        name={checkbox.name}
                        id={checkbox.name}
                        placeholder={checkbox.title}
                        onInput={handleInput}
                        initialValid={true}
                        className="justify-center items-center"
                        inputcls="outline-blue-500 w-8 h-8"
                        isChecked={!!state[checkbox.name]}
                      />
                    </div>
                  );
                })}
              </div>
              {/* publish date */}
              <Input
                element="input"
                type="datetime-local"
                id="publishDate"
                name="publishDate"
                placeholder="Publikálás ideje"
                className="w-100"
                inputcls="w-full border-1 text-center p-2 my-2"
                errorText="Érvénytelen dátum"
                onInput={handleInput}
                initialValid={true}
                value={
                  articleData?.publishDate
                    ? formatLocalDateTime(articleData.publishDate)
                    : ""
                }
              />
            </div>
          </div>
          <div className="w-full col-span-4">
            {formState.isValid && (
              <button
                onClick={() => handleUpdateArticle()}
                className="w-full px-4 py-2 flex flex-row justify-center items-center bg-green-500/75 hover:bg-green-500 text-white">
                Cikk mentése
              </button>
            )}
          </div>
        </>
      )}

      {/**************************  Modals  **********************************/}
      {/* add container modal */}
      <Modal
        show={isModalOpen("containerModal")}
        onClose={() => toggleModal("containerModal")}
        title="Válasszon az alábbi konténerek közül:"
        height={"100%"}>
        <div className="flex flex-row flex-wrap justify-evenly">
          {boxColumns?.map((c, i) => {
            return (
              <div
                key={i}
                className="flex flex-row flex-nowrap mx-px my-4 w-1/6 border pointer-events-auto justify-between items-center border border-dashed border-gray-500/50 hover:border-gray-500 hover:cursor-pointer hover:bg-gray-600/75"
                onClick={() => {
                  handleAddContainer(c);
                }}>
                <div className="flex flex-row flex-nowrap w-full m-1 pointer-events-none">
                  {c?.columnSize?.map((col, idx) => {
                    const colX = col.x;
                    const colY = col.y;

                    return (
                      <div
                        key={idx}
                        className={`h-20 flex flex-row flex-nowrap justify-center items-center border border-white p-2 bg-gray-600/50 `}
                        style={{
                          width: `${((colX / colY) * 100).toFixed(0)}%`,
                        }}>
                        <p className="w-full flex flex-row justify-center items-center text-white text-sm">{`${colX}/${colY}`}</p>
                      </div>
                    );
                  })}
                </div>
              </div>
            );
          })}
        </div>
      </Modal>
      <MediaGalleryModal
        isOpen={mediaStoreOpen}
        onClose={() => setMediaStoreOpen(false)}
        action="articleFeatured"
        onConfirmSelection={(selectedImage) =>
          handleSetArticleFeaturedImage(selectedImage)
        }
      />
    </div>
  );
};

export default EditArticle;
