import React, { useCallback, useEffect, useState } from "react";
import { useSelector, useDispatch } from "react-redux";
import ColumnOption from "./ColumnOption";
// import useColumnActions from "@/hooks/useColumnActions";
import { useModalManager } from "@/hooks/useModalManager";
import Modal from "@/components/shared/modal/Modal";
import { initialData } from "@/data/data";
import icons from "@/assets/icons/icons";
import removeDuplicates from "@/utils/removeDuplicatedColSizes";
import ColumnSettings from "../columnsettings/ColumnSettings";
import useDynamicManager from "@/store/managers/useDynamicManager";
import useColumnSize from "@/hooks/useColumnSize";
import { usePageEditorContextManager } from "@/context/PageEditorContext";

const ColumnOptionsLayout = ({
  data,
  size,
  parent,
  index,
  type,
  lifecycleHandlers,
}) => {
  const { boxColumnsList } = usePageEditorContextManager();
  const [colSizeList, setColSizeList] = useState(boxColumnsList?.columnList);
  const [showColSizeList, setShowColSizeList] = useState(false);
  const [showColSettings, setShowColSettings] = useState(false);
  const [filteredSizes, setFilteredSizes] = useState(null);
  const { column, colSize, setColSize } = useColumnSize(data);

  const [columnToRemove, setColumnToRemove] = useState(null);
  const { toggleModal, isModalOpen } = useModalManager();

  const {
    state,
    changeColumnSize,
    removeColumn,
    updateColumnOrder,
    cloneColumn,
  } = useDynamicManager(type);
  const container = state.containerData?.[parent];
  const changeColSizeHandler = (size) => {
    setColSize(`${size.x}/${size.y}`);
    changeColumnSize({ parentId: parent, innerId: index, size });
    setShowColSizeList(false);
  };

  useEffect(() => {
    const sizeList = [];

    colSizeList?.map((col) => {
      if (col) {
        sizeList.push({ id: col._id, sizes: col.columnSize });
      }
    });
    const filteredSizeList = removeDuplicates(sizeList, colSizeList);

    setFilteredSizes(filteredSizeList);
  }, [colSizeList]);

  const handleRemoveColumnClick = useCallback(() => {
    if (columnToRemove) {
      removeColumn({
        parentId: parent,
        innerId: index,
        size: size,
      }); // Call the removal function from the hook
      setColumnToRemove(null); // Clear the temporary state
      toggleModal("deleteColumnConsent"); // Close the modal
    }
  }, [columnToRemove, removeColumn, toggleModal, parent, index]);
  return (
    <div className="flex flex-row justify-between items-start bg-white w-full relative border-t border-t-[#dedede]  mb-4">
      <ColumnOption
        content={colSize}
        handleClick={() => setShowColSizeList((prev) => !prev)}
      />
      {showColSizeList && filteredSizes && (
        <SizeList
          options={filteredSizes}
          changeHandler={changeColSizeHandler}
        />
      )}
      <div className="sticky top-0 left-0 h-full w-max flex flex-row flex-wrap justify-start items-center z-50 bg-[#198fd9]/50">
        {/* column settings */}
        <ColumnOption
          icon={icons.settings}
          cls={"flex self-start"}
          handleClick={() => setShowColSettings(!showColSettings)}
        />
        {showColSettings && (
          <>
            {/* edit column settings */}
            <ColumnOption
              icon={icons.edit}
              cls={"flex self-start"}
              handleClick={() => toggleModal("columnSettings")}
              title={"Oszlop szerkesztése"}
            />
            {/* remove column */}
            <ColumnOption
              handleClick={() => {
                setColumnToRemove(data);
                setShowColSettings(!showColSettings);
                toggleModal("deleteColumnConsent");
              }}
              icon={icons.trash}
              iconCls={
                "!fill-[#198fd9] !stroke-[#198fd9] !hover:fill-white !hover:stroke-white !hover:bg-white w-[2rem] h-[2rem] p-1"
              }
              title={"Oszlop eltávolítása"}
            />
            {/* clone */}
            <ColumnOption
              icon={icons.clone}
              iconCls={"fill-[#198fd9] stroke-[#198fd9] w-[2rem] h-[2rem] p-1"}
              handleClick={() => {
                cloneColumn({
                  parentId: parent,
                  innerId: index,
                  sizes: data,
                  options:
                    container?.[index]?.options ??
                    initialData.editor.column.defaults,
                });
              }}
            />
            {/* update order */}
            {index != 0 && index !== container?.size?.length - 1 ? (
              <>
                <ColumnOption
                  handleClick={() =>
                    updateColumnOrder({
                      parentId: parent,
                      innerId: index,
                      start: index - 1,
                      end: index,
                    })
                  }
                  index={index}
                  icon={icons.left}
                  iconCls={
                    "fill-[#198fd9] stroke-[#198fd9] w-[2rem] h-[2rem] p-1"
                  }
                />
                <ColumnOption
                  handleClick={() =>
                    updateColumnOrder({
                      parentId: parent,
                      innerId: index,
                      start: index + 1,
                      end: index,
                    })
                  }
                  index={index}
                  icon={icons.right}
                  iconCls={
                    "fill-[#198fd9] stroke-[#198fd9] w-[2rem] h-[2rem] p-1"
                  }
                />
              </>
            ) : index === container?.size?.length - 1 ? (
              <ColumnOption
                handleClick={() =>
                  updateColumnOrder({
                    parentId: parent,
                    innerId: index,
                    start: index - 1,
                    end: index,
                  })
                }
                index={index}
                icon={icons.left}
                iconCls={
                  "fill-[#198fd9] stroke-[#198fd9] w-[2rem] h-[2rem] p-1"
                }
              />
            ) : (
              <ColumnOption
                handleClick={() =>
                  updateColumnOrder({
                    parentId: parent,
                    innerId: index,
                    start: index + 1,
                    end: index,
                  })
                }
                index={index}
                icon={icons.right}
                iconCls={
                  "fill-[#198fd9] stroke-[#198fd9] w-[2rem] h-[2rem] p-1"
                }
              />
            )}
          </>
        )}{" "}
      </div>
      {/* remove consent */}
      <Modal
        title={`Biztosan szeretné eltávolítani ezt az oszlopot? <span style="width: 100%">A művelet nem vonható vissza!</span>`}
        show={isModalOpen("deleteColumnConsent")}
        onClose={() => toggleModal("deleteColumnConsent")}>
        <div className="flex flex-row flex-wrap justify-evenly h-max">
          <button
            className="px-4 py-2 border border-gray-200 text-white hover:bg-red-500/50 bg-red-500"
            onClick={() => handleRemoveColumnClick()}>
            Törlés
          </button>
          <button
            className="px-4 py-2 border border-gray-200 text-black hover:bg-gray-100/50 bg-gray-100"
            onClick={() => toggleModal("deleteColumnConsent")}>
            Mégsem
          </button>
        </div>
      </Modal>
      {/* column edit */}
      <Modal
        title={"Oszlop beállításai"}
        show={isModalOpen("columnSettings")}
        onClose={() => toggleModal("columnSettings")}
        height={"100%"}>
        <ColumnSettings
          parent={parent}
          innerid={index}
          setShowColumnSettings={() => toggleModal("columnSettings")}
          toggleModal={toggleModal}
          modalName={"column-options"}
          type={type}
          lifecycleHandlers={lifecycleHandlers}
        />
      </Modal>
    </div>
  );
};
const SizeList = ({ options, changeHandler }) => (
  <div className="grid grid-cols-6 gap-1 absolute bottom-10 w-[300px] bg-white p-2">
    {options?.map((size, idx) => (
      <p
        key={idx}
        className="p-2 border cursor-pointer hover:bg-gray-600 hover:text-white col-span-1 justify-center items-center"
        onClick={() => changeHandler(size.size)}>
        {size.size.x}/{size.size.y}
      </p>
    ))}
  </div>
);
export default ColumnOptionsLayout;
