import React, { useEffect, useState, useCallback, useRef } from "react";
import Select from "react-select";
import { initialData } from "@/data/data";
import icons from "@/assets/icons/icons";
import dynamic from "next/dynamic";
const ColorPicker = dynamic(() => import("react-best-gradient-color-picker"), {
  ssr: false,
});
import ApiCall from "@/api/api-calls";
import Image from "next/image";
import MediaObject from "@/components/shared/mediaobject/MediaObject";
import { useModalManager } from "@/hooks/useModalManager";
import ModalButton from "@/components/shared/modal/ModalButton";
import useDynamicManager from "@/store/managers/useDynamicManager";
import MediaGalleryModal from "@/components/media-store/GalleryModal/MediaGalleryModal";
const InputField = ({
  label,
  id,
  description,
  defaultValue,
  inputRefs,
  index,
}) => (
  <div className="flex flex-row w-full mb-4 pb-4 border-b">
    <label htmlFor={id} className="w-1/4 text-gray-400 pr-2">
      <p className="font-bold">{label}</p>
      {description && <p className="text-sm text-gray-600">{description}</p>}
    </label>
    <input
      type="text"
      id={id}
      name={id}
      className="w-3/4 border p-2"
      defaultValue={defaultValue}
      ref={inputRefs}
    />
  </div>
);
const ColumnSettings = ({
  setShowColumnSettings,
  parent,
  innerid,
  toggleModal,
  modalName,
  type,
  lifecycleHandlers,
}) => {
  const inputRefs = useRef([]);
  let index = 0;
  const [activeTab, setActiveTab] = useState(1);
  const [selectImageModal, showSelectImageModal] = useState(false);

  const { state, updateColumnOptions, setSelectedImage, resetImageState } =
    useDynamicManager(type);
  const currentColumn = state?.containerData?.[parent]?.content?.[innerid];
  const columnOptions = currentColumn?.options ?? {};
  const [columnLayoutMode, setColumnLayoutMode] = useState(
    columnOptions?.columnLayoutMode
  );
  const [layoutMode, setLayoutMode] = useState(
    columnOptions?.columnLayoutMode ?? "flex"
  );
  // if (!currentColumn?.options) {
  //   return <div className="p-4">Betöltés...</div>;
  // } // or a loader

  const [columnLinkColor, setColumnLinkColor] = useState(
    columnOptions?.columnLinkColor ?? "rgba((33,33,33, 1)"
  );

  const [columnBorderColor, setColumnBorderColor] = useState(
    columnOptions?.columnBorderColor ?? "rgba(8,145,178, 1)"
  );
  const [columnBackgroundColor, setColumnBackgroundColor] = useState(
    columnOptions?.columnBackgroundColor ?? "rgba(255, 255, 255, 0)"
  );
  const [columnBoxShadowColor, setColumnBoxShadowColor] = useState(
    columnOptions?.columnBoxShadowColor ?? "rgba(255, 255, 255, 0)"
  );
  const handleInputMinMax = (e) => {
    if (parseInt(e.target.value) > parseInt(e.target.max)) {
      e.target.value = parseInt(e.target.max);
    }
    if (parseInt(e.target.value) < parseInt(e.target.min)) {
      e.target.value = parseInt(e.target.min);
    }
    e.target.value = parseInt(e.target.value);
    return e.target.value;
  };
  const handleTabClick = (tabNumber) => {
    setActiveTab(tabNumber);
  };
  const handleImageRemove = () => {
    resetImageState();
  };

  const generateSelectedImage = useCallback(() => {
    const i = 0;

    if (state?.selectedImage && state?.selectedImage?._id) {
      return state?.selectedImage?.props?.mimetype === "application/pdf" ? (
        <div
          className={`flex flex-col mr-2 mb-2 items-center border-1 w-full`}
          key={state?.selectedImage._id}>
          <MediaObject
            data={state?.selectedImage}
            index={i}
            data-index={i}
            className={"w-full"}
            id={state?.selectedImage._id}
          />
          <p className="break-all p-1">
            {" "}
            {state?.selectedImage?.props?.basename ??
              selectImageModal.originalname}
          </p>
          <input
            type="hidden"
            name="selectedFile"
            defaultValue={state?.selectedImage}
          />

          <button onClick={handleImageRemove} className="text-red-500 my-4">
            Eltávolítás
          </button>
        </div>
      ) : (
        <div
          className={`flex flex-col mr-2 mb-2 items-center border-1 w-full`}
          key={i}>
          <Image
            src={ApiCall.mediaStore.getResized(
              "/2/2/0",
              state?.selectedImage?.originalname
            )}
            width={150}
            height={100}
            alt={state?.selectedImage?.originalname ?? ""}
            effect="blur"
            loading="lazy"
          />
          <p className="break-all p-1">{state?.selectedImage.props.filename}</p>
          <input
            type="hidden"
            name="selectedFile"
            defaultValue={state?.selectedImage}
          />
          <button onClick={handleImageRemove}>Eltávolítás</button>
        </div>
      );
    }
  }, [state?.selectedImage, handleImageRemove, selectImageModal]);

  const submitColumnOptions = useCallback(() => {
    const content = {
      type: "columnOptions",
      columnLinkColor: columnLinkColor,
      columnBorderColor: columnBorderColor,
      columnBackgroundColor: columnBackgroundColor,
      columnBoxShadowColor: columnBoxShadowColor,
      backgroundImage: state?.selectedImage,
    };
    // Collect grid span inputs
    const gridSpans = { col: {}, row: {} };

    inputRefs.current.forEach((input) => {
      if (input && input.name?.startsWith("gridSpan")) {
        const [, type, bp] = input.name.split("-"); // e.g., gridSpan-col-sm => [gridSpan, col, sm]
        const value = parseInt(input.value);
        if (!isNaN(value)) {
          gridSpans[type][bp] = value;
        }
      }
    });
    content.gridSpans = gridSpans;
    inputRefs?.current?.map((c, i) => {
      if (c) {
        if (c.props) {
          if (Array.isArray(c.props.value)) {
            content[c.props.name] = [];
            if (c?.props?.value?.length === 1) {
              content[c.props.name] = c.props.value?.[0].value;
            } else {
              c?.props?.value?.map((elem) => {
                content[c.props.name]?.push(elem.value);
              });
            }
          } else {
            content[c.props.name] = c.props.value.value;
          }
        } else {
          content[c.name] = c.value;
        }
      }
    });
    console.log("submit column options", content);
    updateColumnOptions({
      parentId: parent,
      innerId: innerid,
      options: content,
    });
    resetImageState();

    setShowColumnSettings();
  }, [
    innerid,
    parent,
    setShowColumnSettings,
    state?.selectedImage,
    columnLinkColor,
    columnBorderColor,
    columnBackgroundColor,
    columnBoxShadowColor,
    inputRefs,
  ]);

  return (
    <div className="relative">
      <div className="column-options-wrapper flex flex-col w-full p-4">
        {/* col tabs */}
        <div className="w-full flex flex-row flex-nowrap border-b mb-4">
          <button
            onClick={() => handleTabClick(1)}
            className={`px-4 py-2 border-t border-l ${
              activeTab === 1 ? "active bg-green-500 text-white" : ""
            }`}>
            Tartalom elrendezése
          </button>
          <button
            onClick={() => handleTabClick(2)}
            className={`px-4 py-2 border-t border-l ${
              activeTab === 2 ? "active bg-green-500 text-white" : ""
            }`}>
            Oszlop beállítások
          </button>
          <button
            onClick={() => handleTabClick(3)}
            className={`px-4 py-2 border-t border-l ${
              activeTab === 3 ? "active bg-green-500 text-white" : ""
            }`}>
            Oszlop design
          </button>
          <button
            onClick={() => handleTabClick(4)}
            className={`px-4 py-2 border-t border-l border-r ${
              activeTab === 4 ? "active bg-green-500 text-white" : ""
            }`}>
            Kép beállítások
          </button>
        </div>
        {currentColumn && (
          <>
            {/* col content order */}
            <div className={`w-full ${activeTab !== 1 ? "hidden" : ""}`}>
              {/* layout mode */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnOrder"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">Tartalom elrendezés: </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki, hogy flex vagy grid legyen a tartalmi
                    elrendezés. (alapértelmezett: flex)
                  </span>
                </label>
                <Select
                  options={initialData?.columnLayoutMode}
                  className="w-3/4"
                  menuPosition="fixed"
                  name="columnLayoutMode"
                  placeholder="Válasszon tartalom elrendezés típust"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={
                    columnOptions?.columnLayoutMode
                      ? initialData?.columnLayoutMode?.filter(
                          (x) => x.value === columnOptions?.columnLayoutMode
                        )
                      : initialData?.columnLayoutMode[0]
                  }
                  onChange={(val, e) => {
                    setColumnLayoutMode(val.value);
                  }}
                />
              </div>

              {/* ✅ NEW: Grid layout options */}
              <div className={`${columnLayoutMode === "grid" ? "" : "hidden"}`}>
                <InputField
                  label="Oszlop kezdete (pl. 1)"
                  description={
                    "Defines where the column starts, counting from left to right. auto = Let the browser decide (default) 1 = Start in the first column of the grid 5 = Start in the 5th column"
                  }
                  id="gridColumnStart"
                  defaultValue={columnOptions.gridColumnStart ?? "auto"}
                  inputRefs={(r) => (inputRefs.current[index++] = r)}
                  index={index}
                />
                <InputField
                  label="Oszlop szélesség (1–12)"
                  id="colSpan"
                  type="number"
                  description={
                    "This is a user-friendly alternative to writing gridColumnEnd: span X. If user sets colSpan = 6, then in code you convert that to:"
                  }
                  defaultValue={columnOptions.colSpan ?? 6}
                  inputRefs={(r) => (inputRefs.current[index++] = r)}
                  index={index}
                />
                <InputField
                  label="Sor kezdete (opcionális)"
                  id="gridRowStart"
                  description={`string ("auto", "2", "span 2") These control vertical positioning (rows)."auto" = Let browser handle it "2" = Start or end in row 2 "span 2" = Span 2 rows`}
                  defaultValue={columnOptions.gridRowStart ?? "auto"}
                  inputRefs={(r) => (inputRefs.current[index++] = r)}
                  index={index}
                />
                <InputField
                  label="Sor vége (opcionális)"
                  id="gridRowEnd"
                  description={`string ("auto", "2", "span 2") These control vertical positioning (rows)."auto" = Let browser handle it "2" = Start or end in row 2 "span 2" = Span 2 rows`}
                  defaultValue={columnOptions.gridRowEnd ?? "auto"}
                  inputRefs={(r) => (inputRefs.current[index++] = r)}
                  index={index}
                />
                <InputField
                  label="colSpan (alternatíva, pl. 6)"
                  id="colSpan"
                  defaultValue={columnOptions.colSpan ?? 6}
                  inputRefs={(r) => (inputRefs.current[index++] = r)}
                  index={index}
                />
                <InputField
                  label="rowSpan (opcionális)"
                  id="rowSpan"
                  defaultValue={columnOptions.rowSpan ?? 1}
                  inputRefs={(r) => (inputRefs.current[index++] = r)}
                  index={index}
                />
                <div cols={1} className="gap-2">
                  <InputField
                    label="colSpan (xs)"
                    id="colSpanXs"
                    defaultValue={columnOptions.colSpanXs ?? 12}
                    inputRefs={(r) => (inputRefs.current[index++] = r)}
                    index={index}
                  />
                  <InputField
                    label="colSpan (sm)"
                    id="colSpanSm"
                    defaultValue={columnOptions.colSpanSm ?? 6}
                    inputRefs={(r) => (inputRefs.current[index++] = r)}
                    index={index}
                  />
                  <InputField
                    label="colSpan (md)"
                    id="colSpanMd"
                    defaultValue={columnOptions.colSpanMd ?? 4}
                    inputRefs={(r) => (inputRefs.current[index++] = r)}
                    index={index}
                  />
                  <InputField
                    label="colSpan (lg)"
                    id="colSpanLg"
                    defaultValue={columnOptions.colSpanLg ?? 3}
                    inputRefs={(r) => (inputRefs.current[index++] = r)}
                    index={index}
                  />
                </div>
              </div>
              {/* <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label htmlFor="gridCols" className="w-1/4 text-gray-600">
                  <p className="font-bold">Grid oszlopok száma:</p>
                  <span className="text-sm">
                    Alapértelmezett: 12 (csak Grid esetén!)
                  </span>
                </label>
                <input
                  type="number"
                  name="gridCols"
                  id="gridCols"
                  min="1"
                  max="12"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={columnOptions?.gridCols ?? 12}
                  className="w-3/4 border p-2"
                />
              </div>
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label htmlFor="colSpan" className="w-1/4 text-gray-600">
                  <p className="font-bold">Osztály szélessége (span):</p>
                  <span className="text-sm">
                    Alapértelmezett: 1 (csak Grid esetén!)
                  </span>
                </label>
                <input
                  type="number"
                  name="colSpan"
                  id="colSpan"
                  min="1"
                  max="12"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={columnOptions?.colSpan ?? 1}
                  className="w-3/4 border p-2"
                />
              </div>
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label htmlFor="gridGap" className="w-1/4 text-gray-600">
                  <p className="font-bold">Rács közök (gap):</p>
                  <span className="text-sm">
                    Alapértelmezett: 10px (csak Grid esetén!)
                  </span>
                </label>
                <input
                  type="text"
                  name="gridGap"
                  id="gridGap"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={columnOptions?.gridGap ?? "10px"}
                  className="w-3/4 border p-2"
                />
              </div>

              <div className="flex flex-col w-full mb-4 pb-4 border-b">
                <p className="text-md font-bold mb-2">
                  Grid span beállítások (breakpointra):
                </p>
                {["sm", "md", "lg", "xl"].map((bp) => (
                  <div key={bp} className="flex flex-row w-full mb-1">
                    <label className="w-1/4 text-right capitalize text-gray-600 pr-4">
                      {bp} oszlopszám / sormagasság
                    </label>
                    <div className="w-3/4 flex gap-2">
                      <input
                        type="number"
                        min={1}
                        max={12}
                        defaultValue={
                          columnOptions?.gridSpans?.col?.[bp] ??
                          initialData?.gridSpans?.col?.[bp] ??
                          ""
                        }
                        name={`gridSpan-col-${bp}`}
                        ref={(r) => (inputRefs.current[index++] = r)}
                        className="w-1/2 border p-2"
                        placeholder={`col-span ${bp}`}
                      />
                      <input
                        type="number"
                        min={1}
                        max={6}
                        defaultValue={
                          columnOptions?.gridSpans?.row?.[bp] ??
                          initialData?.gridSpans?.row?.[bp] ??
                          ""
                        }
                        name={`gridSpan-row-${bp}`}
                        ref={(r) => (inputRefs.current[index++] = r)}
                        className="w-1/2 border p-2"
                        placeholder={`row-span ${bp}`}
                      />
                    </div>
                  </div>
                ))}
              </div>

              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label className="w-1/4 text-gray-600 font-bold pr-4">
                  Manuális pozíció
                </label>
                <div className="w-3/4 grid grid-cols-2 gap-2">
                  <input
                    type="number"
                    name="colStart"
                    defaultValue={columnOptions?.colStart ?? ""}
                    placeholder="Column Start"
                    className="border p-2"
                    ref={(r) => (inputRefs.current[index++] = r)}
                  />
                  <input
                    type="number"
                    name="colEnd"
                    defaultValue={columnOptions?.colEnd ?? ""}
                    placeholder="Column End"
                    className="border p-2"
                    ref={(r) => (inputRefs.current[index++] = r)}
                  />
                  <input
                    type="number"
                    name="rowStart"
                    defaultValue={columnOptions?.rowStart ?? ""}
                    placeholder="Row Start"
                    className="border p-2"
                    ref={(r) => (inputRefs.current[index++] = r)}
                  />
                  <input
                    type="number"
                    name="rowEnd"
                    defaultValue={columnOptions?.rowEnd ?? ""}
                    placeholder="Row End"
                    className="border p-2"
                    ref={(r) => (inputRefs.current[index++] = r)}
                  />
                </div>
              </div>

              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label className="w-1/4 text-gray-600 font-bold pr-4">
                  Grid sablon oszlopok
                </label>
                <textarea
                  name="gridTemplateColumns"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={columnOptions?.gridTemplateColumns ?? ""}
                  className="w-3/4 border p-2"
                  placeholder="pl.: repeat(3, 1fr)"
                />
              </div>
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label className="w-1/4 text-gray-600 font-bold pr-4">
                  Grid sablon sorok
                </label>
                <textarea
                  name="gridTemplateRows"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={columnOptions?.gridTemplateRows ?? ""}
                  className="w-3/4 border p-2"
                  placeholder="pl.: auto 1fr auto"
                />
              </div> */}

              {/* order column content */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnOrder"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Oszlop tartalom elrendezés:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki, hogy sor vagy oszlop legyen a tartalmi
                    elrendezés. (alapértelmezett: Oszlop)
                  </span>
                </label>
                <Select
                  options={initialData?.contentOrder}
                  className="w-3/4"
                  menuPosition={"fixed"}
                  placeholder={
                    "Válassza ki, hogy sor vagy oszlop legyen a tartalmi elrendezés."
                  }
                  backspaceRemovesValue={false}
                  name={"columnOrder"}
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={
                    columnOptions?.columnOrder
                      ? initialData?.contentOrder?.filter(
                          (x) => x.value === columnOptions?.columnOrder
                        )
                      : initialData?.contentOrder[1]
                  }
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    // dispatch(selectParentPage({ parent: val }));
                  }}
                  required
                />
              </div>
              {/* order column content on breakpoints */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnOrderBreakpoints"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Oszlop tartalom elrendezés töréspontoknál:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki, hogy sor vagy oszlop legyen a tartalmi
                    elrendezés a megadott töréspontoknál. (sm: max. 640px, md:
                    max. 768px, lg: max. 1024px, xl: 1280px, 2xl: 1536px)
                  </span>
                </label>
                <Select
                  options={initialData?.orderBreakpoints}
                  className="w-3/4"
                  menuPosition={"fixed"}
                  placeholder={
                    "Válassza ki, hogy sor vagy oszlop legyen a tartalmi elrendezés a megadott töréspontoknál."
                  }
                  isMulti
                  backspaceRemovesValue={false}
                  name={"columnOrderBreakpoints"}
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={
                    columnOptions?.columnOrderBreakpoints
                      ? initialData?.orderBreakpoints?.filter((x) =>
                          state?.containerData?.[parent]?.content?.[
                            innerid
                          ].options?.columnOrderBreakpoints?.includes(x.value)
                        )
                      : ""
                  }
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    // dispatch(selectParentPage({ parent: val }));
                  }}
                  required
                />
              </div>
              {/* column position */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnPosition"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">Tartalom elhelyezkedése: </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki, hogy az oszlopon belül milyen pozícióban
                    helyezkedjenek el az elemek - (alapértelmezett érték:
                    Relatív).
                  </span>
                </label>

                <Select
                  options={initialData.position}
                  className="w-3/4"
                  placeholder={"Tartalom elhelyezkedése"}
                  backspaceRemovesValue={false}
                  name={"columnPosition"}
                  menuPosition={"fixed"}
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={
                    columnOptions
                      ? initialData.position.filter(
                          (x) => x.value === columnOptions.columnPosition
                        )
                      : initialData.position[0]
                  }
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    // dispatch(selectParentPage({ parent: val }));
                  }}
                  required
                />
              </div>
              {/* column position X axis */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnJustify"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Tartalom elhelyezkedése X tengelyen:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki, hogy az oszlopon belül milyen pozícióban
                    helyezkedjenek el az elemek - (alapértelmezett érték:
                    Center).
                  </span>
                </label>

                <Select
                  options={initialData.justify}
                  className="w-3/4"
                  placeholder={"Tartalom elhelyezkedése X tengelyen"}
                  backspaceRemovesValue={false}
                  name={"columnJustify"}
                  menuPosition={"fixed"}
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={
                    columnOptions?.columnJustify
                      ? initialData.justify.filter(
                          (x) => x.value === columnOptions.columnJustify
                        )
                      : initialData.justify[0]
                  }
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    // dispatch(selectParentPage({ parent: val }));
                  }}
                  required
                />
              </div>
              {/* column position Y axis */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnAlign"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Tartalom elhelyezkedése Y tengelyen:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki, hogy az oszlopon belül milyen pozícióban
                    helyezkedjenek el az elemek - (alapértelmezett érték:
                    Center).
                  </span>
                </label>

                <Select
                  options={initialData.align}
                  className="w-3/4"
                  placeholder={"Tartalom elhelyezkedése Y tengelyen"}
                  backspaceRemovesValue={false}
                  name={"columnAlign"}
                  menuPosition={"fixed"}
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={
                    columnOptions?.columnAlign
                      ? initialData.align.filter(
                          (x) => x.value === columnOptions.columnAlign
                        )
                      : initialData.align[1]
                  }
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    // dispatch(selectParentPage({ parent: val }));
                  }}
                  required
                />
              </div>
            </div>
            {/* col settings */}
            <div className={`w-full ${activeTab !== 2 ? "hidden" : ""}`}>
              {/* column overflow */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnOverflow"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Oszlop tartalmának túlcsordulása:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki, hogy milyen formában jelenlen meg a oszlopben
                    található túlcsorduló tartalom. (alapértelmezett: Elrejtve)
                  </span>
                </label>

                <Select
                  options={initialData?.overflow}
                  className="w-3/4"
                  menuPosition={"fixed"}
                  placeholder={"Válasszon oszlop távolságot"}
                  backspaceRemovesValue={false}
                  name={"columnOverflow"}
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={
                    columnOptions
                      ? initialData?.overflow?.filter(
                          (x) => x.value === columnOptions?.columnOverflow
                        )
                      : initialData.overflow[0]
                  }
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    // dispatch(selectParentPage({ parent: val }));
                  }}
                  required
                />
              </div>
              {/* column class*/}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnClass"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">Oszlop css class: </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Megadhatja, hogy a oszlop elemnek milyen class-a legyen.
                  </span>
                </label>
                <div className="w-full flex flex-row flex-nowrap justify-evenly items-center">
                  <div className="flex flex-row flex-nowrap w-3/4 justify-evenly items-center max-h-10">
                    <input
                      type="text"
                      name="columnClass"
                      id="columnClass"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-full border h-max p-2"
                      defaultValue={columnOptions?.columnClass ?? ""}
                    />
                  </div>
                </div>
              </div>
              {/* column height*/}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnHeight"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">Oszlop magasság: </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Megadhatja, hogy a oszlop elemnek mekkora magassága legyen.
                    Alapértelmezett: inherit. Lehetséges értékek: auto,
                    fit-content, inherit, initial, max-content, min-content,
                    revert, unset, px, vh, rem, em
                  </span>
                </label>

                <div className="flex flex-row flex-nowrap justify-evenly items-center">
                  <div className="flex flex-row flex-nowrap w-3/4 justify-evenly items-center max-h-10">
                    <span className="fill-gray-600 w-[2rem] h-[2rem]">
                      {icons.up}
                      {icons.down}
                    </span>
                    <input
                      type="text"
                      name="columnHeight"
                      id="columnHeight"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={columnOptions?.columnHeight ?? "inherit"}
                    />
                  </div>
                </div>
              </div>
              {/* column max height*/}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnMaxHeight"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">Oszlop maximum magasság: </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Megadhatja, hogy a oszlop elemnek mekkora maximum magassága
                    legyen. Alapértelmezett: inherit. Lehetséges értékek: auto,
                    fit-content, inherit, initial, max-content, min-content,
                    revert, unset, px, vh, rem, em
                  </span>
                </label>

                <div className="flex flex-row flex-nowrap justify-evenly items-center">
                  <div className="flex flex-row flex-nowrap w-3/4 justify-evenly items-center max-h-10">
                    <span className="fill-gray-600 w-[2rem] h-[2rem]">
                      {icons.up}
                      {icons.down}
                    </span>
                    <input
                      type="text"
                      name="columnMaxHeight"
                      id="columnMaxHeight"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={columnOptions?.columnMaxHeight ?? "inherit"}
                    />
                  </div>
                </div>
              </div>
              {/* column padding */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnPadding"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">Oszlop padding: </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Megadhatja, hogy a oszlop elemnek mekkora paddingja legyen
                    px-ben. Alapértelmezett: 0 30 0 30
                  </span>
                </label>

                <div className="flex flex-row flex-nowrap justify-evenly items-center">
                  <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                    <span className="fill-gray-600 w-[2rem] h-[2rem]">
                      {icons.up}
                    </span>
                    <input
                      type="text"
                      name="columnPaddingTop"
                      id="columnPaddingTop"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={columnOptions?.columnPaddingTop ?? 0}
                    />
                  </div>
                  <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                    <span className="fill-gray-600 w-[2rem] h-[2rem]">
                      {icons.right}
                    </span>
                    <input
                      type="text"
                      name="columnPaddingRight"
                      id="columnPaddingRight"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={columnOptions?.columnPaddingRight ?? 30}
                    />
                  </div>
                  <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                    <span className="fill-gray-600 w-[2rem] h-[2rem]">
                      {icons.down}
                    </span>
                    <input
                      type="text"
                      name="columnPaddingBottom"
                      id="columnPaddingBottom"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={columnOptions?.columnPaddingBottom ?? 0}
                    />
                  </div>
                  <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                    <span className="fill-gray-600 w-[2rem] h-[2rem]">
                      {icons.left}
                    </span>
                    <input
                      type="text"
                      name="columnPaddingLeft"
                      id="columnPaddingLeft"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={columnOptions?.columnPaddingLeft ?? 30}
                    />
                  </div>
                </div>
              </div>
              {/* column col distance */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnColDistance"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Oszlopben található oszlopok:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki, hogy mekkora távolság legyen px-ben a oszlopben
                    található oszlopok között. (alapértelmezett: 10px)
                  </span>
                </label>

                <Select
                  options={initialData.editor.column.columnColDistance}
                  className="w-3/4"
                  menuPosition={"fixed"}
                  placeholder={"Válasszon oszlop távolságot"}
                  backspaceRemovesValue={false}
                  name={"columnColDistance"}
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={
                    columnOptions?.columnColDistance
                      ? initialData?.editor?.column?.columnColDistance?.filter(
                          (x) =>
                            parseInt(x.value) ===
                            parseInt(columnOptions?.columnColDistance)
                        )
                      : initialData?.editor?.column?.columnColDistance[2]
                  }
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    // dispatch(selectParentPage({ parent: val }));
                  }}
                  required
                />
              </div>
              {/* column margin */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnMargin"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">Oszlop margin: </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Megadhatja, hogy a oszlop elemnek mekkora marginja legyen
                    px-ben. Alapértelmezett: 0
                  </span>
                </label>

                <div className="flex flex-row flex-nowrap justify-evenly items-center">
                  <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                    <span className="fill-gray-600 w-[2rem] h-[2rem]">
                      {icons.up}
                    </span>
                    <input
                      type="text"
                      name="columnMarginTop"
                      id="columnMarginTop"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={columnOptions?.columnMarginTop ?? 0}
                    />
                  </div>
                  <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                    <span className="fill-gray-600 w-[2rem] h-[2rem]">
                      {icons.right}
                    </span>
                    <input
                      type="text"
                      name="columnMarginRight"
                      id="columnMarginRight"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={columnOptions?.columnMarginRight ?? 0}
                    />
                  </div>
                  <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                    <span className="fill-gray-600 w-[2rem] h-[2rem]">
                      {icons.down}
                    </span>
                    <input
                      type="text"
                      name="columnMarginBottom"
                      id="columnMarginBottom"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={columnOptions?.columnMarginBottom ?? 0}
                    />
                  </div>
                  <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                    <span className="fill-gray-600 w-[2rem] h-[2rem]">
                      {icons.left}
                    </span>
                    <input
                      type="text"
                      name="columnMarginLeft"
                      id="columnMarginLeft"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={columnOptions?.columnMarginLeft ?? 0}
                    />
                  </div>
                </div>
              </div>
              {/* column visibility on mobile */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"visibleOnMobile"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Oszlop láthatósága mobil eszközön:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki, hogy megjelenik-e a oszlop mobil eszközön.
                    (alapértelmezett: Látható)
                  </span>
                </label>
                <Select
                  options={initialData?.visibleOnMobile}
                  className="w-3/4"
                  menuPosition={"fixed"}
                  placeholder={
                    "Válassza ki, hogy megjelenik-e a oszlop mobil eszközön"
                  }
                  backspaceRemovesValue={false}
                  name={"visibleOnMobile"}
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={
                    columnOptions?.visibleOnMobile
                      ? initialData?.visibleOnMobile?.filter(
                          (x) => x.value === columnOptions?.visibleOnMobile
                        )
                      : initialData?.visibleOnMobile[1]
                  }
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    // dispatch(selectParentPage({ parent: val }));
                  }}
                  required
                />
              </div>
            </div>
            {/* col design settings */}
            <div className={`w-full ${activeTab !== 3 ? "hidden" : ""}`}>
              {/* column animation type */}
              <div className="flex flex-row w-full mb-4 p-4 border-b">
                <label
                  htmlFor={"animationTypes"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">Oszlop animációja: </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki az oszlop animációját. (alapértelmezett: Nincs
                    animáció)
                  </span>
                </label>

                <Select
                  options={initialData.animation}
                  className="w-3/4"
                  menuPosition={"fixed"}
                  placeholder={"Válasszon animáció típust"}
                  backspaceRemovesValue={false}
                  name={"animationTypes"}
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={
                    columnOptions?.animationTypes
                      ? initialData.animation.filter(
                          (x) => x.value === columnOptions.animationTypes
                        )
                      : initialData.animation[0]
                  }
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    // dispatch(selectParentPage({ parent: val }));
                  }}
                  required
                />
              </div>
              {/* column animation delay */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"animationDelay"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Oszlop animáció késleltetése:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Megadhatja, hogy a oszlop elemnek mennyi legyen az animáció
                    késleltetése. Alapértelmezett: 1s
                  </span>
                </label>
                <div className="flex flex-row flex-nowrap justify-evenly items-center">
                  <input
                    type="text"
                    name="animationDelay"
                    id="animationDelay"
                    ref={(r) => (inputRefs.current[index++] = r)}
                    className="w-3/4 border h-max p-2"
                    defaultValue={columnOptions?.animationDelay ?? 1}
                  />
                </div>
              </div>
              {/* column animation duration */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"animationDuration"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Oszlop animáció időtartama:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Megadhatja, hogy a oszlop elemnek mennyi legyen az animáció
                    időtartama. (pl.: 0.5, 2, 4) Alapértelmezett: 3s
                  </span>
                </label>
                <div className="flex flex-row flex-nowrap justify-evenly items-center">
                  <input
                    type="text"
                    name="animationDuration"
                    id="animationDuration"
                    ref={(r) => (inputRefs.current[index++] = r)}
                    className="w-3/4 border h-max p-2"
                    defaultValue={columnOptions?.animationDuration ?? 3}
                  />
                </div>
              </div>
              {/* column border */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnBorder"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">Oszlop kerete: </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Megadhatja, hogy a oszlop elemnek milyen vastag kerete
                    legyen px-ben. Alapértelmezett: 0
                  </span>
                </label>
                <div className="flex flex-row flex-nowrap justify-evenly items-center">
                  <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                    <span className="fill-gray-600 w-[2rem] h-[2rem]">
                      {icons.up}
                    </span>
                    <input
                      type="text"
                      name="columnBorderTop"
                      id="columnBorderTop"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={columnOptions?.columnBorderTop ?? 0}
                    />
                  </div>
                  <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                    <span className="fill-gray-600  w-[2rem] h-[2rem]">
                      {icons.right}
                    </span>
                    <input
                      type="text"
                      name="columnBorderRight"
                      id="columnBorderRight"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={columnOptions?.columnBorderRight ?? 0}
                    />
                  </div>
                  <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                    <span className="fill-gray-600 w-[2rem] h-[2rem]">
                      {icons.down}
                    </span>
                    <input
                      type="text"
                      name="columnBorderBottom"
                      id="columnBorderBottom"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={columnOptions?.columnBorderBottom ?? 0}
                    />
                  </div>
                  <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                    <span className="fill-gray-600 w-[2rem] h-[2rem]">
                      {icons.left}
                    </span>
                    <input
                      type="text"
                      name="columnBorderLeft"
                      id="columnBorderLeft"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={columnOptions?.columnBorderLeft ?? 0}
                    />
                  </div>
                </div>
              </div>
              {/* column border style */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"borderStyle"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">Keret stílusa: </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki, hogy milyen stílusa legyen a keretnek.
                    Alapértelmezett: solid
                  </span>
                </label>
                <Select
                  options={initialData.borderStyle}
                  className="w-3/4"
                  menuPosition={"fixed"}
                  placeholder={"Elválasztó stílusa"}
                  backspaceRemovesValue={false}
                  name={"borderStyle"}
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={
                    columnOptions?.borderStyle
                      ? initialData.borderStyle.filter(
                          (x) => x.value === columnOptions?.borderStyle
                        )
                      : initialData.borderStyle[2]
                  }
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                  }}
                  required
                />
              </div>
              {/* column border color */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnBorderColor"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Oszlop keret színének beállítása:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Megadhatja, hogy milyen színű legyen a oszlop kerete.
                    Alapértelmezett: #ffffff
                  </span>
                </label>

                <ColorPicker
                  value={columnBorderColor}
                  onChange={setColumnBorderColor}
                />
              </div>
              {/* column background color */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnBackgroundColor"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Oszlop háttér színének beállítása:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Megadhatja, hogy milyen színű legyen a oszlop háttere.
                    Alapértelmezett: rgba(255, 255, 255, 1)
                  </span>
                </label>

                <ColorPicker
                  value={columnBackgroundColor}
                  onChange={setColumnBackgroundColor}
                />
              </div>
              {/* box shadow */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnBoxShadow"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">Oszlop árnyék: </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki, hogy legyen-e a oszlopnek árnyéka.
                    (alapértelmezett: Nem)
                  </span>
                </label>

                <Select
                  options={initialData.bool}
                  className="w-3/4"
                  menuPosition={"fixed"}
                  placeholder={"Válasszon oszlop árnyék ki/be kapcsolást"}
                  backspaceRemovesValue={false}
                  name={"columnBoxShadow"}
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={
                    columnOptions
                      ? initialData.bool.filter(
                          (x) => x.value === columnOptions.columnBoxShadow
                        )
                      : initialData.bool[1]
                  }
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    // dispatch(selectParentPage({ parent: val }));
                  }}
                  required
                />
              </div>
              {/* column box shadow position */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnBoxShadowPosition"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Oszlop árnyékolás pozíciója:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Pozitív érték esetén az árnyék a oszlop alatt és jobbra,
                    negatív érték esetén fölötte és balra jelenik meg (pl.: 5px)
                  </span>
                </label>

                <div className="flex flex-row flex-nowrap justify-evenly items-center">
                  <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                    <span className="flex flex-col justify-center items-center fill-gray-600 w-[2rem] h-[2rem]">
                      {icons.up}
                      {icons.down}
                    </span>
                    <input
                      type="text"
                      name="columnBoxShadowVertical"
                      id="columnBoxShadowVertical"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={columnOptions?.columnBoxShadowVertical ?? 0}
                    />
                  </div>
                  <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                    <span className="flex flex-col justify-center items-center fill-gray-600 w-[2rem] h-[2rem]">
                      {icons.left}
                      {icons.right}
                    </span>
                    <input
                      type="text"
                      name="columnBoxShadowHorizontal"
                      id="columnBoxShadowHorizontal"
                      ref={(r) => (inputRefs.current[index++] = r)}
                      className="w-3/4 border h-max p-2"
                      defaultValue={
                        columnOptions?.columnBoxShadowHorizontal ?? 0
                      }
                    />
                  </div>
                </div>
              </div>
              {/* column box shadow blur radius */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnBoxShadowBlurRadius"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Oszlop árnyék elmosás mérték - blur radius:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Az árnyékolás elmosásának értéke - px-ben. 0-100 közötti
                    értéket adhat meg. Alapértelmezett: 0
                  </span>
                </label>

                <input
                  type="text"
                  name="columnBoxShadowBlurRadius"
                  id="columnBoxShadowBlurRadius"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  onChange={(e) => handleInputMinMax(e)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={columnOptions?.columnBoxShadowBlurRadius ?? 0}
                />
              </div>
              {/* column box shadow spread radius */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnBoxShadowSpreadRadius"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Oszlop árnyék szórási értéke - spread radius:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Az árnyékolás szórási értéke - px-ben. -100 és +100 közötti
                    értéket adhat meg. Alapértelmezett: 0
                  </span>
                </label>

                <input
                  type="text"
                  name="columnBoxShadowSpreadRadius"
                  id="columnBoxShadowSpreadRadius"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  onChange={(e) => handleInputMinMax(e)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={columnOptions?.columnBoxShadowSpreadRadius ?? 0}
                />
              </div>
              {/* column bos shadow color */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnBoxShadowColor"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Oszlop árnyék színének beállítása:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Megadhatja, hogy milyen színű legyen a oszlop által vetett
                    árnyék. Alapértelmezett: #ffffff
                  </span>
                </label>

                <ColorPicker
                  value={columnBoxShadowColor}
                  onChange={setColumnBoxShadowColor}
                />
              </div>
              {/* column link color */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"columnLinkColor"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Oszlopben levő linkek színének beállítása:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Megadhatja, hogy milyen színű legyen a oszlopben található
                    linkek. Alapértelmezett: #212121
                  </span>
                </label>

                <ColorPicker
                  value={columnLinkColor}
                  onChange={setColumnLinkColor}
                />
              </div>
            </div>
            <div className={`w-full ${activeTab !== 4 ? "hidden" : ""}`}>
              {/* select image */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"imageSelect"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">Kép kiválasztása: </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Töltsön fel képet, vagy válasszon a médiatárból!
                  </span>
                </label>
                <div className="flex flex-row w-3/4 justify-end items-center">
                  {generateSelectedImage()}
                  <button
                    onClick={() => showSelectImageModal(true)}
                    className=" border border-gray-300 px-4 py-2 text-gray-500 hover:border-gray-600 hover:text-gray-900"
                    btntext="">
                    Kép kiválasztása
                  </button>
                </div>
              </div>
              {/* image position */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"imageBackgroundPosition"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">Kép pozíciója: </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki a háttér megjelenítése pozícióját. Csak abban az
                    esetben érvényesül, ha a kép típusa: Háttérkép
                    (alapértelmezett: Inherit - Örökölt)
                  </span>
                </label>

                <Select
                  options={initialData.editor.image.imageBackgroundPosition}
                  className="w-3/4"
                  menuPosition={"fixed"}
                  placeholder={"Válasszon háttér pozíciót"}
                  backspaceRemovesValue={false}
                  name={"imageBackgroundPosition"}
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={
                    columnOptions?.imageBackgroundPosition
                      ? initialData?.editor?.image?.imageBackgroundPosition?.filter(
                          (x) =>
                            x.value === columnOptions?.imageBackgroundPosition
                        )
                      : initialData?.editor?.image?.imageBackgroundPosition[0]
                  }
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    // dispatch(selectParentPage({ parent: val }));
                  }}
                  required
                />
              </div>
              {/* image object fit */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"imageObjectFit"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Kép objektum illeszkedése:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki, hogy a képet tartalmazó dobozban hogyan
                    viselkedjen a kép. Alapértelmezett: Cover (Kitölti a teljes
                    doboz méretet)
                  </span>
                </label>
                <Select
                  options={initialData.editor.image.imageObjectFit}
                  className="w-3/4"
                  menuPosition={"fixed"}
                  placeholder={"Válasszon háttér pozíciót"}
                  backspaceRemovesValue={false}
                  name={"imageObjectFit"}
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={
                    columnOptions?.imageObjectFit
                      ? initialData?.editor?.image?.imageObjectFit.filter(
                          (x) => x.value === columnOptions?.imageObjectFit
                        )
                      : initialData?.editor?.image?.imageObjectFit[1]
                  }
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    // dispatch(selectParentPage({ parent: val }));
                  }}
                  required
                />
              </div>
              {/* image object position */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"imageObjectPosition"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">Kép objektum pozíciója: </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki, hogy a kép melyik részére fókuszáljon a
                    dobozban. Alapértelmezett: Center (A kép közepe kerül
                    fókuszba)
                  </span>
                </label>
                <Select
                  options={initialData?.editor?.image?.imageObjectPosition}
                  className="w-3/4"
                  menuPosition={"fixed"}
                  placeholder={"Válasszon háttér pozíciót"}
                  backspaceRemovesValue={false}
                  name={"imageObjectPosition"}
                  ref={(r) => (inputRefs.current[index++] = r)}
                  isMulti
                  defaultValue={
                    columnOptions?.imageObjectPosition
                      ? initialData?.editor?.image?.imageObjectPosition?.filter(
                          (x) =>
                            state?.containerData?.[parent]?.content?.[
                              innerid
                            ].options?.imageObjectPosition?.includes(x.value)
                        )
                      : initialData?.editor?.image?.imageObjectPosition[1]
                  }
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    // dispatch(selectParentPage({ parent: val }));
                  }}
                  required
                />
              </div>
              {/* image dimensions */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"altText"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">
                    Kép dimenziójának megadása:{" "}
                  </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Adjon meg aránypárt, amely alapján a kép generálódik. Az itt
                    megadott érték (*100) lesz a kép pixelben megadott
                    szélessége és magassága (pl.: 16:9 - 1600px:900px méretű kép
                    generálódik). Alapértelmezett: 4:3
                  </span>
                </label>
                <div className="w-3/4 flex flex-row flex-nowrap justify-between">
                  <input
                    type="number"
                    name="imgWidth"
                    id="imgWidth"
                    ref={(r) => (inputRefs.current[index++] = r)}
                    className="w-1/2 border h-max p-2"
                    defaultValue={columnOptions?.imgWidth ?? 4}
                  />
                  <input
                    type="number"
                    name="imgHeight"
                    id="imgHeight"
                    ref={(r) => (inputRefs.current[index++] = r)}
                    className="w-1/2 border h-max p-2"
                    defaultValue={columnOptions?.imgHeight ?? 3}
                  />
                </div>
              </div>
              {/* image cut to size */}
              <div className="flex flex-row w-full mb-4 pb-4 border-b">
                <label
                  htmlFor={"imageCutToSize"}
                  className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
                  <p className="w-full font-bold">Kép vágása: </p>
                  <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                    Válassza ki, hogy a kép vágva legyen-e a megadott arányra.
                    Alapértelmezett: Igen
                  </span>
                </label>
                <Select
                  options={initialData?.bool}
                  className="w-3/4"
                  menuPosition={"fixed"}
                  placeholder={"Válasszon háttér pozíciót"}
                  backspaceRemovesValue={false}
                  name={"imageCutToSize"}
                  ref={(r) => (inputRefs.current[index++] = r)}
                  defaultValue={
                    columnOptions?.imageCutToSize
                      ? initialData?.bool?.filter(
                          (x) => x.value === columnOptions?.imageCutToSize
                        )
                      : initialData?.bool[0]
                  }
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    // dispatch(selectParentPage({ parent: val }));
                  }}
                  required
                />
              </div>
            </div>
          </>
        )}
      </div>
      <ModalButton btntext={"Mentés"} callback={submitColumnOptions} />

      <MediaGalleryModal
        isOpen={selectImageModal}
        onClose={() => {
          showSelectImageModal(false);
        }}
        action="container-image"
        onConfirmSelection={(selectedImage) => {
          setSelectedImage({ selectedImage: selectedImage });
          showSelectImageModal(false); // ✅ Close the modal manually
        }}
      />
    </div>
  );
};

export default ColumnSettings;
