import React, { useEffect, useState, useCallback, useRef } from "react";
import Select from "react-select";
import { initialData } from "@/data/data";
import Button from "@/components/shared/button/Button";
import { useSelector, useDispatch } from "react-redux";
import icons from "@/assets/icons/icons";
import Image from "next/image";
import MediaObject from "@/components/shared/mediaobject/MediaObject";
const InputField = ({
  label,
  id,
  description,
  defaultValue,
  inputRefs,
  index,
}) => (
  <div className="flex flex-row w-full mb-4 pb-4 border-b">
    <label htmlFor={id} className="w-1/4 text-gray-400 pr-2">
      <p className="font-bold">{label}</p>
      {description && <p className="text-sm text-gray-600">{description}</p>}
    </label>
    <input
      type="text"
      id={id}
      name={id}
      className="w-3/4 border p-2"
      defaultValue={defaultValue}
      ref={inputRefs}
    />
  </div>
);
const SelectField = ({
  label,
  id,
  options,
  selected,
  inputRefs,
  isMulti = false,
}) => (
  <div className="flex flex-row w-full mb-4 pb-4 border-b">
    <label htmlFor={id} className="w-1/4 text-gray-400 pr-2">
      <p className="font-bold">{label}</p>
    </label>
    <Select
      options={options}
      className="w-3/4"
      menuPosition={"fixed"}
      name={id}
      isMulti={isMulti}
      placeholder={`Válasszon ${label.toLowerCase()}...`}
      backspaceRemovesValue={false}
      defaultValue={options.find((x) => x.value === selected) || options[0]}
      ref={inputRefs}
    />
  </div>
);
import dynamic from "next/dynamic";
const ColorPicker = dynamic(() => import("react-best-gradient-color-picker"), {
  ssr: false,
});
import ApiCall from "@/api/api-calls";
// import MediaContentStore from "../MediaStore/MediaContentStore";
import useDynamicManager from "@/store/managers/useDynamicManager";
import MediaGalleryModal from "@/components/media-store/GalleryModal/MediaGalleryModal";

const ContainerSettings = ({
  toggleModal,
  modalName,
  type,
  lifecycleHandlers,
}) => {
  const [selectImageModal, showSelectImageModal] = useState(false);
  const [activeTab, setActiveTab] = useState(1);
  const { state, updateContainerOptions, setSelectedImage, resetImageState } =
    useDynamicManager(type);
  const containerOptions = state.selectedContainer.options ?? {};
  const [containerLayoutMode, setContainerLayoutMode] = useState(
    containerOptions?.layoutSelect
  );

  const [containerLinkColor, setContainerLinkColor] = useState(
    containerOptions?.containerLinkColor ?? "rgba((33,33,33, 1)"
  );
  const [containerWidth, setContainerWidth] = useState(
    containerOptions?.containerWidth ?? initialData.settings.layout[0]
  );
  const [containerLinkHoverColor, setContainerLinkHoverColor] = useState(
    containerOptions?.containerLinkHoverColor ?? "rgba(8,145,178, 1)"
  );
  const [containerBorderColor, setContainerBorderColor] = useState(
    containerOptions?.containerBorderColor ?? "rgba(8,145,178, 1)"
  );
  const [containerBackgroundColor, setContainerBackgroundColor] = useState(
    containerOptions?.containerBackgroundColor ?? "rgba(255, 255, 255, 0)"
  );
  const [containerBoxShadowColor, setContainerBoxShadowColor] = useState(
    containerOptions?.containerBoxShadowColor ?? "rgba(255, 255, 255,0)"
  );
  const inputRefs = useRef([]);
  let index = 0;

  const submitContainerOptions = () => {
    const content = {
      type: "containerOptions",
      containerLinkColor: containerLinkColor,
      containerLinkHoverColor: containerLinkHoverColor,
      containerBorderColor: containerBorderColor,
      containerBackgroundColor: containerBackgroundColor,
      containerBoxShadowColor: containerBoxShadowColor,
      backgroundImage: state.selectedImage,
    };
    const gridSpans = { col: {}, row: {} };
    inputRefs.current.forEach((input) => {
      if (input && input.name?.startsWith("gridSpan")) {
        const [, type, bp] = input.name.split("-"); // e.g., gridSpan-col-sm => [gridSpan, col, sm]
        const value = parseInt(input.value);
        if (!isNaN(value)) {
          gridSpans[type][bp] = value;
        }
      }
    });
    content.gridSpans = gridSpans;
    inputRefs.current.map((c, i) => {
      if (c) {
        if (c.props) {
          if (Array.isArray(c.props.value)) {
            content[c.props.name] = [];
            if (c?.props?.value?.length === 1) {
              content[c.props.name] = c.props.value[0].value;
            } else {
              c?.props?.value?.map((elem) => {
                content[c.props.name]?.push(elem.value);
              });
            }
          } else {
            content[c.props.name] = c.props.value.value;
          }
        } else {
          content[c.name] = c.value;
        }
      }
    });
    console.log("content", content);
    updateContainerOptions({
      options: content,
    });
    resetImageState();

    toggleModal(modalName);
  };

  // Create an array of combined options
  const combinedBreakpointOptions = [];
  initialData.editor.container.breakpoints.forEach((breakpoint) => {
    initialData.editor.container.contentOrder.forEach((contentOrder) => {
      combinedBreakpointOptions.push({
        value: ` ${breakpoint.value}-${contentOrder.value}`,
        label: `${breakpoint.label} - ${contentOrder.label}`,
      });
    });
  });
  const handleImageRemove = () => {
    resetImageState();
  };

  const handleInputMinMax = (e) => {
    if (parseInt(e.target.value) > parseInt(e.target.max)) {
      e.target.value = parseInt(e.target.max);
    }
    if (parseInt(e.target.value) < parseInt(e.target.min)) {
      e.target.value = parseInt(e.target.min);
    }
    e.target.value = parseInt(e.target.value);
    return e.target.value;
  };

  const handleTabClick = (tabNumber) => {
    setActiveTab(tabNumber);
  };

  const generateSelectedImage = useCallback(() => {
    const i = 0;

    if (state.selectedImage && state.selectedImage?._id) {
      return state.selectedImage?.props?.mimetype === "application/pdf" ? (
        <div
          className={`flex flex-col mr-2 mb-2 items-center border-1 w-full`}
          key={state.selectedImage._id}>
          <MediaObject
            data={state.selectedImage}
            index={i}
            data-index={i}
            className={"w-full"}
            id={state.selectedImage._id}
          />
          <p className="break-all p-1">
            {" "}
            {state.selectedImage?.props?.basename ??
              selectImageModal.originalname}
          </p>
          <input
            type="hidden"
            name="selectedFile"
            defaultValue={state.selectedImage}
          />

          <button onClick={handleImageRemove} className="text-red-500 my-4">
            Eltávolítás
          </button>
        </div>
      ) : (
        <div
          className={`flex flex-col mr-2 mb-2 items-center border-1 w-full}`}
          key={i}>
          <Image
            src={ApiCall.mediaStore.getResized(
              "/2/2/0",
              state.selectedImage?.originalname
            )}
            width={150}
            height={100}
            alt={state.selectedImage?.originalname ?? ""}
            effect="blur"
            loading="lazy"
          />
          <p className="break-all p-1">{state.selectedImage.props.filename}</p>
          <input
            type="hidden"
            name="selectedFile"
            defaultValue={state.selectedImage}
          />
          <button onClick={handleImageRemove}>Eltávolítás</button>
        </div>
      );
    }
  }, [state.selectedImage]);
  console.log("first", initialData);
  return (
    <div className="relative">
      <div className="container-options-wrapper flex flex-col w-full p-4">
        {/* container tabs */}
        <div className="w-full flex flex-row flex-nowrap border-b mb-4">
          <button
            onClick={() => handleTabClick(1)}
            className={`px-4 py-2 border-t border-l ${
              activeTab === 1 ? "active bg-green-500 text-white" : ""
            }`}>
            Tartalom elrendezése
          </button>
          <button
            onClick={() => handleTabClick(2)}
            className={`px-4 py-2 border-t border-l ${
              activeTab === 2 ? "active bg-green-500 text-white" : ""
            }`}>
            Konténer beállítások
          </button>
          <button
            onClick={() => handleTabClick(3)}
            className={`px-4 py-2 border-t border-l ${
              activeTab === 3 ? "active bg-green-500 text-white" : ""
            }`}>
            Konténer design
          </button>
          <button
            onClick={() => handleTabClick(4)}
            className={`px-4 py-2 border-t border-l border-r ${
              activeTab === 4 ? "active bg-green-500 text-white" : ""
            }`}>
            Kép beállítások
          </button>
        </div>
        {/* container content order */}
        <div className={`w-full ${activeTab !== 1 ? "hidden" : ""}`}>
          {/* container width */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerWidth"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Konténer tartalom szélessége: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy teljes szélességű vagy boxed legyen a konténer
                szélessége. (alapértelmezett: Boxed)
              </span>
            </label>
            <Select
              options={initialData.settings.layout}
              className="w-3/4"
              menuPosition={"fixed"}
              placeholder={
                "Válassza ki, hogy teljes szélességű vagy boxed legyen a konténer szélessége."
              }
              backspaceRemovesValue={false}
              name={"containerWidth"}
              ref={(r) => (inputRefs.current[index++] = r)}
              defaultValue={
                containerOptions?.containerWidth
                  ? initialData.settings.layout.filter(
                      (x) => x.value === containerOptions?.containerWidth
                    )
                  : initialData.settings.layout[1]
              }
              onChange={(val, e) => {
                // second parameter shows action triggered
                // dispatch(selectParentPage({ parent: val }));
              }}
              required
            />
          </div>
          {/* order container content */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"contentOrder"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Konténer tartalom elrendezés: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy sor vagy oszlop legyen a tartalmi elrendezés.
                (alapértelmezett: Sor)
              </span>
            </label>
            <Select
              options={initialData.contentOrder}
              className="w-3/4"
              menuPosition={"fixed"}
              placeholder={
                "Válassza ki, hogy sor vagy oszlop legyen a tartalmi elrendezés."
              }
              backspaceRemovesValue={false}
              name={"contentOrder"}
              ref={(r) => (inputRefs.current[index++] = r)}
              defaultValue={
                containerOptions?.contentOrder
                  ? initialData.contentOrder.find(
                      (x) => x.value === containerOptions?.contentOrder
                    )
                  : initialData.contentOrder[0]
              }
              onChange={(val, e) => {
                // second parameter shows action triggered
                // dispatch(selectParentPage({ parent: val }));
              }}
              required
            />
          </div>
          {/* order container content on breakpoints */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"contentOrderBreakpoints"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">
                Konténer tartalom elrendezés töréspontoknál:{" "}
              </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy sor vagy oszlop legyen a tartalmi elrendezés.
                (alapértelmezett: Sor)
              </span>
            </label>
            <Select
              options={initialData.orderBreakpoints}
              className="w-3/4"
              menuPosition={"fixed"}
              placeholder={
                "Válassza ki, hogy sor vagy oszlop legyen a tartalmi elrendezés."
              }
              isMulti
              backspaceRemovesValue={false}
              name={"contentOrderBreakpoints"}
              ref={(r) => (inputRefs.current[index++] = r)}
              defaultValue={
                containerOptions?.contentOrderBreakpoints
                  ? initialData.orderBreakpoints.filter((x) =>
                      containerOptions?.contentOrderBreakpoints?.includes(
                        x.value
                      )
                    )
                  : ""
              }
              onChange={(val, e) => {
                // second parameter shows action triggered
                // dispatch(selectParentPage({ parent: val }));
              }}
              required
            />
          </div>
          {/* Layout display type */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"layoutSelect"}
              className="w-sm-12 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Layout elrendezés típusa: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy milyen elrendezésben jelenjen(ek) meg a
                bejegyzés(ek) - (alapértelmezett érték: Flex)
              </span>
            </label>
            <Select
              options={initialData.containerLayoutMode}
              className="w-sm-36"
              menuPosition={"fixed"}
              placeholder={"Válasszon elrendezést"}
              backspaceRemovesValue={false}
              name={"layoutSelect"}
              ref={(r) => (inputRefs.current[index++] = r)}
              defaultValue={
                containerOptions?.layoutSelect
                  ? initialData.containerLayoutMode.find(
                      (x) => x.value === containerOptions?.layoutSelect
                    )
                  : initialData.containerLayoutMode[0]
              }
              onChange={(val, e) => {
                console.log("val", val);
                setContainerLayoutMode(val.value);
              }}
              required
            />
          </div>
          {
            <div
              className={`${containerLayoutMode === "grid" ? "" : "hidden"}`}>
              <SelectField
                label="Grid layout mód"
                id="gridLayoutMode"
                options={initialData.gridLayoutMode}
                inputRefs={(r) => (inputRefs.current[index++] = r)}
                selected={containerOptions?.gridLayoutMode}
                onChange={(val, e) => {}}
              />

              <InputField
                label="Grid oszlopok száma"
                id="gridCols"
                defaultValue={containerOptions.gridCols ?? 12}
                inputRefs={(r) => (inputRefs.current[index++] = r)}
                index={index}
              />
              <InputField
                label="Grid gap (pl. 10px vagy 1rem)"
                id="gridGap"
                defaultValue={containerOptions.gridGap ?? "10px"}
                inputRefs={(r) => (inputRefs.current[index++] = r)}
                index={index}
              />
              <InputField
                label="Grid sablon oszlopok (pl. repeat(3, 1fr))"
                id="gridTemplateColumns"
                defaultValue={
                  containerOptions.gridTemplateColumns ??
                  "repeat(auto-fit, minmax(250px, 1fr))"
                }
                inputRefs={(r) => (inputRefs.current[index++] = r)}
                index={index}
              />
              <InputField
                label="Grid oszlopok (xs)"
                id="gridColsXs"
                defaultValue={containerOptions.gridColsXs ?? "1"}
                inputRefs={(r) => (inputRefs.current[index++] = r)}
                index={index}
              />
              <InputField
                label="Grid oszlopok (sm)"
                id="gridColsSm"
                defaultValue={containerOptions.gridColsSm ?? "1"}
                inputRefs={(r) => (inputRefs.current[index++] = r)}
                index={index}
              />
              <InputField
                label="Grid oszlopok (md)"
                id="gridColsMd"
                defaultValue={containerOptions.gridColsMd ?? "1"}
                inputRefs={(r) => (inputRefs.current[index++] = r)}
                index={index}
              />
              <InputField
                label="Grid oszlopok (lg)"
                id="gridColsLg"
                defaultValue={containerOptions.gridColsLg ?? "1"}
                inputRefs={(r) => (inputRefs.current[index++] = r)}
                index={index}
              />
              <InputField
                label="Grid sablon sorok (pl. auto 1fr auto)"
                id="gridTemplateRows"
                defaultValue={containerOptions.gridTemplateRows ?? ""}
                inputRefs={(r) => (inputRefs.current[index++] = r)}
                index={index}
              />
            </div>
          }

          {/* container position X axis */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerJustify"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">
                Tartalom elhelyezkedése X tengelyen:{" "}
              </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy a konténeren belül milyen pozícióban
                helyezkedjenek el az elemek - (alapértelmezett érték: Center).
              </span>
            </label>

            <Select
              options={initialData.justify}
              className="w-3/4"
              placeholder={"Tartalom elhelyezkedése X tengelyen"}
              backspaceRemovesValue={false}
              name={"containerJustify"}
              menuPosition={"fixed"}
              ref={(r) => (inputRefs.current[index++] = r)}
              defaultValue={
                containerOptions?.containerJustify
                  ? initialData.justify.filter(
                      (x) => x.value === containerOptions.containerJustify
                    )
                  : initialData.justify[0]
              }
              onChange={(val, e) => {
                // second parameter shows action triggered
                // dispatch(selectParentPage({ parent: val }));
              }}
              required
            />
          </div>
          {/* column position Y axis */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerAlign"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">
                Tartalom elhelyezkedése Y tengelyen:{" "}
              </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy a konténeren belül milyen pozícióban
                helyezkedjenek el az elemek - (alapértelmezett érték: Center).
              </span>
            </label>

            <Select
              options={initialData.align}
              className="w-3/4"
              placeholder={"Tartalom elhelyezkedése Y tengelyen"}
              backspaceRemovesValue={false}
              name={"containerAlign"}
              menuPosition={"fixed"}
              ref={(r) => (inputRefs.current[index++] = r)}
              defaultValue={
                containerOptions?.containerAlign
                  ? initialData.align.filter(
                      (x) => x.value === containerOptions.containerAlign
                    )
                  : initialData.align[1]
              }
              onChange={(val, e) => {
                // second parameter shows action triggered
                // dispatch(selectParentPage({ parent: val }));
              }}
              required
            />
          </div>
          {/* container col distance */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerColDistance"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">
                Konténerben található oszlopok:{" "}
              </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy mekkora távolság legyen px-ben a konténerben
                található oszlopok között. (alapértelmezett: 10px)
              </span>
            </label>

            <Select
              options={initialData.editor.container.containerColDistance}
              className="w-3/4"
              menuPosition={"fixed"}
              placeholder={"Válasszon oszlop távolságot"}
              backspaceRemovesValue={false}
              name={"containerColDistance"}
              ref={(r) => (inputRefs.current[index++] = r)}
              defaultValue={
                containerOptions?.containerColDistance
                  ? initialData.editor.container.containerColDistance.filter(
                      (x) =>
                        parseInt(x.value) ===
                        parseInt(containerOptions?.containerColDistance)
                    )
                  : initialData.editor.container.containerColDistance[2]
              }
              onChange={(val, e) => {
                // second parameter shows action triggered
                // dispatch(selectParentPage({ parent: val }));
              }}
              required
            />
          </div>
          {/* container visibility on mobile */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"visibleOnMobile"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">
                Konténer láthatósága mobil eszközön:{" "}
              </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy megjelenik-e a konténer mobil eszközön.
                (alapértelmezett: Látható)
              </span>
            </label>
            <Select
              options={initialData.visibleOnMobile}
              className="w-3/4"
              menuPosition={"fixed"}
              placeholder={
                "Válassza ki, hogy megjelenik-e a konténer mobil eszközön"
              }
              backspaceRemovesValue={false}
              name={"visibleOnMobile"}
              ref={(r) => (inputRefs.current[index++] = r)}
              defaultValue={
                containerOptions?.visibleOnMobile
                  ? initialData.visibleOnMobile.filter(
                      (x) => x.value === containerOptions?.visibleOnMobile
                    )
                  : initialData.visibleOnMobile[1]
              }
              onChange={(val, e) => {
                // second parameter shows action triggered
                // dispatch(selectParentPage({ parent: val }));
              }}
              required
            />
          </div>
        </div>
        {/* container settings */}
        <div className={`w-full ${activeTab !== 2 ? "hidden" : ""}`}>
          {/* container max heigth */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerMaxHeight"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Konténer maximum magasság: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Megadhatja, hogy a konténer elemnek mekkora maximum magassága
                legyen px-ben. Alapértelmezett:
              </span>
            </label>

            <div className="flex flex-row flex-nowrap justify-evenly items-center">
              <div className="flex flex-row flex-nowrap w-3/4 justify-evenly items-center max-h-10">
                <span className="fill-gray-600 w-8 h-8">
                  {icons.up}
                  {icons.down}
                </span>
                <input
                  type="text"
                  name="containerMaxHeight"
                  id="containerMaxHeight"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={containerOptions?.containerMaxHeight ?? "auto"}
                />
              </div>
            </div>
          </div>
          {/* container margin */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerMargin"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Konténer margin: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Megadhatja, hogy a konténer elemnek mekkora marginja legyen
                px-ben. Alapértelmezett: 0
              </span>
            </label>

            <div className="flex flex-row flex-nowrap justify-evenly items-center">
              <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                <span className="fill-gray-600 w-8 h-8">{icons.up}</span>
                <input
                  type="text"
                  name="containerMarginTop"
                  id="containerMarginTop"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={containerOptions?.containerMarginTop ?? 0}
                />
              </div>
              <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                <span className="fill-gray-600 w-8 h-8">{icons.right}</span>
                <input
                  type="text"
                  name="containerMarginRight"
                  id="containerMarginRight"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={containerOptions?.containerMarginRight ?? 0}
                />
              </div>
              <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                <span className="fill-gray-600 w-8 h-8">{icons.down}</span>
                <input
                  type="text"
                  name="containerMarginBottom"
                  id="containerMarginBottom"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={containerOptions?.containerMarginBottom ?? 0}
                />
              </div>
              <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                <span className="fill-gray-600 w-8 h-8">{icons.left}</span>
                <input
                  type="text"
                  name="containerMarginLeft"
                  id="containerMarginLeft"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={containerOptions?.containerMarginLeft ?? 0}
                />
              </div>
            </div>
          </div>
          {/* container padding */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerPadding"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Konténer padding: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Megadhatja, hogy a konténer elemnek mekkora paddingja legyen
                px-ben. Alapértelmezett: 0 30 0 30
              </span>
            </label>
            <div className="flex flex-row flex-nowrap justify-evenly items-center">
              <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                <span className="fill-gray-600 w-8 h-8">{icons.up}</span>
                <input
                  type="text"
                  name="containerPaddingTop"
                  id="containerPaddingTop"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={containerOptions?.containerPaddingTop ?? 0}
                />
              </div>
              <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                <span className="fill-gray-600 w-8 h-8">{icons.right}</span>
                <input
                  type="text"
                  name="containerPaddingRight"
                  id="containerPaddingRight"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={containerOptions?.containerPaddingRight ?? 30}
                />
              </div>
              <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                <span className="fill-gray-600 w-8 h-8">{icons.down}</span>
                <input
                  type="text"
                  name="containerPaddingBottom"
                  id="containerPaddingBottom"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={containerOptions?.containerPaddingBottom ?? 0}
                />
              </div>
              <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                <span className="fill-gray-600 w-8 h-8">{icons.left}</span>
                <input
                  type="text"
                  name="containerPaddingLeft"
                  id="containerPaddingLeft"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={containerOptions?.containerPaddingLeft ?? 30}
                />
              </div>
            </div>
          </div>
          {/* container overflow */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerOverflow"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">
                Konténer tartalmának túlcsordulása:{" "}
              </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy milyen formában jelenlen meg a konténerben
                található túlcsorduló tartalom. (alapértelmezett: Elrejtve)
              </span>
            </label>

            <Select
              options={initialData.overflow}
              className="w-3/4"
              menuPosition={"fixed"}
              placeholder={"Válasszon oszlop távolságot"}
              backspaceRemovesValue={false}
              name={"containerOverflow"}
              ref={(r) => (inputRefs.current[index++] = r)}
              defaultValue={
                containerOptions?.containerOverflow
                  ? initialData.overflow.filter(
                      (x) => x.value === containerOptions.containerOverflow
                    )
                  : initialData.overflow[0]
              }
              onChange={(val, e) => {
                // second parameter shows action triggered
                // dispatch(selectParentPage({ parent: val }));
              }}
              required
            />
          </div>
        </div>
        {/* container design settings */}
        <div className={`w-full ${activeTab !== 3 ? "hidden" : ""}`}>
          {/* container background color */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerBackgroundColor"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">
                Konténer háttér színének beállítása:{" "}
              </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Megadhatja, hogy milyen színű legyen a konténer háttere.
                Alapértelmezett: rgba(255, 255, 255, 1)
              </span>
            </label>

            <ColorPicker
              value={containerBackgroundColor}
              onChange={setContainerBackgroundColor}
            />
          </div>
          {/* box auto shadow */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"autoShadow"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Konténer automatikus árnyék: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy legyen-e automatikus konténernek árnyéka.
                (alapértelmezett: Nem)
              </span>
            </label>

            <Select
              options={initialData.editor.container.autoShadow}
              className="w-3/4"
              menuPosition={"fixed"}
              placeholder={"Válasszon konténer árnyék ki/be kapcsolást"}
              backspaceRemovesValue={false}
              name={"autoShadow"}
              ref={(r) => (inputRefs.current[index++] = r)}
              defaultValue={
                containerOptions?.autoShadow
                  ? initialData.editor.container.autoShadow.filter(
                      (x) => x.value === containerOptions?.autoShadow
                    )
                  : initialData.editor.container.autoShadow[5]
              }
              onChange={(val, e) => {
                // second parameter shows action triggered
                // dispatch(selectParentPage({ parent: val }));
              }}
              required
            />
          </div>
          {/* container border */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerBorder"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Konténer kerete: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Megadhatja, hogy a konténer elemnek mekkora kerete legyen
                px-ben. Alapértelmezett: 0
              </span>
            </label>
            <div className="flex flex-row flex-nowrap justify-evenly items-center">
              <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                <span className="fill-gray-600 w-8 h-8">{icons.up}</span>
                <input
                  type="text"
                  name="containerBorderTop"
                  id="containerBorderTop"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={containerOptions?.containerBorderTop ?? 0}
                />
              </div>
              <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                <span className="fill-gray-600  w-8 h-8">{icons.right}</span>
                <input
                  type="text"
                  name="containerBorderRight"
                  id="containerBorderRight"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={containerOptions?.containerBorderRight ?? 0}
                />
              </div>
              <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                <span className="fill-gray-600 w-8 h-8">{icons.down}</span>
                <input
                  type="text"
                  name="containerBorderBottom"
                  id="containerBorderBottom"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={containerOptions?.containerBorderBottom ?? 0}
                />
              </div>
              <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                <span className="fill-gray-600 w-8 h-8">{icons.left}</span>
                <input
                  type="text"
                  name="containerBorderLeft"
                  id="containerBorderLeft"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={containerOptions?.containerBorderLeft ?? 0}
                />
              </div>
            </div>
          </div>
          {/* container border design */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"borderDesign"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Konténer keret stílus: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy a konténer keretének milyen stílusa legyen.
                (alapértelmezett: Solid)
              </span>
            </label>
            <Select
              options={initialData.borderStyle}
              className="w-3/4"
              menuPosition={"fixed"}
              placeholder={
                "Válassza ki, hogy a konténer keretének milyen stílusa legyen."
              }
              backspaceRemovesValue={false}
              name={"borderDesign"}
              ref={(r) => (inputRefs.current[index++] = r)}
              defaultValue={
                containerOptions?.borderDesign
                  ? initialData.borderStyle.filter(
                      (x) => x.value === containerOptions?.borderDesign
                    )
                  : initialData.borderStyle[2]
              }
              onChange={(val, e) => {
                // second parameter shows action triggered
                // dispatch(selectParentPage({ parent: val }));
              }}
              required
            />
          </div>
          {/* container border color */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerBorderColor"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">
                Konténer keret színének beállítása:{" "}
              </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Megadhatja, hogy milyen színű legyen a konténer kerete.
                Alapértelmezett: #ffffff
              </span>
            </label>

            <ColorPicker
              value={containerBorderColor}
              onChange={setContainerBorderColor}
            />
          </div>
          {/* box shadow */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerBoxShadow"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Konténer árnyék: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy legyen-e a konténernek árnyéka.
                (alapértelmezett: Nem)
              </span>
            </label>

            <Select
              options={initialData.editor.container.containerBoxShadow}
              className="w-3/4"
              menuPosition={"fixed"}
              placeholder={"Válasszon konténer árnyék ki/be kapcsolást"}
              backspaceRemovesValue={false}
              name={"containerBoxShadow"}
              ref={(r) => (inputRefs.current[index++] = r)}
              defaultValue={
                containerOptions?.containerBoxShadow
                  ? initialData.editor.container.containerBoxShadow.filter(
                      (x) => x.value === containerOptions.containerBoxShadow
                    )
                  : initialData.editor.container.containerBoxShadow[1]
              }
              onChange={(val, e) => {
                // second parameter shows action triggered
                // dispatch(selectParentPage({ parent: val }));
              }}
              required
            />
          </div>
          {/* container box shadow position */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerBoxShadowPosition"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">
                Konténer árnyékolás pozíciója:{" "}
              </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Pozitív érték esetén az árnyék a konténer alatt és jobbra,
                negatív érték esetén fölötte és balra jelenik meg (pl.: 5px)
              </span>
            </label>

            <div className="flex flex-row flex-nowrap justify-evenly items-center">
              <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                <span className="flex flex-col justify-center items-center fill-gray-600 w-8 h-8">
                  {icons.up}
                  {icons.down}
                </span>
                <input
                  type="text"
                  name="containerBoxShadowVertical"
                  id="containerBoxShadowVertical"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={
                    containerOptions?.containerBoxShadowVertical ?? 0
                  }
                />
              </div>
              <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
                <span className="flex flex-col justify-center items-center fill-gray-600 w-8 h-8">
                  {icons.left}
                  {icons.right}
                </span>
                <input
                  type="text"
                  name="containerBoxShadowHorizontal"
                  id="containerBoxShadowHorizontal"
                  ref={(r) => (inputRefs.current[index++] = r)}
                  className="w-3/4 border h-max p-2"
                  defaultValue={
                    containerOptions?.containerBoxShadowHorizontal ?? 0
                  }
                />
              </div>
            </div>
          </div>
          {/* container box shadow blur radius */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerBoxShadowBlurRadius"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">
                Konténer árnyék elmosás mérték - blur radius:{" "}
              </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Az árnyékolás elmosásának értéke - px-ben. 0-100 közötti értéket
                adhat meg. Alapértelmezett: 0
              </span>
            </label>

            <input
              type="text"
              name="containerBoxShadowBlurRadius"
              id="containerBoxShadowBlurRadius"
              ref={(r) => (inputRefs.current[index++] = r)}
              onChange={(e) => handleInputMinMax(e)}
              className="w-3/4 border h-max p-2"
              defaultValue={containerOptions?.containerBoxShadowBlurRadius ?? 0}
            />
          </div>
          {/* container box shadow spread radius */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerBoxShadowSpreadRadius"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">
                Konténer árnyék szórási értéke - spread radius:{" "}
              </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Az árnyékolás szórási értéke - px-ben. -100 és +100 közötti
                értéket adhat meg. Alapértelmezett: 0
              </span>
            </label>

            <input
              type="text"
              name="containerBoxShadowSpreadRadius"
              id="containerBoxShadowSpreadRadius"
              ref={(r) => (inputRefs.current[index++] = r)}
              onChange={(e) => handleInputMinMax(e)}
              className="w-3/4 border h-max p-2"
              defaultValue={
                containerOptions?.containerBoxShadowSpreadRadius ?? 0
              }
            />
          </div>
          {/* container box shadow color */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerBoxShadowColor"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">
                Konténer árnyék színének beállítása:{" "}
              </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Megadhatja, hogy milyen színű legyen a konténer által vetett
                árnyék. Alapértelmezett: #ffffff
              </span>
            </label>

            <ColorPicker
              value={containerBoxShadowColor}
              onChange={setContainerBoxShadowColor}
            />
          </div>
          {/* container link color */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerLinkColor"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">
                Konténerben levő linkek színének beállítása:{" "}
              </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Megadhatja, hogy milyen színű legyen a konténerben található
                linkek. Alapértelmezett: #212121
              </span>
            </label>

            <ColorPicker
              value={containerLinkColor}
              onChange={setContainerLinkColor}
            />
          </div>
          {/* container link hover color */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"containerLinkHoverColor"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">
                Konténerben levő linkek hover effektjének beállítása:{" "}
              </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Megadhatja, hogy milyen színű legyen a konténerben található
                linkek hover effektje (erre a színre változik a link, ha az
                egeret fölé viszi). Alapértelmezett: #0891b2
              </span>
            </label>

            <ColorPicker
              value={containerLinkHoverColor}
              onChange={setContainerLinkHoverColor}
            />
          </div>
        </div>
        {/* container image settings */}
        <div className={`w-full ${activeTab !== 4 ? "hidden" : ""}`}>
          {/* select image */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"imageSelect"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Kép kiválasztása: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Töltsön fel képet, vagy válasszon a médiatárból!
              </span>
            </label>
            <div className="flex flex-row w-3/4 justify-end items-center">
              {generateSelectedImage()}
              <Button
                onClick={() => showSelectImageModal(true)}
                className=" border border-gray-300 px-4 py-2 text-gray-500 hover:border-gray-600 hover:text-gray-900"
                btntext="Kép kiválasztása"></Button>
            </div>
          </div>
          {/* image position */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"imageBackgroundPosition"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Kép pozíciója: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki a háttér megjelenítése pozícióját. Csak abban az
                esetben érvényesül, ha a kép típusa: Háttérkép (alapértelmezett:
                Inherit - Örökölt)
              </span>
            </label>

            <Select
              options={initialData.editor.image.imageBackgroundPosition}
              className="w-3/4"
              menuPosition={"fixed"}
              placeholder={"Válasszon háttér pozíciót"}
              backspaceRemovesValue={false}
              name={"imageBackgroundPosition"}
              ref={(r) => (inputRefs.current[index++] = r)}
              defaultValue={
                containerOptions?.imageBackgroundPosition
                  ? initialData?.editor?.image?.imageBackgroundPosition?.filter(
                      (x) =>
                        x.value === containerOptions?.imageBackgroundPosition
                    )
                  : initialData?.editor?.image?.imageBackgroundPosition[0]
              }
              onChange={(val, e) => {
                // second parameter shows action triggered
                // dispatch(selectParentPage({ parent: val }));
              }}
              required
            />
          </div>
          {/* image object fit */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"imageObjectFit"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Kép objektum illeszkedése: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy a képet tartalmazó dobozban hogyan viselkedjen
                a kép. Alapértelmezett: Cover (Kitölti a teljes doboz méretet)
              </span>
            </label>
            <Select
              options={initialData.editor.image.imageObjectFit}
              className="w-3/4"
              menuPosition={"fixed"}
              placeholder={"Válasszon háttér pozíciót"}
              backspaceRemovesValue={false}
              name={"imageObjectFit"}
              ref={(r) => (inputRefs.current[index++] = r)}
              defaultValue={
                containerOptions?.imageObjectFit
                  ? initialData?.editor?.image?.imageObjectFit.filter(
                      (x) => x.value === containerOptions?.imageObjectFit
                    )
                  : initialData?.editor?.image?.imageObjectFit[1]
              }
              onChange={(val, e) => {
                // second parameter shows action triggered
                // dispatch(selectParentPage({ parent: val }));
              }}
              required
            />
          </div>
          {/* image object position */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"imageObjectPosition"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Kép objektum pozíciója: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy a kép melyik részére fókuszáljon a dobozban.
                Alapértelmezett: Center (A kép közepe kerül fókuszba)
              </span>
            </label>
            <Select
              options={initialData?.editor?.image?.imageObjectPosition}
              className="w-3/4"
              menuPosition={"fixed"}
              placeholder={"Válasszon háttér pozíciót"}
              backspaceRemovesValue={false}
              name={"imageObjectPosition"}
              ref={(r) => (inputRefs.current[index++] = r)}
              isMulti
              defaultValue={
                containerOptions?.imageObjectPosition
                  ? initialData?.editor?.image?.imageObjectPosition?.filter(
                      (x) =>
                        containerOptions?.imageObjectPosition?.includes(x.value)
                    )
                  : initialData?.editor?.image?.imageObjectPosition[1]
              }
              onChange={(val, e) => {
                // second parameter shows action triggered
                // dispatch(selectParentPage({ parent: val }));
              }}
              required
            />
          </div>
          {/* image dimensions */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"altText"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Kép dimenziójának megadása: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Adjon meg aránypárt, amely alapján a kép generálódik. Az itt
                megadott érték (*100) lesz a kép pixelben megadott szélessége és
                magassága (pl.: 16:9 - 1600px:900px méretű kép generálódik).
                Alapértelmezett: 4:3
              </span>
            </label>
            <div className="w-3/4 flex flex-row flex-nowrap justify-between">
              <input
                type="number"
                name="imgWidth"
                id="imgWidth"
                ref={(r) => (inputRefs.current[index++] = r)}
                className="w-1/2 border h-max p-2"
                defaultValue={containerOptions?.imgWidth ?? 4}
              />
              <input
                type="number"
                name="imgHeight"
                id="imgHeight"
                ref={(r) => (inputRefs.current[index++] = r)}
                className="w-1/2 border h-max p-2"
                defaultValue={containerOptions?.imgHeight ?? 3}
              />
            </div>
          </div>
          {/* image cut to size */}
          <div className="flex flex-row w-full mb-4 pb-4 border-b">
            <label
              htmlFor={"imageCutToSize"}
              className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold">Kép vágása: </p>
              <span className="w-full text-sm text-gray-600 whitespace-pre-line">
                Válassza ki, hogy a kép vágva legyen-e a megadott arányra.
                Alapértelmezett: Igen
              </span>
            </label>
            <Select
              options={initialData?.bool}
              className="w-3/4"
              menuPosition={"fixed"}
              placeholder={"Válasszon háttér pozíciót"}
              backspaceRemovesValue={false}
              name={"imageCutToSize"}
              ref={(r) => (inputRefs.current[index++] = r)}
              defaultValue={
                containerOptions?.imageCutToSize
                  ? initialData?.bool?.filter(
                      (x) => x.value === containerOptions?.imageCutToSize
                    )
                  : initialData?.bool[0]
              }
              onChange={(val, e) => {
                // second parameter shows action triggered
                // dispatch(selectParentPage({ parent: val }));
              }}
              required
            />
          </div>
        </div>
      </div>
      <div className="flex flex-row w-full justify-end sticky bottom-0 bg-white p-2 border border-t">
        <Button
          onClick={submitContainerOptions}
          className="px-4 py-2 uppercase border bg-green-500/75 text-white hover:border-green-500 hover:bg-green-500 disabled:bg-gray-500 disabled:pointer-events-none"
          btntext="Mentés"
        />
      </div>

      <MediaGalleryModal
        isOpen={selectImageModal}
        onClose={() => {
          showSelectImageModal(false);
        }}
        action="container-image"
        onConfirmSelection={(selectedImage) => {
          setSelectedImage({ selectedImage: selectedImage });
          showSelectImageModal(false); // ✅ Close the modal manually
        }}
      />
    </div>
  );
};

export default ContainerSettings;

// "use client";
// import React, {
//   useState,
//   useRef,
//   useCallback,
//   useEffect,
//   useMemo,
// } from "react";
// import dynamic from "next/dynamic";
// import ColorPickerComponent from "@/components/shared/colorpicker/ColorPickerComponent";
// import Select from "react-select";

// import Button from "@/components/shared/button/Button";
// import MediaGalleryModal from "@/components/media-store/GalleryModal/MediaGalleryModal";
// import MediaObject from "@/components/shared/mediaobject/MediaObject";
// import Image from "next/image";
// import ApiCall from "@/api/api-calls";
// import { initialData } from "@/data/data";
// import icons from "@/assets/icons/icons";
// import useDynamicManager from "@/store/managers/useDynamicManager";
// // Reusable input component
// const InputField = ({
//   label,
//   id,
//   description,
//   defaultValue,
//   inputRefs,
//   index,
// }) => (
//   <div className="flex flex-row w-full mb-4 pb-4 border-b">
//     <label htmlFor={id} className="w-1/4 text-gray-400 pr-2">
//       <p className="font-bold">{label}</p>
//       {description && <p className="text-sm text-gray-600">{description}</p>}
//     </label>
//     <input
//       type="text"
//       id={id}
//       name={id}
//       className="w-3/4 border p-2"
//       defaultValue={defaultValue}
//       ref={(r) => (inputRefs.current[index.current++] = r)}
//     />
//   </div>
// );

// // Reusable select component
// const SelectField = ({ label, id, options, selected, inputRefs, index }) => (
//   <div className="flex flex-row w-full mb-4 pb-4 border-b">
//     <label htmlFor={id} className="w-1/4 text-gray-400 pr-2">
//       <p className="font-bold">{label}</p>
//     </label>
//     <Select
//       options={options}
//       className="w-3/4"
//       menuPosition={"fixed"}
//       name={id}
//       isMulti
//       placeholder={`Válasszon ${label.toLowerCase()}...`}
//       backspaceRemovesValue={false}
//       defaultValue={options.find((x) => x.value === selected) || options[0]}
//       ref={(r) => (inputRefs.current[index.current++] = r)}
//     />
//   </div>
// );

// // Reusable color picker
// const ColorField = ({ label, value, onChange }) => (
//   <div className="flex flex-row w-full mb-4 pb-4 border-b">
//     <label className="w-1/4 text-gray-400 pr-2">
//       <p className="font-bold">{label}</p>
//     </label>
//     <ColorPickerComponent value={value} onChange={onChange} />
//   </div>
// );
// const ContainerSettings = ({ toggleModal, modalName, type }) => {
//   const [activeTab, setActiveTab] = useState(1);
//   const [selectImageModal, showSelectImageModal] = useState(false);
//   const inputRefs = useRef([]);
//   const index = useRef(0);
//   const { state, updateContainerOptions, setSelectedImage, resetImageState } =
//     useDynamicManager(type);

//   const containerOptions = useMemo(
//     () => state.selectedContainer.options ?? {},
//     [state.selectedContainer.options]
//   );

//   const [colors, setColors] = useState({
//     containerLinkColor:
//       containerOptions.containerLinkColor ?? "rgba(33,33,33,1)",
//     containerLinkHoverColor:
//       containerOptions.containerLinkHoverColor ?? "rgba(8,145,178,1)",
//     containerBorderColor:
//       containerOptions.containerBorderColor ?? "rgba(8,145,178,1)",
//     containerBackgroundColor:
//       containerOptions.containerBackgroundColor ?? "rgba(255,255,255,1)",
//     containerBoxShadowColor:
//       containerOptions.containerBoxShadowColor ?? "rgba(255,255,255,1)",
//   });

//   useEffect(() => {
//     if (!containerOptions) return;

//     setColors({
//       containerLinkColor:
//         containerOptions.containerLinkColor ?? "rgba(33,33,33,1)",
//       containerLinkHoverColor:
//         containerOptions.containerLinkHoverColor ?? "rgba(8,145,178,1)",
//       containerBorderColor:
//         containerOptions.containerBorderColor ?? "rgba(8,145,178,1)",
//       containerBackgroundColor:
//         containerOptions.containerBackgroundColor ?? "rgba(255,255,255,1)",
//       containerBoxShadowColor:
//         containerOptions.containerBoxShadowColor ?? "rgba(255,255,255,1)",
//     });
//   }, [containerOptions]);

//   const renderContainerLayoutTab = (
//     containerOptions,
//     initialData,
//     inputRefs,
//     index
//   ) => (
//     <>
//       <SelectField
//         label="Layout elrendezés típusa"
//         id="containerLayoutMode"
//         options={initialData.containerLayoutMode}
//         selected={containerOptions.containerLayoutMode}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <InputField
//         label="Grid oszlopok száma"
//         id="gridCols"
//         defaultValue={containerOptions.gridCols ?? 12}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <InputField
//         label="Grid gap (pl. 10px vagy 1rem)"
//         id="gridGap"
//         defaultValue={containerOptions.gridGap ?? "10px"}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <InputField
//         label="Grid sablon oszlopok (pl. repeat(3, 1fr))"
//         id="gridTemplateColumns"
//         defaultValue={containerOptions.gridTemplateColumns ?? ""}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <InputField
//         label="Grid sablon sorok (pl. auto 1fr auto)"
//         id="gridTemplateRows"
//         defaultValue={containerOptions.gridTemplateRows ?? ""}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <SelectField
//         label="Konténer tartalom szélessége"
//         id="containerWidth"
//         options={initialData.settings.layout}
//         selected={containerOptions.containerWidth}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <SelectField
//         label="Konténer tartalom elrendezés"
//         id="contentOrder"
//         options={initialData.contentOrder}
//         selected={containerOptions.contentOrder}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <SelectField
//         label="Konténer tartalom elrendezés töréspontoknál"
//         id="contentOrderBreakpoints"
//         options={initialData.orderBreakpoints}
//         selected={containerOptions.contentOrderBreakpoints}
//         inputRefs={inputRefs}
//         index={index}
//       />

//       <SelectField
//         label="Tartalom elhelyezkedése X tengelyen"
//         id="containerJustify"
//         options={initialData.justify}
//         selected={containerOptions.containerJustify}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <SelectField
//         label="Tartalom elhelyezkedése Y tengelyen"
//         id="containerAlign"
//         options={initialData.align}
//         selected={containerOptions.containerAlign}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <SelectField
//         label="Konténerben található oszlopok távolsága (px)"
//         id="containerColDistance"
//         options={initialData.editor.container.containerColDistance}
//         selected={containerOptions.containerColDistance}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <SelectField
//         label="Konténer láthatósága mobil eszközön"
//         id="visibleOnMobile"
//         options={initialData.visibleOnMobile}
//         selected={containerOptions.visibleOnMobile}
//         inputRefs={inputRefs}
//         index={index}
//       />
//     </>
//   );
//   const renderContainerSettingsTab = (
//     containerOptions,
//     inputRefs,
//     index,
//     icons
//   ) => (
//     <>
//       <InputField
//         label="Konténer maximum magassága (px)"
//         id="containerMaxHeight"
//         defaultValue={containerOptions.containerMaxHeight ?? "auto"}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       {/* Container Margin Inputs */}
//       <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
//         {[
//           { id: "containerMarginTop", icon: icons.up },
//           { id: "containerMarginRight", icon: icons.right },
//           { id: "containerMarginBottom", icon: icons.down },
//           { id: "containerMarginLeft", icon: icons.left },
//         ].map(({ id, icon }) => (
//           <InputField
//             key={id}
//             label={`Konténer margó: ${id.replace("containerMargin", "")}`}
//             id={id}
//             defaultValue={containerOptions[id] ?? 0}
//             inputRefs={inputRefs}
//             index={index}
//           />
//         ))}
//       </div>

//       {/* Container Padding Inputs */}
//       <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
//         {[
//           { id: "containerPaddingTop", icon: icons.up },
//           { id: "containerPaddingRight", icon: icons.right },
//           { id: "containerPaddingBottom", icon: icons.down },
//           { id: "containerPaddingLeft", icon: icons.left },
//         ].map(({ id, icon }) => (
//           <InputField
//             key={id}
//             label={`Konténer padding: ${id.replace("containerPadding", "")}`}
//             id={id}
//             defaultValue={
//               containerOptions[id] ??
//               (id.includes("Right") || id.includes("Left") ? 30 : 0)
//             }
//             inputRefs={inputRefs}
//             index={index}
//           />
//         ))}
//       </div>

//       <SelectField
//         label="Konténer tartalom túlcsordulás kezelése"
//         id="containerOverflow"
//         options={initialData.overflow}
//         selected={containerOptions.containerOverflow}
//         inputRefs={inputRefs}
//         index={index}
//       />
//     </>
//   );

//   const renderContainerDesignTab = (
//     containerOptions,
//     initialData,
//     inputRefs,
//     index,
//     icons,
//     colors,
//     setColors
//   ) => (
//     <>
//       <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
//         <ColorField
//           label="Konténer háttér szín"
//           value={colors.containerBackgroundColor}
//           onChange={(val) =>
//             setColors((prev) => ({ ...prev, containerBackgroundColor: val }))
//           }
//         />
//         <SelectField
//           label="Automatikus árnyék"
//           id="autoShadow"
//           options={initialData.editor.container.autoShadow}
//           selected={containerOptions.autoShadow}
//           inputRefs={inputRefs}
//           index={index}
//         />
//       </div>
//       <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
//         {["Top", "Right", "Bottom", "Left"].map((dir) => (
//           <InputField
//             key={`containerBorder${dir}`}
//             label={`Keret (${dir}) (px)`}
//             id={`containerBorder${dir}`}
//             defaultValue={containerOptions[`containerBorder${dir}`] ?? 0}
//             inputRefs={inputRefs}
//             index={index}
//           />
//         ))}
//       </div>
//       <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
//         <ColorField
//           label="Keret szín"
//           value={colors.containerBorderColor}
//           onChange={(val) =>
//             setColors((prev) => ({ ...prev, containerBorderColor: val }))
//           }
//         />
//         <SelectField
//           label="Keret stílus"
//           id="borderDesign"
//           options={initialData.borderStyle}
//           selected={containerOptions.borderDesign}
//           inputRefs={inputRefs}
//           index={index}
//         />
//       </div>
//       <SelectField
//         label="Konténer árnyék"
//         id="containerBoxShadow"
//         options={initialData.editor.container.containerBoxShadow}
//         selected={containerOptions.containerBoxShadow}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
//         <InputField
//           label="Árnyék pozíció (vertical)"
//           id="containerBoxShadowVertical"
//           defaultValue={containerOptions.containerBoxShadowVertical ?? 0}
//           inputRefs={inputRefs}
//           index={index}
//         />
//         <InputField
//           label="Árnyék pozíció (horizontal)"
//           id="containerBoxShadowHorizontal"
//           defaultValue={containerOptions.containerBoxShadowHorizontal ?? 0}
//           inputRefs={inputRefs}
//           index={index}
//         />
//         <InputField
//           label="Árnyék blur radius"
//           id="containerBoxShadowBlurRadius"
//           defaultValue={containerOptions.containerBoxShadowBlurRadius ?? 0}
//           inputRefs={inputRefs}
//           index={index}
//         />
//         <InputField
//           label="Árnyék spread radius"
//           id="containerBoxShadowSpreadRadius"
//           defaultValue={containerOptions.containerBoxShadowSpreadRadius ?? 0}
//           inputRefs={inputRefs}
//           index={index}
//         />
//       </div>
//       <ColorField
//         label="Árnyék szín"
//         value={colors.containerBoxShadowColor}
//         onChange={(val) =>
//           setColors((prev) => ({ ...prev, containerBoxShadowColor: val }))
//         }
//       />
//       <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
//         <ColorField
//           label="Link szín"
//           value={colors.containerLinkColor}
//           onChange={(val) =>
//             setColors((prev) => ({ ...prev, containerLinkColor: val }))
//           }
//         />
//         <ColorField
//           label="Link hover szín"
//           value={colors.containerLinkHoverColor}
//           onChange={(val) =>
//             setColors((prev) => ({ ...prev, containerLinkHoverColor: val }))
//           }
//         />
//       </div>
//     </>
//   );

//   const renderContainerImageTab = (
//     containerOptions,
//     initialData,
//     inputRefs,
//     index,
//     generateSelectedImage,
//     showSelectImageModal
//   ) => (
//     <>
//       <div className="flex flex-row w-full mb-4 pb-4 border-b">
//         <label htmlFor="imageSelect" className="w-1/4 text-gray-400 pr-2">
//           <p className="font-bold">Kép kiválasztása</p>
//           <p className="text-sm text-gray-600">
//             Töltsön fel képet vagy válasszon a médiatárból.
//           </p>
//         </label>
//         <div className="w-3/4 flex justify-end items-center">
//           {generateSelectedImage()}
//           <Button
//             onClick={() => showSelectImageModal(true)}
//             className="border border-gray-300 px-4 py-2 text-gray-500 hover:border-gray-600 hover:text-gray-900"
//             btntext="Kép kiválasztása"
//           />
//         </div>
//       </div>
//       <SelectField
//         label="Háttér pozíció"
//         id="imageBackgroundPosition"
//         options={initialData.editor.image.imageBackgroundPosition}
//         selected={containerOptions.imageBackgroundPosition}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <SelectField
//         label="Object fit"
//         id="imageObjectFit"
//         options={initialData.editor.image.imageObjectFit}
//         selected={containerOptions.imageObjectFit}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <SelectField
//         label="Object pozíció (több választás)"
//         id="imageObjectPosition"
//         options={initialData.editor.image.imageObjectPosition}
//         selected={containerOptions.imageObjectPosition}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <InputField
//         label="Kép arány - szélesség"
//         id="imgWidth"
//         defaultValue={containerOptions.imgWidth ?? 4}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <InputField
//         label="Kép arány - magasság"
//         id="imgHeight"
//         defaultValue={containerOptions.imgHeight ?? 3}
//         inputRefs={inputRefs}
//         index={index}
//       />
//       <SelectField
//         label="Kép vágása"
//         id="imageCutToSize"
//         options={initialData.bool}
//         selected={containerOptions.imageCutToSize}
//         inputRefs={inputRefs}
//         index={index}
//       />
//     </>
//   );

//   const handleColorChange = (key, value) => {
//     setColors((prev) => ({ ...prev, [key]: value }));
//   };

//   const handleImageRemove = () => {
//     resetImageState();
//   };

//   const generateSelectedImage = useCallback(() => {
//     const image = state.selectedImage;
//     if (!image || !image._id) return null;

//     const isPDF = image?.props?.mimetype === "application/pdf";
//     const imageElement = isPDF ? (
//       <MediaObject data={image} className="w-full" id={image._id} />
//     ) : (
//       <Image
//         src={ApiCall.mediaStore.getResized("/2/2/0", image.originalname)}
//         width={150}
//         height={100}
//         alt={image?.originalname ?? ""}
//         loading="lazy"
//       />
//     );

//     return (
//       <div
//         className="flex flex-col mr-2 mb-2 items-center border-1 w-full"
//         key={image._id}>
//         {imageElement}
//         <p className="break-all p-1">
//           {image?.props?.filename ?? image?.originalname}
//         </p>
//         <input type="hidden" name="selectedFile" defaultValue={image} />
//         <button onClick={handleImageRemove} className="text-red-500 my-4">
//           Eltávolítás
//         </button>
//       </div>
//     );
//   }, [state.selectedImage]);

//   const submitContainerOptions = () => {
//     const content = {
//       type: "containerOptions",
//       backgroundImage: state.selectedImage,
//       ...colors,
//     };
//     // Collect grid span inputs
//     const gridSpans = { col: {}, row: {} };

//     inputRefs.current.forEach((input) => {
//       if (input && input.name?.startsWith("gridSpan")) {
//         const [, type, bp] = input.name.split("-"); // e.g., gridSpan-col-sm => [gridSpan, col, sm]
//         const value = parseInt(input.value);
//         if (!isNaN(value)) {
//           gridSpans[type][bp] = value;
//         }
//       }
//     });
//     content.gridSpans = gridSpans;
//     inputRefs.current.forEach((ref) => {
//       if (!ref) return;

//       if (ref.props?.value) {
//         const val = ref.props.value;
//         content[ref.props.name] = Array.isArray(val)
//           ? val.map((x) => x.value)
//           : val.value;
//       } else if (ref.name) {
//         content[ref.name] = ref.value;
//       }
//     });
//     // content.containerLayoutMode = containerLayoutMode;
//     console.log("content: ", content);
//     updateContainerOptions({ options: content });
//     resetImageState();
//     toggleModal(modalName);
//   };

//   const renderTabs = [
//     { id: 1, label: "Tartalom elrendezése" },
//     { id: 2, label: "Konténer beállítások" },
//     { id: 3, label: "Konténer design" },
//     { id: 4, label: "Kép beállítások" },
//   ];

//   return (
//     <div className="relative">
//       <div className="container-options-wrapper flex flex-col w-full p-4">
//         <div className="w-full flex flex-row flex-nowrap border-b mb-4">
//           {renderTabs.map((tab) => (
//             <button
//               key={tab.id}
//               onClick={() => setActiveTab(tab.id)}
//               className={`px-4 py-2 border-t border-l ${
//                 activeTab === tab.id ? "bg-green-500 text-white" : ""
//               }`}>
//               {tab.label}
//             </button>
//           ))}
//         </div>
//         <div className="w-full">
//           <div className={activeTab !== 1 ? "hidden" : ""}>
//             {renderContainerLayoutTab(
//               containerOptions,
//               initialData,
//               inputRefs,
//               index
//             )}
//           </div>
//           <div className={activeTab !== 2 ? "hidden" : ""}>
//             {renderContainerSettingsTab(
//               containerOptions,
//               inputRefs,
//               index,
//               icons
//             )}
//           </div>
//           <div className={activeTab !== 3 ? "hidden" : ""}>
//             {renderContainerDesignTab(
//               containerOptions,
//               initialData,
//               inputRefs,
//               index,
//               icons,
//               colors,
//               setColors
//             )}
//           </div>
//           <div className={activeTab !== 4 ? "hidden" : ""}>
//             {renderContainerImageTab(
//               containerOptions,
//               initialData,
//               inputRefs,
//               index,
//               generateSelectedImage,
//               showSelectImageModal
//             )}
//           </div>
//         </div>
//       </div>

//       <div className="flex justify-end sticky bottom-0 bg-white p-2 border-t">
//         <Button
//           onClick={submitContainerOptions}
//           className="px-4 py-2 uppercase border bg-green-500/75 text-white hover:border-green-500 hover:bg-green-500"
//           btntext="Mentés"
//         />
//       </div>

//       <MediaGalleryModal
//         isOpen={selectImageModal}
//         onClose={() => showSelectImageModal(false)}
//         action="container-image"
//         onConfirmSelection={(img) => {
//           setSelectedImage({ selectedImage: img });
//           showSelectImageModal(false);
//         }}
//       />
//     </div>
//   );
// };

// export default ContainerSettings;
