"use client";
import React, { useCallback, useEffect, useState } from "react";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";
import useDynamicManager from "@/store/managers/useDynamicManager";
import Modal from "@/components/shared/modal/Modal";
import { initialData } from "@/data/data";
import icons from "@/assets/icons/icons";
import { useModalManager } from "@/hooks/useModalManager";
import Button from "@/components/shared/button/Button";
import useSelectableElements from "@/hooks/selectableElements";
import PageEditorFormFields from "../page-box/PageEditorFormFields";
import { useForm } from "@/hooks/custom-form-hook";
import Container from "@/components/editor/container/Container";
import { addContainer } from "@/store/slices/pageEditorSlice";
import useColumnActions from "@/hooks/useColumnActions";
import useContainerActions from "@/hooks/useContainerActions";

const CreatePageEditor = () => {
  const [pageEditorData, setPageEditorData] = useState(null);
  const [pagesList, setPagesList] = useState(null);
  const [boxColumns, setBoxColumns] = useState(null);
  const [formState, handleInput, handleSetInputData] = useForm([], false);
  const { toggleModal, isModalOpen } = useModalManager();
  const { state, ...rest } = useDynamicManager("pageEditor");
  const { handleAddContainer } = useContainerActions(
    initialData,
    toggleModal,
    "pageEditor"
  );

  useEffect(() => {
    const fetchPageEditorData = async () => {
      const response = await ApiCall.pageEditor.getCreatePageEditor();
      if (response.status === 200) {
        setPageEditorData(response?.data?.body?.data);
        setPagesList(response?.data?.body?.pagesList);
        toast.success("Oldaleszerkesztő adatok betöltve.");
      } else {
        toast.error("Oldalszerkesző adatok betöltése sikertelen.");
      }
    };
    fetchPageEditorData();
  }, []);

  useEffect(() => {
    const fetchBoxColumnsData = async () => {
      const response = await ApiCall.boxColumns.getAll();
      if (response.status === 200) {
        setBoxColumns(response?.data?.columnList);
        toast.success("Oszlop adatok betöltve.");
      } else {
        toast.error("Oszlop adatok betöltése sikertelen.");
      }
    };
    fetchBoxColumnsData();
  }, []);

  const handlePageEditorCreate = useCallback(() => {
    const sendData = async () => {
      const ret = {
        inputs: formState.inputs,
        boxes: state.containerData,
        isPublished: state?.isPublishedPage,
        showInMenu: state?.showPageInMenu,
        hasParent: state?.hasParentPage,
        parent: state?.selectedParentPage,
      };
      const response = await ApiCall.pageEditor.putCreatePageEditor(ret);
      if (response.status === 200) {
        toast.success(response.data.message || "Sikeresen létrehozva!");
      } else {
        toast.error(response.error || "Sikertelen mentés!");
      }
    };
    sendData();
  }, [state, formState]);

  return (
    <div className="w-full grid grid-cols-1 md:grid-cols-6">
      <div className="w-full col-span-5">
        <PageEditorFormFields
          data={pageEditorData}
          handleInput={handleInput}
          initialValid={false}
        />

        <Button
          className="w-full flex flex-row flex-nowrap justify-center items-center cursor-pointer border border-gray-600/50 text-white hover:bg-green-600 hover:text-white bg-gray-600 py-2 px-4 font-bold my-4"
          onClick={() => toggleModal("containerModal")}
          btntext="Konténer hozzáadása"
          icon={icons.add}
          iconclass="flex w-[1.5rem] h-[1.5rem] fill-white stroke-white ml-4"></Button>
        {state &&
          state?.containerData?.map((c, idx) => {
            return (
              <Container
                key={idx}
                data={c}
                alldata={state?.containerData}
                parent={idx}
                handleTitleChange={rest.setContainerTitle}
                type={"pageEditor"}
              />
            );
          })}
        <Button
          btntext="Mentés"
          disabled={!formState.isValid}
          onClick={handlePageEditorCreate}
          className="w-full cursor-pointer rounded bg-green-500/75 disabled:bg-gray-300 enabled:hover:bg-red-500 px-4 py-2 text-white"
        />
      </div>

      <div className="w-full col-span-1">
        <div className="flex flex-col w-full justify-center items-center">
          <label htmlFor="isPublished">Publikált oldal:</label>
          <input
            id={"isPublished"}
            type={"checkbox"}
            name={"isPublished"}
            onChange={(e) =>
              rest.setPublishedPage({ isPublishedPage: e.target.checked })
            }
            checked={!!state?.isPublishedPage}
            className={"flex flex-col m-2 w-6 h-6"}
          />
          <label htmlFor="showInMenu">Megjelenik a menüben:</label>
          <input
            id={"showInMenu"}
            type={"checkbox"}
            name={"showInMenu"}
            onChange={(e) =>
              rest.setShowPageInMenu({ showPageInMenu: e.target.checked })
            }
            checked={!!state?.showPageInMenu}
            className={"flex flex-col m-2 w-6 h-6"}
          />
          <label htmlFor="hasParent">Aloldalként megjelöl:</label>
          <input
            id={"hasParent"}
            type={"checkbox"}
            name={"hasParent"}
            onChange={(e) =>
              rest.setHasParentPage({ hasParentPage: e.target.checked })
            }
            checked={!!state?.hasParentPage}
            className={"flex flex-col m-2 w-6 h-6"}
          />
        </div>
      </div>

      {/**************************  Modals  **********************************/}
      {/* add container modal */}
      <Modal
        show={isModalOpen("containerModal")}
        onClose={() => toggleModal("containerModal")}
        title="Válasszon az alábbi konténerek közül:"
        height={"100%"}>
        <div className="flex flex-row flex-wrap justify-evenly">
          {boxColumns?.map((c, i) => {
            return (
              <div
                key={i}
                className="flex flex-row flex-nowrap mx-px my-4 w-1/6 border pointer-events-auto justify-between items-center border border-dashed border-gray-500/50 hover:border-gray-500 hover:cursor-pointer hover:bg-gray-600/75"
                onClick={() => {
                  handleAddContainer(c);
                }}>
                <div className="flex flex-row flex-nowrap w-full m-1 pointer-events-none">
                  {c?.columnSize?.map((col, idx) => {
                    const colX = col.x;
                    const colY = col.y;

                    return (
                      <div
                        key={idx}
                        className={`h-20 flex flex-row flex-nowrap justify-center items-center border border-white p-2 bg-gray-600/50 `}
                        style={{
                          width: `${((colX / colY) * 100).toFixed(0)}%`,
                        }}>
                        <p className="w-full flex flex-row justify-center items-center text-white text-sm">{`${colX}/${colY}`}</p>
                      </div>
                    );
                  })}
                </div>
              </div>
            );
          })}
        </div>
      </Modal>
    </div>
  );
};

export default CreatePageEditor;
