"use client";

import { useEffect, useState } from "react";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";
import EmailRegistryForm, { EmailUpsertPayload } from "./EmailRegistryForm";
import type { Row } from "@/app/admin/email-registry/list/page";
import Select from "react-select";

type CpanelStatusInitial = {
  suspendedLogin: boolean;
  suspendedIncoming: boolean;
  suspendedOutgoing: boolean;
};

type CpanelIdentity = {
  domain: string;
  localPart: string;
  email: string;
};

type AutoresponderFormState = {
  enabled: boolean;
  fromName: string;
  subject: string;
  body: string;
  isHtml: boolean;
  intervalHours: number;
  start: string; // ISO string for datetime-local
  stop: string;
};

type QuotaUnit = "MB" | "GB" | "TB" | "PB";

function convertToMb(value: number, unit: QuotaUnit): number {
  switch (unit) {
    case "MB":
      return value;
    case "GB":
      return value * 1024;
    case "TB":
      return value * 1024 * 1024;
    case "PB":
      return value * 1024 * 1024 * 1024;
  }
}

function getCpanelIdentity(row: Row): CpanelIdentity | null {
  if (!row.cpanel_exists) return null;

  const email = row.cpanel_email || row.email;
  if (!email) return null;

  const [localPart, domainFromEmail] = email.split("@");
  const domain = row.cpanel_domain || domainFromEmail;

  if (!localPart || !domain) return null;

  return {
    domain,
    localPart,
    email,
  };
}

function CpanelStatusEditor({
  identity,
  initial,
  emailId,
  onUpdated,
}: {
  identity: CpanelIdentity;
  initial: CpanelStatusInitial;
  emailId: string;
  onUpdated?: (next: CpanelStatusInitial) => void;
}) {
  type Mode = "allow" | "suspend";

  const [login, setLogin] = useState<Mode>(
    initial.suspendedLogin ? "suspend" : "allow"
  );
  const [incoming, setIncoming] = useState<Mode>(
    initial.suspendedIncoming ? "suspend" : "allow"
  );
  const [outgoing, setOutgoing] = useState<Mode>(
    initial.suspendedOutgoing ? "suspend" : "allow"
  );
  const [saving, setSaving] = useState(false);

  // When parent updates "initial", sync local selects:
  useEffect(() => {
    setLogin(initial.suspendedLogin ? "suspend" : "allow");
    setIncoming(initial.suspendedIncoming ? "suspend" : "allow");
    setOutgoing(initial.suspendedOutgoing ? "suspend" : "allow");
  }, [
    initial.suspendedLogin,
    initial.suspendedIncoming,
    initial.suspendedOutgoing,
  ]);

  const options = [
    { value: "allow", label: "Engedélyezve" },
    { value: "suspend", label: "Felfüggesztve" },
  ];

  const save = async () => {
    setSaving(true);
    try {
      // 1) Update registry statuses for reporting/export
      const registryPayload: Partial<EmailUpsertPayload> = {
        receive_status: incoming === "suspend" ? "suspended" : "allow",
        send_status: outgoing === "suspend" ? "suspended" : "allow",
        login_status: login === "suspend" ? "suspended" : "allow",
      };

      await ApiCall.emailRegistry.update(emailId, registryPayload);

      // 2) Update cPanel suspension flags
      await ApiCall.emailRegistry.updateAccount(
        identity.domain,
        identity.localPart,
        {
          suspendLogin: login === "suspend",
          suspendIncoming: incoming === "suspend",
          suspendOutgoing: outgoing === "suspend",
        }
      );

      const nextFlags: CpanelStatusInitial = {
        suspendedLogin: login === "suspend",
        suspendedIncoming: incoming === "suspend",
        suspendedOutgoing: outgoing === "suspend",
      };

      toast.success("cPanel + registry korlátozások frissítve");
      onUpdated?.(nextFlags);
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ??
          "cPanel / registry korlátozások frissítése sikertelen"
      );
    } finally {
      setSaving(false);
    }
  };

  return (
    <div className="mt-4 rounded border p-3 bg-slate-50/80 space-y-3">
      <div className="flex items-center justify-between">
        <div className="text-xs font-semibold text-gray-700">
          cPanel postafiók korlátozások
        </div>
        <span className="text-xs text-gray-500">{identity.email}</span>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-3 text-xs">
        <div className="flex flex-col">
          <label className="text-xs text-gray-500">
            Bejövő levél fogadása (cPanel)
          </label>
          <Select
            classNamePrefix="er"
            options={options}
            value={options.find((o) => o.value === incoming)}
            onChange={(v) => setIncoming((v as any)?.value ?? "allow")}
          />
        </div>

        <div className="flex flex-col">
          <label className="text-xs text-gray-500">
            Levél küldése (cPanel)
          </label>
          <Select
            classNamePrefix="er"
            options={options}
            value={options.find((o) => o.value === outgoing)}
            onChange={(v) => setOutgoing((v as any)?.value ?? "allow")}
          />
        </div>

        <div className="flex flex-col">
          <label className="text-xs text-gray-500">
            Bejelentkezés postafiókba (cPanel)
          </label>
          <Select
            classNamePrefix="er"
            options={options}
            value={options.find((o) => o.value === login)}
            onChange={(v) => setLogin((v as any)?.value ?? "allow")}
          />
        </div>
      </div>

      <div className="flex justify-end">
        <button
          type="button"
          onClick={save}
          disabled={saving}
          className="px-3 py-2 bg-gray-700/80 hover:bg-gray-700 text-white rounded text-xs disabled:opacity-50">
          {saving ? "Mentés…" : "cPanel korlátozások mentése"}
        </button>
      </div>

      <p className="text-xs text-gray-500">
        Ezek a beállítások közvetlenül a cPanel postafiók felfüggesztési
        állapotát módosítják (bejövő / kimenő / bejelentkezés).
      </p>
    </div>
  );
}
function CpanelAutoresponderEditor({ identity }: { identity: CpanelIdentity }) {
  const [state, setState] = useState<AutoresponderFormState>({
    enabled: false,
    fromName: "",
    subject: "",
    body: "",
    isHtml: false,
    intervalHours: 24,
    start: "",
    stop: "",
  });

  const [saving, setSaving] = useState(false);

  function update<K extends keyof AutoresponderFormState>(
    key: K,
    value: AutoresponderFormState[K]
  ) {
    setState((prev) => ({ ...prev, [key]: value }));
  }

  const handleSave = async () => {
    setSaving(true);
    try {
      await ApiCall.emailRegistry.updateAccount(
        identity.domain,
        identity.localPart,
        {
          autoresponder: {
            enabled: state.enabled,
            fromName: state.fromName,
            subject: state.subject,
            body: state.body,
            isHtml: state.isHtml,
            intervalHours: state.intervalHours,
            start: state.start || null,
            stop: state.stop || null,
          },
        }
      );

      toast.success("Automatikus válasz beállítva (cPanel)");
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Automatikus válasz mentése sikertelen"
      );
    } finally {
      setSaving(false);
    }
  };

  return (
    <form
      autoComplete="off"
      onSubmit={(e) => {
        e.preventDefault();
        handleSave();
      }}
      className="mt-4 rounded border p-3 bg-white/60 space-y-3">
      <div className="flex items-center justify-between">
        <div className="font-medium text-gray-800">
          Automatikus válasz (cPanel)
        </div>
        <span className="text-xs text-gray-500">{identity.email}</span>
      </div>

      <div className="flex items-center gap-2 text-xs">
        <label className="flex items-center gap-2">
          <input
            type="checkbox"
            checked={state.enabled}
            onChange={(e) => update("enabled", e.target.checked)}
          />
          <span>Automatikus válasz engedélyezése</span>
        </label>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-3 text-xs">
        <div className="flex flex-col">
          <label className="text-xs text-gray-500 mb-1">Feladó név</label>
          <input
            type="text"
            className="border rounded px-2 py-1"
            placeholder="Pl. Cég neve"
            value={state.fromName}
            onChange={(e) => update("fromName", e.target.value)}
            disabled={!state.enabled}
          />
        </div>

        <div className="flex flex-col">
          <label className="text-xs text-gray-500 mb-1">Tárgy</label>
          <input
            type="text"
            className="border rounded px-2 py-1"
            placeholder="Pl. Hivatalos távollét"
            value={state.subject}
            onChange={(e) => update("subject", e.target.value)}
            disabled={!state.enabled}
          />
        </div>
      </div>

      <div className="flex flex-col text-xs">
        <label className="text-xs text-gray-500 mb-1">Üzenet</label>
        <textarea
          className="border rounded px-2 py-1 min-h-[100px]"
          placeholder="Köszönöm levelét, jelenleg nem vagyok elérhető..."
          value={state.body}
          onChange={(e) => update("body", e.target.value)}
          disabled={!state.enabled}
        />
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-3 text-xs">
        <div className="flex flex-col">
          <label className="text-xs text-gray-500 mb-1">
            Válasz intervallum (óra)
          </label>
          <input
            type="number"
            min={0}
            className="border rounded px-2 py-1"
            value={state.intervalHours}
            onChange={(e) =>
              update(
                "intervalHours",
                e.target.value === "" ? 0 : Number(e.target.value)
              )
            }
            disabled={!state.enabled}
          />
        </div>

        <div className="flex flex-col">
          <label className="text-xs text-gray-500 mb-1">Kezdés</label>
          <input
            type="datetime-local"
            className="border rounded px-2 py-1"
            value={state.start}
            onChange={(e) => update("start", e.target.value)}
            disabled={!state.enabled}
          />
          <span className="text-[10px] text-gray-500 mt-1">
            Üresen hagyva: azonnal.
          </span>
        </div>

        <div className="flex flex-col">
          <label className="text-xs text-gray-500 mb-1">Lezárás</label>
          <input
            type="datetime-local"
            className="border rounded px-2 py-1"
            value={state.stop}
            onChange={(e) => update("stop", e.target.value)}
            disabled={!state.enabled}
          />
          <span className="text-[10px] text-gray-500 mt-1">
            Üresen hagyva: korlátlan ideig.
          </span>
        </div>
      </div>

      <div className="flex items-center justify-between text-xs">
        <label className="flex items-center gap-2">
          <input
            type="checkbox"
            checked={state.isHtml}
            onChange={(e) => update("isHtml", e.target.checked)}
            disabled={!state.enabled}
          />
          <span>Üzenet HTML formátumban</span>
        </label>

        <button
          type="submit"
          disabled={saving}
          className="px-3 py-2 bg-gray-700/80 hover:bg-gray-700 text-white rounded disabled:opacity-50">
          {saving ? "Mentés…" : "Automatikus válasz mentése"}
        </button>
      </div>

      <p className="text-[11px] text-gray-500">
        A beállítások közvetlenül a cPanel Email &gt; Autoresponders modulját
        módosítják.
      </p>
    </form>
  );
}
export default function ExpandedEmailRegistry({
  id,
  row,
  onSaved,
  onDeleted,
}: {
  id: string;
  row: Row;
  onSaved?: () => void;
  onDeleted?: () => void;
}) {
  const [loading, setLoading] = useState(true);
  const [item, setItem] = useState<any>(null);

  // Local source of truth for suspension booleans
  const [cpanelFlags, setCpanelFlags] = useState<CpanelStatusInitial>({
    suspendedLogin: !!row.cpanel_suspended_login,
    suspendedIncoming: !!row.cpanel_suspended_incoming,
    suspendedOutgoing: !!row.cpanel_suspended_outgoing,
  });

  // NEW: quota UI state (initial from row)
  const initialQuota =
    row.cpanel_diskquota != null ? Number(row.cpanel_diskquota) : 0;

  const [quotaType, setQuotaType] = useState<"limited" | "unlimited">(
    initialQuota > 0 ? "limited" : "unlimited"
  );
  const [quotaValue, setQuotaValue] = useState<number>(
    initialQuota > 0 ? initialQuota : 1024 // default 1024 MB if limited
  );
  const [quotaUnit, setQuotaUnit] = useState<QuotaUnit>("MB");
  const [savingQuota, setSavingQuota] = useState(false);

  const load = async () => {
    setLoading(true);
    try {
      const r = await ApiCall.emailRegistry.getOne(id);
      const nextItem = r.data?.item ?? null;
      setItem(nextItem);

      // If API returns cpanel_* flags, use them; otherwise keep current flags
      if (nextItem) {
        setCpanelFlags((prev) => ({
          suspendedLogin:
            typeof nextItem.cpanel_suspended_login === "boolean"
              ? nextItem.cpanel_suspended_login
              : prev.suspendedLogin,
          suspendedIncoming:
            typeof nextItem.cpanel_suspended_incoming === "boolean"
              ? nextItem.cpanel_suspended_incoming
              : prev.suspendedIncoming,
          suspendedOutgoing:
            typeof nextItem.cpanel_suspended_outgoing === "boolean"
              ? nextItem.cpanel_suspended_outgoing
              : prev.suspendedOutgoing,
        }));
      }
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Betöltés sikertelen");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (id) load();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [id]);

  // const handleSave = async (payload: EmailUpsertPayload) => {
  //   try {
  //     await ApiCall.emailRegistry.update(id, payload);
  //     toast.success("Frissítve");
  //     onSaved?.();
  //     await load();
  //   } catch (e: any) {
  //     console.error(e);
  //     toast.error(e?.response?.data?.error ?? "Mentés sikertelen");
  //   }
  // };

  const handleSave = async (payload: EmailUpsertPayload) => {
    try {
      // 1) Update registry (Postgres)
      await ApiCall.emailRegistry.update(id, payload);

      // 2) If we have a cPanel mailbox, also sync forwarders to cPanel
      const identity = getCpanelIdentity({
        ...row,
        email: payload.email ?? row.email, // just in case email was changed
      });

      if (row.cpanel_exists && identity) {
        const forwardTo = payload.is_forwarded ? payload.forward_to ?? [] : [];

        await ApiCall.emailRegistry.updateAccount(
          identity.domain,
          identity.localPart,
          {
            // we only care about forwarders here;
            // suspensions/password can still be handled by CpanelStatusEditor
            forwardTo,
            enableForward: !!payload.is_forwarded,
            // if you later add quota in the form, you can pass newQuotaMb here too
          }
        );
      }

      toast.success("Frissítve (nyilvántartás + cPanel átirányítás)");
      onSaved?.();
      await load();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Mentés sikertelen");
    }
  };

  const handleDelete = async () => {
    if (!confirm("Biztosan törli?")) return;
    try {
      await ApiCall.emailRegistry.remove(id);
      toast.success("Törölve");
      onDeleted?.();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Törlés sikertelen");
    }
  };

  function PasswordEditor({ emailId }: { emailId: string }) {
    const [loading, setLoading] = useState(true);
    const [hasPwd, setHasPwd] = useState(false);
    const [updatedAt, setUpdatedAt] = useState<string | null>(null);
    const [show, setShow] = useState(false);
    const [value, setValue] = useState("");

    const loadMeta = async () => {
      setLoading(true);
      try {
        const r = await ApiCall.emailRegistry.getPasswordMeta(emailId);
        const meta = r?.data?.item;
        const hp = !!meta?.has_password;
        setHasPwd(hp);
        setUpdatedAt(meta?.updated_at ?? null);
        if (!hp) setValue("");
      } catch (e: any) {
        toast.error(e?.response?.data?.error ?? "Jelszó betöltése sikertelen");
      } finally {
        setLoading(false);
      }
    };

    const reveal = async () => {
      try {
        const r = await ApiCall.emailRegistry.getPasswordReveal(emailId);
        const pwd = r?.data?.item?.password ?? "";
        setValue(pwd);
        setShow(true);
      } catch (e: any) {
        toast.error(e?.response?.data?.error ?? "Nem sikerült megjeleníteni");
      }
    };

    const save = async () => {
      try {
        await ApiCall.emailRegistry.setPassword(emailId, value ?? "");
        toast.success(value ? "Jelszó mentve" : "Jelszó törölve");
        await loadMeta();
      } catch (e: any) {
        toast.error(e?.response?.data?.error ?? "Mentés sikertelen");
      }
    };

    useEffect(() => {
      setValue("");
      setShow(false);
      loadMeta();
    }, [emailId]);

    return (
      <form
        autoComplete="off"
        onSubmit={(e) => e.preventDefault()}
        className="mt-4 rounded border p-3 bg-white/50 space-y-2">
        <div className="flex items-center justify-between">
          <div className="font-medium">Postafiók jelszó (nyilvántartás)</div>
          {loading ? (
            <span className="text-xs text-gray-500">Betöltés…</span>
          ) : hasPwd ? (
            <span className="text-xs text-emerald-700">Beállítva</span>
          ) : (
            <span className="text-xs text-gray-500">Nincs beállítva</span>
          )}
        </div>

        {updatedAt && (
          <div className="text-xs text-gray-500">
            Utoljára módosítva: {updatedAt}
          </div>
        )}

        <div className="flex gap-2 items-center">
          {/* Honeypots to absorb autofill/password managers */}
          <input
            type="text"
            name="username"
            autoComplete="username"
            tabIndex={-1}
            aria-hidden="true"
            style={{
              position: "absolute",
              opacity: 0,
              height: 0,
              width: 0,
              pointerEvents: "none",
            }}
          />
          <input
            type="password"
            name="password"
            autoComplete="current-password"
            tabIndex={-1}
            aria-hidden="true"
            style={{
              position: "absolute",
              opacity: 0,
              height: 0,
              width: 0,
              pointerEvents: "none",
            }}
          />

          <input
            type={show ? "text" : "password"}
            className="border rounded px-3 py-2 w-full md:w-96"
            placeholder="Új jelszó…"
            value={value}
            onChange={(e) => setValue(e.target.value)}
            name={`emailpwd-${emailId}`} // unique name helps avoid autofill
            autoComplete="new-password"
            autoCorrect="off"
            autoCapitalize="off"
            spellCheck={false}
            inputMode="text"
            data-lpignore="true"
            data-1p-ignore="true"
            data-bwignore="true"
          />

          <button
            type="button"
            onClick={() => setShow((s) => !s)}
            className="px-3 py-2 rounded border">
            {show ? "Elrejt" : "Mutat"}
          </button>

          <button
            type="button"
            onClick={save}
            className="px-3 py-2 bg-emerald-600/80 hover:bg-emerald-600 text-white rounded">
            Mentés
          </button>

          <button
            type="button"
            onClick={async () => {
              if (!confirm("Biztosan törli a jelszót?")) return;
              setValue("");
              await save();
            }}
            className="px-3 py-2 bg-red-600/80 hover:bg-red-600 text-white rounded">
            Jelszó törlése
          </button>

          <button
            type="button"
            onClick={async () => {
              if (!hasPwd) {
                toast.error("Ehhez a postafiókhoz nincs jelszó elmentve.");
                return;
              }
              await reveal();
            }}
            disabled={!hasPwd}
            className="px-3 py-2 bg-gray-700/80 hover:bg-gray-700 text-white rounded disabled:opacity-50 disabled:cursor-not-allowed">
            Megjelenítés
          </button>
        </div>

        <p className="text-xs text-gray-500">
          A jelszó titkosítva kerül tárolásra. A “Megjelenítés” gomb csak
          megfelelő jogosultság mellett érhető el.
        </p>
      </form>
    );
  }

  function CpanelPasswordEditor({ identity }: { identity: CpanelIdentity }) {
    const [value, setValue] = useState("");
    const [show, setShow] = useState(false);
    const [saving, setSaving] = useState(false);

    const save = async () => {
      setSaving(true);
      try {
        const payload: { newPassword?: string } = {};
        if (value.trim().length > 0) {
          payload.newPassword = value;
        }

        if (!payload.newPassword) {
          toast.error("Adj meg egy új jelszót.");
          setSaving(false);
          return;
        }

        await ApiCall.emailRegistry.updateAccount(
          identity.domain,
          identity.localPart,
          payload
        );

        toast.success("cPanel jelszó beállítva");
      } catch (e: any) {
        console.error(e);
        toast.error(
          e?.response?.data?.error ?? "cPanel jelszó mentése sikertelen"
        );
      } finally {
        setSaving(false);
      }
    };

    const clear = async () => {
      if (
        !confirm(
          "Törlöd a mező tartalmát? A cPanel jelszó ettől még nem változik."
        )
      )
        return;
      setValue("");
    };

    return (
      <form
        autoComplete="off"
        onSubmit={(e) => e.preventDefault()}
        className="mt-4 rounded border p-3 bg-white/50 space-y-2">
        <div className="flex items-center justify-between">
          <div className="font-medium">cPanel postafiók jelszó (szerveren)</div>
          <span className="text-xs text-gray-500">{identity.email}</span>
        </div>

        <div className="flex gap-2 items-center">
          {/* Honeypots to absorb autofill/password managers */}
          <input
            type="text"
            name="cpanel-username"
            autoComplete="username"
            tabIndex={-1}
            aria-hidden="true"
            style={{
              position: "absolute",
              opacity: 0,
              height: 0,
              width: 0,
              pointerEvents: "none",
            }}
          />
          <input
            type="password"
            name="cpanel-password"
            autoComplete="current-password"
            tabIndex={-1}
            aria-hidden="true"
            style={{
              position: "absolute",
              opacity: 0,
              height: 0,
              width: 0,
              pointerEvents: "none",
            }}
          />

          <input
            type={show ? "text" : "password"}
            className="border rounded px-3 py-2 w-full md:w-96"
            placeholder="Új cPanel jelszó…"
            value={value}
            onChange={(e) => setValue(e.target.value)}
            name={`cpanelpwd-${identity.email}`}
            autoComplete="new-password"
            autoCorrect="off"
            autoCapitalize="off"
            spellCheck={false}
            inputMode="text"
            data-lpignore="true"
            data-1p-ignore="true"
            data-bwignore="true"
          />

          <button
            type="button"
            onClick={() => setShow((s) => !s)}
            className="px-3 py-2 rounded border">
            {show ? "Elrejt" : "Mutat"}
          </button>

          <button
            type="button"
            onClick={save}
            disabled={saving}
            className="px-3 py-2 bg-emerald-600/80 hover:bg-emerald-600 text-white rounded disabled:opacity-50">
            {saving ? "Mentés…" : "Mentés"}
          </button>

          <button
            type="button"
            onClick={clear}
            disabled={saving}
            className="px-3 py-2 bg-red-600/80 hover:bg-red-600 text-white rounded disabled:opacity-50">
            Mező ürítése
          </button>
        </div>

        <p className="text-xs text-gray-500">
          A cPanel API nem teszi lehetővé a jelenlegi jelszó megjelenítését vagy
          törlését, csak új jelszó beállítását.
        </p>
      </form>
    );
  }

  const identity = getCpanelIdentity(row);

  if (loading)
    return <div className="p-3 text-sm text-gray-500">Betöltés…</div>;
  if (!item)
    return <div className="p-3 text-sm text-red-600">Nem található.</div>;

  return (
    <div className="rounded border p-3 bg-white/50 space-y-3 max-w-[80vw]">
      <EmailRegistryForm
        initial={{
          id: item.id, // internal only

          email: item.email,
          type: item.type ?? null,

          handler_people_ids:
            item.handler_people_ids ??
            (item.handler_person_id ? [item.handler_person_id] : []),
          handler_people:
            item.handler_people ??
            (item.handler_person ? [item.handler_person] : []),
          handler_primary_person_id:
            item.handler_primary_person_id ?? item.handler_person_id ?? null,
          handler_segment: item.handler_segment_id ?? null,
          handler_segment_id: item.handler_segment_id ?? null,

          user_status: item.user_status ?? null,
          user_status_id: item.user_status_id ?? null,

          groups: item.groups ?? [],
          is_forwarded: !!item.is_forwarded,

          forward_to: Array.isArray(item.forward_to)
            ? item.forward_to
            : item.forward_to
            ? String(item.forward_to)
                .split(",")
                .map((s: string) => s.trim())
                .filter(Boolean)
            : [],

          should_archive: !!item.should_archive,
          notes: item.notes ?? "",

          receive_status: item.receive_status ?? "allow",
          send_status: item.send_status ?? "allow",
          login_status: item.login_status ?? "allow",
        }}
        onSubmit={handleSave}
      />
      <PasswordEditor emailId={id} />

      {/* cPanel info */}
      <div className="rounded border p-3 bg-slate-50/80 space-y-1 text-sm">
        <div className="font-medium text-gray-800">cPanel postafiók</div>

        {!row.cpanel_exists && (
          <div className="text-xs text-red-700">
            Ehhez az e-mail címhez nem tartozik cPanel postafiók.
          </div>
        )}

        {row.cpanel_exists && (
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-x-6 gap-y-1 text-xs text-gray-700">
            <div>
              <span className="font-semibold">Fiók: </span>
              {row.cpanel_email || row.email}
            </div>
            <div>
              <span className="font-semibold">Domain: </span>
              {row.cpanel_domain || "-"}
            </div>
            <div>
              <span className="font-semibold">Felhasználó: </span>
              {row.cpanel_user || "-"}
            </div>
            <div>
              <span className="font-semibold">Állapot: </span>
              {row.cpanel_status === "ok" && (
                <span className="text-emerald-700 font-semibold">OK</span>
              )}
              {row.cpanel_status === "suspended" && (
                <span className="text-red-700 font-semibold">
                  Felfüggesztve
                </span>
              )}
              {row.cpanel_status === "missing" && (
                <span className="text-gray-700">Nincs cPanel fiók</span>
              )}
            </div>

            {/* cPanel quota (display + editor like in cPanel) */}
            {row.cpanel_exists && identity && (
              <div className="sm:col-span-2 mt-2 space-y-2 text-xs">
                {/* current usage */}
                <div>
                  <span className="font-semibold">Jelenlegi kvóta: </span>
                  {(() => {
                    const used = Number(row.cpanel_diskused ?? 0);
                    const quota =
                      row.cpanel_diskquota != null
                        ? Number(row.cpanel_diskquota)
                        : null;

                    if (quota === null || quota <= 0) {
                      return (
                        <>
                          {Math.round(used)} MB /{" "}
                          <span className="font-semibold text-emerald-700">
                            Korlátlan
                          </span>
                        </>
                      );
                    }

                    return (
                      <>
                        {Math.round(used)} / {Math.round(quota)} MB
                      </>
                    );
                  })()}
                </div>

                {/* editor, like cPanel: limited vs unlimited */}
                <div className="border rounded p-2 bg-white/60">
                  <div className="flex items-center justify-between mb-1">
                    <div className="font-semibold text-gray-700">
                      Postafiók tárhely
                    </div>
                    <span
                      className="text-[10px] text-gray-500 cursor-default"
                      title="The amount of space that your email account can use to store emails.">
                      i
                    </span>
                  </div>

                  <div className="space-y-2">
                    {/* Limited radio + number + unit */}
                    <label className="flex items-center gap-2">
                      <input
                        type="radio"
                        name={`quotaType-${identity.email}`}
                        className="h-3 w-3"
                        checked={quotaType === "limited"}
                        onChange={() => setQuotaType("limited")}
                      />
                      <span>Kvóta</span>
                      <div className="flex items-center gap-1 ml-3">
                        <input
                          type="number"
                          min={1}
                          className="border rounded px-2 py-1 w-24"
                          disabled={quotaType !== "limited"}
                          value={quotaValue}
                          onChange={(e) =>
                            setQuotaValue(
                              e.target.value === "" ? 0 : Number(e.target.value)
                            )
                          }
                        />

                        <select
                          className="border rounded px-1 py-1 text-xs"
                          disabled={quotaType !== "limited"}
                          value={quotaUnit}
                          onChange={(e) =>
                            setQuotaUnit(e.target.value as QuotaUnit)
                          }>
                          <option value="MB">MB</option>
                          <option value="GB">GB</option>
                          <option value="TB">TB</option>
                          <option value="PB">PB</option>
                        </select>
                      </div>
                    </label>

                    {/* Unlimited radio */}
                    <label className="flex items-center gap-2">
                      <input
                        type="radio"
                        name={`quotaType-${identity.email}`}
                        className="h-3 w-3"
                        checked={quotaType === "unlimited"}
                        onChange={() => setQuotaType("unlimited")}
                      />
                      <span>Korlátlan</span>
                    </label>

                    {/* Save button + small validation */}
                    <div className="mt-2 flex items-center gap-2">
                      <button
                        type="button"
                        onClick={async () => {
                          if (quotaType === "limited") {
                            if (!quotaValue || quotaValue <= 0) {
                              toast.error(
                                "Adj meg egy nullánál nagyobb kvótaértéket."
                              );
                              return;
                            }
                          }

                          try {
                            setSavingQuota(true);

                            const newQuotaMb =
                              quotaType === "unlimited"
                                ? 0 // cPanel: 0 = unlimited
                                : convertToMb(quotaValue, quotaUnit);

                            await ApiCall.emailRegistry.updateAccount(
                              identity.domain,
                              identity.localPart,
                              { newQuotaMb }
                            );

                            toast.success("cPanel kvóta frissítve");
                            await load(); // refresh data from backend
                          } catch (e: any) {
                            console.error(e);
                            toast.error(
                              e?.response?.data?.error ??
                                "Kvóta frissítése sikertelen"
                            );
                          } finally {
                            setSavingQuota(false);
                          }
                        }}
                        disabled={
                          savingQuota ||
                          (quotaType === "limited" &&
                            (!quotaValue || quotaValue <= 0))
                        }
                        className="px-2 py-1 bg-gray-700/80 hover:bg-gray-700 text-white rounded disabled:opacity-50 text-xs">
                        {savingQuota ? "Mentés…" : "Kvóta mentése"}
                      </button>

                      {quotaType === "limited" && quotaValue > 0 && (
                        <span className="text-[11px] text-gray-500">
                          Mentve: {quotaValue} {quotaUnit} (cPanel-ben MB-ban
                          tárolva)
                        </span>
                      )}
                    </div>
                  </div>
                </div>
              </div>
            )}

            {/* <div>
              <span className="font-semibold">Bejelentkezés: </span>
              {cpanelFlags.suspendedLogin ? "Felfüggesztve" : "Engedélyezve"}
            </div>
            <div>
              <span className="font-semibold">Bejövő levelek: </span>
              {cpanelFlags.suspendedIncoming ? "Felfüggesztve" : "Engedélyezve"}
            </div>
            <div>
              <span className="font-semibold">Kimenő levelek: </span>
              {cpanelFlags.suspendedOutgoing ? "Felfüggesztve" : "Engedélyezve"}
            </div> */}
          </div>
        )}
      </div>

      {row.cpanel_exists && identity && (
        <div className="sm:col-span-2 mt-2 flex items-center gap-2 text-xs">
          <span className="font-semibold">Kvóta módosítása: </span>
          <input
            type="number"
            min={0}
            defaultValue={
              row.cpanel_diskquota != null ? Number(row.cpanel_diskquota) : 0
            }
            className="border rounded px-2 py-1 w-24"
            onChange={(e) => {
              // you can manage this with useState if you want; here only example
            }}
          />
          <button
            type="button"
            className="px-2 py-1 bg-gray-700/80 hover:bg-gray-700 text-white rounded"
            onClick={async () => {
              const val = prompt(
                "Új kvóta MB-ban (0 = korlátlan)",
                row.cpanel_diskquota != null
                  ? String(row.cpanel_diskquota)
                  : "0"
              );
              if (!val) return;
              const quotaMb = Number(val);
              if (Number.isNaN(quotaMb) || quotaMb < 0) {
                toast.error("Érvénytelen kvóta érték.");
                return;
              }
              try {
                await ApiCall.emailRegistry.updateAccount(
                  identity.domain,
                  identity.localPart,
                  { newQuotaMb: quotaMb }
                );
                toast.success("cPanel kvóta frissítve");
                await load(); // refresh used/quota display
              } catch (e: any) {
                console.error(e);
                toast.error(
                  e?.response?.data?.error ?? "Kvóta frissítése sikertelen"
                );
              }
            }}>
            Kvóta mentése
          </button>
        </div>
      )}

      {row.cpanel_exists && identity && (
        <>
          <CpanelStatusEditor
            identity={identity}
            emailId={id}
            initial={cpanelFlags}
            onUpdated={(next) => {
              setCpanelFlags(next);
              // optional: refresh item from backend as well
              load();
            }}
          />
          <CpanelAutoresponderEditor identity={identity} />
          <CpanelPasswordEditor identity={identity} />
        </>
      )}

      <div className="flex items-center justify-between">
        <span className="text-xs text-gray-500">
          Utoljára módosítva: {item.updated_at ?? "-"}
        </span>
        {/* <button
          onClick={handleDelete}
          className="px-3 py-2 bg-red-600/80 hover:bg-red-600 text-white rounded">
          Törlés
        </button> */}
      </div>
    </div>
  );
}
