"use client";
import { useEffect, useState, useRef, useCallback } from "react";

const InfiniteScrollList = ({
  fetchItems, // (page, limit) => Promise<{ items: [], total: number }>
  renderItem, // (item) => JSX.Element
  limit = 20,
  initialPage = 1,
  onSessionExpired = null,
  onError = null,
}) => {
  const [items, setItems] = useState([]);
  const [page, setPage] = useState(initialPage);
  const [loading, setLoading] = useState(false);
  const [hasMore, setHasMore] = useState(true);
  const loaderRef = useRef();

  const loadItems = useCallback(async () => {
    if (loading || !hasMore) return;

    setLoading(true);
    try {
      const { items: newItems, total } = await fetchItems(page, limit);

      setItems((prev) => [...prev, ...newItems]);
      setPage((prev) => prev + 1);

      if (items.length + newItems.length >= total) {
        setHasMore(false);
      }
    } catch (err) {
      console.error("❌ InfiniteScrollList Error:", err);

      const isAuthError =
        err?.message?.toLowerCase().includes("session expired") ||
        err?.type === "auth";

      if (isAuthError && typeof onSessionExpired === "function") {
        onSessionExpired();
        return;
      }

      if (typeof onError === "function") {
        onError(err);
      }
    } finally {
      setLoading(false);
    }
  }, [page, loading, hasMore]);

  // Trigger initial load
  useEffect(() => {
    loadItems();
  }, []);

  // Observe bottom loader to auto-fetch next page
  useEffect(() => {
    if (!loaderRef.current || !hasMore) return;

    const observer = new IntersectionObserver(
      (entries) => {
        if (entries[0].isIntersecting && !loading) {
          loadItems();
        }
      },
      { threshold: 1.0 }
    );

    observer.observe(loaderRef.current);
    return () => observer.disconnect();
  }, [loaderRef.current, hasMore, loading]);

  return (
    <>
      <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
        {items.map((item, idx) => (
          <div key={idx}>{renderItem(item)}</div>
        ))}
      </div>

      {loading && (
        <div className="text-center py-4 text-gray-500">Loading more...</div>
      )}

      {hasMore && !loading && (
        <div ref={loaderRef} className="py-4 text-center text-sm text-gray-400">
          Scroll to load more
        </div>
      )}
    </>
  );
};

export default InfiniteScrollList;
